package com.instabug.apm.screenloading.handler

import android.app.Activity
import android.os.Build
import android.os.Bundle
import android.os.Handler
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.constants.ErrorMessages
import com.instabug.apm.constants.UiLoadingMetric
import com.instabug.apm.di.Provider
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.screenloading.repo.NativeScreenLoadingRepo
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks
import com.instabug.apm.uitrace.util.screenName
import com.instabug.apm.util.runOrReportAPMError
import com.instabug.apm.util.view.InstabugViews
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.factory.ParameterizedFactory
import java.util.concurrent.Executor

interface NativeEndScreenLoadingHandler {
    fun <ActivityType : Activity> endScreenLoading(
        activityClass: Class<ActivityType>,
        timeMetric: EventTimeMetricCapture,
    )
}

class NativeScreenLoadingHandlerImpl(
    private val executor: Executor,
    private val mainThreadHandlerProvider: Provider<Handler>,
    private val configurationProvider: APMConfigurationProvider,
    private val nativeScreenLoadingRepo: NativeScreenLoadingRepo,
    private val logger: Logger
) : NativeEndScreenLoadingHandler, APMUiTraceActivityCallbacks {

    private val Activity.isValidToHandleScreenLoading: Boolean
        get() = !InstabugViews.isInstabugActivity(this) && configurationProvider.isAutoUiLoadingMetricsFullyEnabled
    private val isNeededToHandleActivityResumedPostRun
        get() = BuildFieldsProvider.provideBuildVersion() < Build.VERSION_CODES.Q

    override fun onActivityPreCreated(
        activity: Activity,
        savedInstanceState: Bundle?,
        timeMetric: EventTimeMetricCapture,
        uitraceId: Long
    ) = runIfCanHandleScreenLoading(activity, "onActivityPreCreated") {
        nativeScreenLoadingRepo.apply {
            start(activity.screenName, uitraceId)
            addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_PRE_CREATED)
        }
    }

    override fun onActivityCreated(
        activity: Activity,
        savedInstanceState: Bundle?,
        timeMetric: EventTimeMetricCapture,
        uiTraceId: Long
    ) = runIfCanHandleScreenLoading(activity, "onActivityCreated") {
        nativeScreenLoadingRepo.apply {
            startIfNotStarted(activity.screenName, uiTraceId)
            addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_CREATED)
        }
    }

    override fun onActivityPostCreated(
        activity: Activity,
        savedInstanceState: Bundle?,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityPostCreated") {
        nativeScreenLoadingRepo
            .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_POST_CREATED)
    }

    override fun onActivityPreStarted(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityPreStarted") {
        nativeScreenLoadingRepo
            .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_PRE_STARTED)
    }

    override fun onActivityStarted(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityStarted") {
        nativeScreenLoadingRepo
            .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_STARTED)
    }

    override fun onActivityPostStarted(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityPostStarted") {
        nativeScreenLoadingRepo
            .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_POST_STARTED)
    }

    override fun onActivityPreResumed(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityPreResumed") {
        nativeScreenLoadingRepo
            .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_PRE_RESUMED)
    }

    override fun onActivityResumed(
        activity: Activity,
        timeMetric: EventTimeMetricCapture,
        uiTraceId: Long
    ) {
        sendActivityResumedPostRunEvent(activity)
        runIfCanHandleScreenLoading(activity, "onActivityResumed") {
            nativeScreenLoadingRepo
                .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_RESUMED)
        }
    }

    private fun sendActivityResumedPostRunEvent(activity: Activity) {
        if (!isNeededToHandleActivityResumedPostRun) return
        mainThreadHandlerProvider().postAtFrontOfQueue {
            handleActivityResumedPostRun(activity, EventTimeMetricCapture())
        }
    }

    private fun handleActivityResumedPostRun(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityResumedPostRun") {
        nativeScreenLoadingRepo.addResumedPostRunStage(activity.screenName, timeMetric)
    }

    override fun onActivityPostResumed(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityPostResumed") {
        nativeScreenLoadingRepo
            .addStage(activity.screenName, timeMetric, UiLoadingMetric.ON_ACTIVITY_POST_RESUMED)
    }

    override fun onActivityPaused(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = runIfCanHandleScreenLoading(activity, "onActivityPaused") {
        nativeScreenLoadingRepo.end(activity.screenName)
    }

    override fun <ActivityType : Activity> endScreenLoading(
        activityClass: Class<ActivityType>,
        timeMetric: EventTimeMetricCapture
    ) = runOrReportAPMError("Error while reporting native endScreenLoading", logger) {
        nativeScreenLoadingRepo.setEndScreenLoadingStage(activityClass.screenName, timeMetric)
            .takeUnless { it }
            ?.let { logger.logSDKWarning(ErrorMessages.END_SCREEN_NOT_DISPATCHED_INVALID_SCREEN) }
    }

    private inline fun runIfCanHandleScreenLoading(
        activity: Activity,
        callback: String,
        crossinline operation: () -> Unit
    ) = executor.execute {
        runOrReportAPMError("error while handling native screen loading callback: $callback", logger) {
            if (activity.isValidToHandleScreenLoading) operation()
        }
    }
}

class NativeScreenLoadingHandlerFactory(
    private val executor: Executor,
    private val mainThreadHandlerProvider: Provider<Handler>,
    private val configurationProvider: APMConfigurationProvider,
    private val logger: Logger
) : ParameterizedFactory<NativeScreenLoadingHandlerImpl, NativeScreenLoadingRepo> {
    override fun create(type: NativeScreenLoadingRepo): NativeScreenLoadingHandlerImpl =
        NativeScreenLoadingHandlerImpl(
            executor,
            mainThreadHandlerProvider,
            configurationProvider,
            type,
            logger
        )
}