package com.instabug.apm.networking.mapping.uitrace;

import static com.instabug.apm.webview.webview_trace.ConstantsKt.WEB_VIEWS_SYNC_ROOT_KEY;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.UiLoadingModel;
import com.instabug.apm.cache.model.UiTraceCacheModel;
import com.instabug.apm.networking.mapping.uiloading.UiLoadingMapper;
import com.instabug.apm.webview.webview_trace.WebViewCacheModelToJsonMapper;
import com.instabug.apm.webview.webview_trace.model.WebViewCacheModel;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Implementation of {@link UiTraceCacheModel} serialization
 */
@SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
public class UiTraceMapperImpl implements UiTraceMapper {

    @NonNull
    private final UiLoadingMapper uiLoadingMapper;
    @NonNull
    private final WebViewCacheModelToJsonMapper webViewMapper;

    public UiTraceMapperImpl(
            @NonNull UiLoadingMapper uiLoadingMapper,
            @NonNull WebViewCacheModelToJsonMapper webViewMapper
    ) {
        this.uiLoadingMapper = uiLoadingMapper;
        this.webViewMapper = webViewMapper;
    }

    @Override
    public JSONArray toJsonArray(List<UiTraceCacheModel> uiTraces) throws JSONException {
        JSONArray jsonArray = new JSONArray();
        for (UiTraceCacheModel uiTrace : uiTraces) {
            jsonArray.put(toJson(uiTrace));
        }
        return jsonArray;
    }

    @Override
    public JSONObject toJson(UiTraceCacheModel uiTrace) throws JSONException {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put(UiTraceMappingKeys.NAME, uiTrace.getName());
        jsonObject.put(UiTraceMappingKeys.DURATION, uiTrace.getDuration());
        long largeDropsDuration = uiTrace.getLargeDropsDuration();
        long smallDropsDuration = uiTrace.getSmallDropsDuration();
        if (!(largeDropsDuration == -1 || smallDropsDuration == -1)) {
            jsonObject.put(UiTraceMappingKeys.LARGE_DROP_DURATION, largeDropsDuration);
            jsonObject.put(UiTraceMappingKeys.SMALL_DROP_DURATION, smallDropsDuration);
        }
        jsonObject.put(UiTraceMappingKeys.USER_DEFINED_TRACE, uiTrace.isUserDefined());
        jsonObject.put(UiTraceMappingKeys.START_TIME, uiTrace.getStartTimestamp());
        jsonObject.put(UiTraceMappingKeys.REFRESH_RATE, uiTrace.getRefreshRate());
        if (uiTrace.getBatteryLevel() != -1) {
            jsonObject.put(UiTraceMappingKeys.BATTERY_LEVEL, uiTrace.getBatteryLevel());
        }
        if (uiTrace.getContainerName() != null) {
            jsonObject.put(UiTraceMappingKeys.CONTAINER_NAME, uiTrace.getContainerName());
        }
        if (uiTrace.getScreenTitle() != null) {
            jsonObject.put(UiTraceMappingKeys.SCREEN_TITLE, uiTrace.getScreenTitle());
        }
        if (uiTrace.getOrientation() != null) {
            jsonObject.put(UiTraceMappingKeys.ORIENTATION, uiTrace.getOrientation());
        }
        if (uiTrace.getModuleName() != null) {
            jsonObject.put(UiTraceMappingKeys.MODULE_NAME, uiTrace.getModuleName());
        }
        if (uiTrace.getPowerSaveMode() != null) {
            jsonObject.put(UiTraceMappingKeys.POWER_SAVE_MODE_ENABLED, uiTrace.getPowerSaveMode());
        }

        UiLoadingModel uiLoadingModel = uiTrace.getUiLoadingModel();
        if (uiLoadingModel != null && uiLoadingMapper != null) {
            JSONObject uiLoadingJsonObject = uiLoadingMapper.toJson(uiLoadingModel);
            if (uiLoadingJsonObject != null) {
                jsonObject.put(UiTraceMappingKeys.UI_LOADING_MODEL, uiLoadingJsonObject);
            }
        }
        putWebViewTraces(jsonObject, uiTrace.getWebViewTraces());
        return jsonObject;
    }

    private void putWebViewTraces(@NonNull JSONObject object, @Nullable List<WebViewCacheModel> models) throws JSONException {
        JSONArray webViewsJson = webViewMapper.map(models);
        if (webViewsJson != null) object.put(WEB_VIEWS_SYNC_ROOT_KEY, webViewsJson);
    }
}
