package com.instabug.apm.handler.uitrace.uiloading;

import static com.instabug.apm.constants.UiLoadingMetric.ACTIVITY_EVENTS_COUNT;
import static com.instabug.apm.constants.UiLoadingMetric.ON_ACTIVITY_POST_RESUMED;
import static com.instabug.apm.constants.UiLoadingMetric.ON_ACTIVITY_RESUMED_POST_RUN;
import static com.instabug.apm.constants.UiLoadingMetric.ON_CUSTOM_SCREEN_LOADING_END;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.UiLoadingModel;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.constants.UiLoadingMetric;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.EmptyTimeMetric;
import com.instabug.apm.model.EventTimeMetricCapture;

import java.util.Arrays;

public class UiLoadingMetricHandlerImpl implements UiLoadingMetricHandler {

    private final EventTimeMetricCapture[] activityEventsTimeMetric = new EventTimeMetricCapture[ACTIVITY_EVENTS_COUNT];
    private final ActivityEventToUiLoadingModelMapper mapper;
    private final Logger apmLogger;

    public UiLoadingMetricHandlerImpl(
            @NonNull ActivityEventToUiLoadingModelMapper mapper,
            @NonNull Logger apmLogger
    ) {
        this.mapper = mapper;
        this.apmLogger = apmLogger;
        Arrays.fill(activityEventsTimeMetric, new EmptyTimeMetric());
    }

    @Override
    public void addActivityEvent(
            @UiLoadingMetric.ActivityEvents int activityEvent,
            @NonNull EventTimeMetricCapture timeMetric
    ) {
        if (isValidEventIndex(activityEvent)) {
            activityEventsTimeMetric[activityEvent] = timeMetric;
        }
    }

    @Override
    public void onScreenLoadingEnded(EventTimeMetricCapture timeMetric) {
        int activityEvent = ON_CUSTOM_SCREEN_LOADING_END;
        if (!hasEventNeverBeenReportedBefore(activityEvent)) {
            if (apmLogger != null) {
                apmLogger.logSDKWarning(ErrorMessages.END_SCREEN_NOT_DISPATCHED_MULTIPLE_CALLS);
            }
        } else {
            if (hasLastAutomaticUiLoadingEventNotBeenCollectedBeforeTimeNano(timeMetric.getNanoTime())) {
                if (apmLogger != null) {
                    apmLogger.logSDKError(ErrorMessages.END_SCREEN_NOT_DISPATCHED_CALLED_TOO_EARLY);
                }
            }
            activityEventsTimeMetric[activityEvent] = timeMetric;
        }
    }

    @Nullable
    @Override
    public UiLoadingModel getUiLoadingModel() {
        if (mapper == null) {
            return null;
        }
        UiLoadingModel uiLoadingModel = mapper.getUiLoadingModel(activityEventsTimeMetric);
        if (uiLoadingModel != null && uiLoadingModel.isValid()) {
            return uiLoadingModel;
        }
        return null;
    }

    private boolean isValidEventIndex(@UiLoadingMetric.ActivityEvents int eventIndex) {
        if (eventIndex >= 0 && eventIndex < activityEventsTimeMetric.length) {
            return hasEventNeverBeenReportedBefore(eventIndex);
        }
        return false;
    }

    private boolean hasEventBeenReportedAfterTimeNano(
            @UiLoadingMetric.ActivityEvents int eventIndex,
            long timeNano
    ) {
        return activityEventsTimeMetric[eventIndex].getNanoTime() > timeNano;
    }

    @Override
    public boolean hasEventNeverBeenReportedBefore(@UiLoadingMetric.ActivityEvents int eventIndex) {
        return activityEventsTimeMetric[eventIndex] instanceof EmptyTimeMetric;
    }

    private boolean hasEventNotBeenReportedOrBeenReportedAfterTimeNano(
            @UiLoadingMetric.ActivityEvents int eventIndex,
            long timeNano
    ) {
        return hasEventNeverBeenReportedBefore(eventIndex) ||
                hasEventBeenReportedAfterTimeNano(eventIndex, timeNano);
    }

    private boolean hasLastAutomaticUiLoadingEventNotBeenCollectedBeforeTimeNano(long timeNano) {
        return hasEventNotBeenReportedOrBeenReportedAfterTimeNano(ON_ACTIVITY_POST_RESUMED, timeNano)
                && hasEventNotBeenReportedOrBeenReportedAfterTimeNano(ON_ACTIVITY_RESUMED_POST_RUN, timeNano);
    }
}
