package com.instabug.apm.compose.compose_spans

import com.instabug.apm.compose.compose_spans.handler.ComposeSpansHandler
import com.instabug.apm.compose.compose_spans.model.ComposeSpanEvent
import com.instabug.apm.compose.compose_spans.model.ComposeSpanEventId
import com.instabug.apm.compose.compose_spans.model.ComposeSpansCacheModel
import com.instabug.apm.compose.compose_spans.model.ComposeSpansModel
import com.instabug.apm.handler.session.SessionHandler
import com.instabug.library.map.Mapper
import com.instabug.library.util.collections.LimitedLinkedHashmap

class ComposeSpansEventsRepositoryImpl(
    private val handler: ComposeSpansHandler,
    private val sessionHandler: SessionHandler,
    private val mapper: Mapper<ComposeSpansModel, ComposeSpansCacheModel?>,
    private val nameSanitizer: Mapper<String, String?>
): ComposeSpansEventsRepository {

    private val eventsMap: MutableMap<String, ComposeSpansModel> = LimitedLinkedHashmap(50)

    override fun addEvent(event: ComposeSpanEvent) {
        val key = event.getKey()
        when (event.eventId) {
            ComposeSpanEventId.CompositionStarted -> {
                saveModelToDiskAndDeleteItFromMemory(key)
                createNewModel(key, event)
                addEventToMemory(key, event)
            }
            ComposeSpanEventId.RenderingEnded -> {
                addEventToMemory(key, event)
                saveModelToDiskAndDeleteItFromMemory(key)
            }
            ComposeSpanEventId.Dispose -> {
                saveModelToDiskAndDeleteItFromMemory(key)
            }
            else -> {
                addEventToMemory(key, event)
            }
        }
    }

    private fun createNewModel(key: String, event: ComposeSpanEvent) {
        nameSanitizer.map(event.composableName)?.let { sanitizedName ->
            eventsMap[key] = ComposeSpansModel(
                sanitizedName
            )
        }
    }

    private fun removeModelFromMemory(key: String): ComposeSpansModel? = eventsMap.remove(key)

    private fun saveModelToDiskAndDeleteItFromMemory(key: String) {
        removeModelFromMemory(key)?.saveToDisk()
    }

    private fun addEventToMemory(key: String, event: ComposeSpanEvent) {
        eventsMap[key]?.set(event.eventId, event.timeCapture)
    }

    private fun ComposeSpansModel.saveToDisk() {
        mapper.map(this)?.let { cacheModel ->
            handler.save(cacheModel, sessionHandler.currentSession?.id)
        }
    }

    private fun ComposeSpanEvent.getKey(): String =
        "$id-$composableName"
}