package com.instabug.apm.compose

import com.instabug.apm.compose.compose_spans.ComposeSpansServiceLocator
import com.instabug.apm.model.EventTimeMetricCapture
import java.util.Collections
import java.util.concurrent.Executor

object ComposeEventDispatcher {

    private val listeners =
        Collections.synchronizedSet(HashSet<ComposeEventListener>())

    private val executor: Executor by lazy { ComposeSpansServiceLocator.eventsDispatcherExecutor }

    fun addListener(listener: ComposeEventListener) {
        listeners += listener
    }

    fun removeListener(listener: ComposeEventListener) {
        listeners -= listener
    }

    operator fun plusAssign(listener: ComposeEventListener) = addListener(listener)

    operator fun minusAssign(listener: ComposeEventListener) = removeListener(listener)


    fun onCompositionStarted(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onCompositionStarted)


    fun onCompositionEnded(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onCompositionEnded)


    fun onMeasuringAndLayoutStarted(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onMeasuringAndLayoutStarted)


    fun onMeasuringAndLayoutEnded(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onMeasuringAndLayoutEnded)


    fun onRenderingStarted(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onRenderingStarted)


    fun onRenderingEnded(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onRenderingEnded)


    fun onDispose(id: Int, screenName: String) =
        dispatchEvent(id, screenName, ComposeEventListener::onDispose)

    private inline fun dispatchEvent(
        id: Int,
        screenName: String,
        crossinline event: (ComposeEventListener, Int, String, EventTimeMetricCapture) -> Unit
    ) {
        val timeMetric = EventTimeMetricCapture()
        executor.execute {
            synchronized(listeners) {
                listeners.forEach { event.invoke(it, id, screenName, timeMetric) }
            }
        }
    }
}