package com.instabug.apm.cache.handler.networklog;

import androidx.annotation.Nullable;

import com.instabug.apm.model.APMNetworkLog;

import java.util.List;
import java.util.Map;

public interface DanglingNetworkLogCacheHandler {

    /**
     * Inserts network log in cache
     *
     * @param networkLog network log to be cached
     * @return id of cached network log
     */
    long insertNetworkLog(APMNetworkLog networkLog);

    /**
     * Update cached network log values
     *
     * @param networkLog updated network log object
     */
    void updateNetworkLog(APMNetworkLog networkLog);

    /**
     * Removes invalid network logs
     * Invalid network log doesn't have (status code, error code, error message)
     */
    void cleanUp();

    /**
     * Keeps only last x number of records in cache and deletes other records
     *
     * @param limit number of logs to keep
     */
    void trimToLimit(long limit);

    /**
     * Gets list of cached network logs by limit
     *
     * @param limit max count of network logs to be retrieved
     * @return list of cached network logs. Empty list in case of no cached logs
     */
    @Nullable
    List<APMNetworkLog> getNetworkLogsByLimit(long limit);

    /**
     * Removes cached network logs by limit
     *
     * @param limit max count of network logs to be removed
     */
    void deleteNetworkLogsByLimit(int limit);

    /**
     * Removes all cached network logs
     */
    void removeAll();

    /**
     * Removes all graphQl related data
     */
    void removeGraphQlData();

    /**
     * Removes all GRPC related records
     */
    void removeGrpcData();

    /**
     * Adds new attribute to specific trace
     *
     * @param traceId   id of the associated trace
     * @param traceName name of the associated trace used for logging
     * @param key       attribute key
     * @param value     attribute value
     */
    void addAttribute(long traceId, String traceName, String key, String value);


    /**
     * Adds new attribute to specific trace
     *
     * @param traceId   id of the associated trace
     * @return  Map of attributes' keys and value
     */
    @Nullable
    Map<String, String> getTraceAttributes(long traceId);

    /**
     * Removes all W3C external trace id related data including generated and captured.
     */
    void clearW3CExternalTraceIdCache();

    /**
     * Removes generated W3C trace id.
     */
    void clearGeneratedW3CExternalTraceIdCache();

    /**
     * Removes captured W3C trace id.
     */
    void clearCapturedW3CExternalTraceIdCache();
}
