package com.instabug.apm.appflow.configuration

import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.configuration.APMPreferencePropertyFactory
import com.instabug.library.util.LimitConstraintApplier

interface AppFlowConfigurationProvider {
    var featureEnabled: Boolean
    val enabled: Boolean
    var attributesEnabled: Boolean
    var requestLimit: Int
    var storeLimit: Int
    var attributesStoreLimit: Int
    var idlingTimeThresholdMs: Int
    fun reset()
}

class AppFlowConfigurationProviderImpl(
    private val apmConfigurationProvider: APMConfigurationProvider,
    private val limitApplier: LimitConstraintApplier,
    preferencePropertyFactory: APMPreferencePropertyFactory
) : AppFlowConfigurationProvider {

    private val featureEnabledPref = preferencePropertyFactory.create(
        APP_FLOW_FEATURE_ENABLED_PREF_KEY,
        APP_FLOW_FEATURE_ENABLED_DEFAULT
    )
    private val attributesEnabledPref = preferencePropertyFactory.create(
        APP_FLOW_ATTRIBUTES_ENABLED_PREF_KEY,
        APP_FLOW_ATTRIBUTES_ENABLED_DEFAULT
    )
    private val requestLimitPref = preferencePropertyFactory.create(
        APP_FLOW_REQUEST_LIMIT_PREF_KEY,
        APP_FLOW_REQUEST_LIMIT_DEFAULT
    )
    private val storeLimitPref = preferencePropertyFactory.create(
        APP_FLOW_STORE_LIMIT_PREF_KEY,
        APP_FLOW_STORE_LIMIT_DEFAULT
    )
    private val attributesStoreLimitPref = preferencePropertyFactory.create(
        APP_FLOW_ATTRIBUTES_STORE_LIMIT_PREF_KEY,
        APP_FLOW_ATTRIBUTES_STORE_LIMIT_DEFAULT
    )
    private val idlingTimeThresholdMsPref = preferencePropertyFactory.create(
        APP_FLOW_IDLING_TIME_THRESHOLD_MS_PREF_KEY,
        APP_FLOW_IDLING_TIME_THRESHOLD_MS_DEFAULT
    )

    private var _requestLimit: Int by requestLimitPref
    private var _storeLimit: Int by storeLimitPref

    override var featureEnabled: Boolean by featureEnabledPref
    override var attributesEnabled: Boolean by attributesEnabledPref
    override var requestLimit: Int
        set(value) {
            _requestLimit = value
        }
        get() = limitApplier.applyConstraints(_requestLimit)
    override var storeLimit: Int
        set(value) {
            _storeLimit = value
        }
        get() = limitApplier.applyConstraints(_storeLimit)
    override var attributesStoreLimit: Int by attributesStoreLimitPref
    override var idlingTimeThresholdMs: Int by idlingTimeThresholdMsPref
    override val enabled: Boolean
        get() = apmConfigurationProvider.isAPMEnabled && featureEnabled

    override fun reset() {
        featureEnabledPref.clear()
        attributesEnabledPref.clear()
        requestLimitPref.clear()
        storeLimitPref.clear()
        attributesStoreLimitPref.clear()
        idlingTimeThresholdMsPref.clear()
    }
}