package com.instabug.apm.model;

import static com.instabug.apm.constants.Constants.EXECUTION_TRACES_THREAD_EXECUTOR;
import static com.instabug.apm.constants.ErrorMessages.DEPRECATED_END_EXECUTION_TRACE;
import static com.instabug.apm.constants.ErrorMessages.DEPRECATED_SET_EXECUTION_TRACE_ATTRIBUTE;
import static com.instabug.apm.constants.ErrorMessages.TRACE_NOT_ENDED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.TRACE_NOT_ENDED_FEATURE_NOT_AVAILABLE;

import android.annotation.SuppressLint;
import android.os.Parcel;
import android.os.Parcelable;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.APM;
import com.instabug.apm.cache.model.ExecutionTraceCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.constants.DefaultValues;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.handler.executiontraces.ExecutionTracesHandler;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.apichecker.APIChecker;

import org.json.JSONObject;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Executor;

public class ExecutionTrace implements Parcelable {

    public static final Parcelable.Creator<ExecutionTrace> CREATOR = new Parcelable.Creator<ExecutionTrace>() {
        @Override
        public ExecutionTrace createFromParcel(Parcel source) {
            return new ExecutionTrace(source);
        }

        @Override
        public ExecutionTrace[] newArray(int size) {
            return new ExecutionTrace[size];
        }
    };

    private transient final Executor executor = ServiceLocator.getSingleThreadPoolExecutor(EXECUTION_TRACES_THREAD_EXECUTOR);
    private transient final ExecutionTracesHandler executionTracesHandler = ServiceLocator.getExecutionTracesHandler();
    private transient final Logger apmLogger = ServiceLocator.getApmLogger();

    @NonNull
    private final Map<String, String> attrs;
    private final String name;
    private final long id;
    private boolean startedInBackground;
    private boolean endedInBackground;

    private final long startTime;
    private final long startTimeMicro;
    private long endTimeMicro = -1;
    private boolean isEnded = false;

    public ExecutionTrace(final String name) {
        Random randomSeed = new Random();
        id = randomSeed.nextLong();
        attrs = new LinkedHashMap<>();
        this.name = name;
        startTimeMicro = System.nanoTime() / 1000;
        startTime = System.currentTimeMillis() * 1000;
        startedInBackground = ServiceLocator.getSessionHandler().getCurrentSession() == null;
        apmLogger.logSDKDebug("Execution trace " + name + " has started.");
    }

    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    protected ExecutionTrace(Parcel in) {
        id = in.readLong();
        int attrsSize = in.readInt();
        this.attrs = new LinkedHashMap<>(attrsSize);
        for (int i = 0; i < attrsSize; i++) {
            String key = in.readString();
            String value = in.readString();
            this.attrs.put(key, value);
        }
        this.name = in.readString();
        this.startTime = in.readLong();
        this.startTimeMicro = in.readLong();
        this.endTimeMicro = in.readLong();
    }

    /**
     * Ends this instance of Execution Trace.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-execution-traces">here</a>
     *
     * @deprecated see {@link APM#endFlow(String)}
     */
    @Deprecated
    public void end() {
        endTimeMicro = System.nanoTime() / 1000;
        apmLogger.w(DEPRECATED_END_EXECUTION_TRACE);
        endedInBackground = ServiceLocator.getSessionHandler().getCurrentSession() == null;
        APIChecker.checkAndRunInExecutor("ExecutionTrace.end", () -> executor.execute(() -> {
            APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
            if (!apmConfigurationProvider.isAPMEnabled()) {
                apmLogger.e(TRACE_NOT_ENDED_APM_NOT_ENABLED.replace("$s", name));
            } else if (!apmConfigurationProvider.isExecutionTraceFeatureEnabled()) {
                apmLogger.e(TRACE_NOT_ENDED_FEATURE_NOT_AVAILABLE.replace("$s", name));
            } else {
                long duration = endTimeMicro - startTimeMicro;
                ExecutionTraceCacheModel cacheModel = new ExecutionTraceCacheModel.Builder(name, id)
                        .setStartInBackground(startedInBackground)
                        .setAttrs(attrs)
                        .setDuration(duration)
                        .setEndedInBackground(endedInBackground)
                        .setStartTime(startTime)
                        .setAttrs(attrs)
                        .build();
                executionTracesHandler.insertTrace(cacheModel);
                apmLogger.logSDKDebug("Execution trace " + name + " has ended.\n"
                        + "Total duration: " + duration + " ms\n"
                        + "Attributes: " + new JSONObject(attrs));
                isEnded = true;
            }
        }));
    }

    /**
     * Sets custom attributes for this instance of ExecutionTrace.
     * <br/>
     * Setting an attribute value to null will remove its corresponding key if it already exists.
     * <br/>
     * Attribute key name cannot exceed 30 characters.
     * Leading and trailing whitespaces are also ignored.
     * Does not accept empty strings or null.
     * <br/>
     * Attribute value name cannot exceed 60 characters,
     * leading and trailing whitespaces are also ignored.
     * Does not accept empty strings.
     * <br/>
     * If a trace is ended, attributes will not be added and existing ones will not be updated.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-execution-traces">here</a>
     *
     * @param key   Execution Trace attribute value.
     * @param value Execution Trace attribute key. Null to remove attribute
     * @deprecated see {@link APM#setFlowAttribute(String, String, String)}
     */
    @MainThread
    @Deprecated
    public synchronized void setAttribute(@NonNull final String key, @Nullable final String value) {
        executor.execute(() -> {
            apmLogger.w(DEPRECATED_SET_EXECUTION_TRACE_ATTRIBUTE);
            if (isAtrributeKeyValid(key)) {
                String trimmedKey = key.trim();
                if (value != null) {
                    if (isAttributeValueValid(trimmedKey, value)) {
                        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
                        int maxAttributesCount = apmConfigurationProvider.getExecutionTraceStoreAttributesLimit();
                        if (attrs.size() == maxAttributesCount) {
                            apmLogger.e(ErrorMessages.MAX_ATTRIBUTES_COUNT_REACHED.replace("$s1", key)
                                    .replace("$s2", name)
                                    .replace("$s3", maxAttributesCount + ""));
                        } else {
                            attrs.put(trimmedKey, value.trim());
                        }
                    }
                } else {
                    attrs.remove(trimmedKey);
                }
            }
        });

    }

    private boolean isAttributeValueValid(String trimmedKey, String value) {
        String trimmedValue = value.trim();
        if (trimmedValue.length() == 0) {
            apmLogger.e(ErrorMessages.ATTRIBUTE_VALUE_EMPTY
                    .replace("$s1", trimmedKey)
                    .replace("$s2", name));
            return false;
        }
        if (trimmedValue.length() > DefaultValues.Traces.ATTRIBUTE_VALUE_LENGTH) {
            apmLogger.e(ErrorMessages.ATTRIBUTE_VALUE_INVALID_LENGTH
                    .replace("$s1", trimmedKey)
                    .replace("$s2", name));
            return false;
        }
        if (isEnded) {
            apmLogger.e(ErrorMessages.ADD_ATTRIBUTE_AFTER_END_EXECUTION_TRACE
                    .replace("$s1", trimmedKey)
                    .replace("$s2", name));
            return false;
        }
        return true;
    }

    private boolean isAtrributeKeyValid(String key) {
        if (key == null || key.trim().isEmpty()) {
            apmLogger.e(ErrorMessages.ATTRIBUTE_KEY_NULL_OR_EMPTY.replace("$s", name));
            return false;
        }
        final String trimmedKey = key.trim();
        if (trimmedKey.length() > DefaultValues.Traces.ATTRIBUTE_KEY_LENGTH) {
            apmLogger.e(ErrorMessages.ATTRIBUTE_KEY_INVALID_LENGTH
                    .replace("$s1", key)
                    .replace("$s2", name));
            return false;
        }
        return true;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeLong(id);
        dest.writeInt(this.attrs.size());
        for (Map.Entry<String, String> entry : this.attrs.entrySet()) {
            dest.writeString(entry.getKey());
            dest.writeString(entry.getValue());
        }
        dest.writeString(this.name);
        dest.writeLong(this.startTime);
        dest.writeLong(this.startTimeMicro);
        dest.writeLong(this.endTimeMicro);
    }
}
