package com.instabug.apm.cache.handler.session;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APM_APP_FLOW_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APM_APP_FLOW_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APP_LAUNCH_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APP_LAUNCH_TOTAL_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_COMPOSE_SPANS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_COMPOSE_SPANS_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXECUTION_TRACES_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXECUTION_TRACES_TOTAL_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXPERIMENTS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_FRAGMENT_SPANS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_FRAGMENT_SPANS_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_NETWORK_LOGS_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_NETWORK_LOGS_TOTAL_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_SESSION_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_UI_TRACES_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_UI_TRACES_TOTAL_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_WEB_VIEW_TRACES_DROPPED_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.COLUMN_WEB_VIEW_TRACES_TOTAL_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.apm.cache.model.SessionMetaData;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;


@WorkerThread
public class SessionMetaDataCacheHandlerImpl implements SessionMetaDataCacheHandler {

    private final Object lock = new Object();
    private final Logger logger = ServiceLocator.getApmLogger();

    @Override
    public void addToAppLaunchesTotalCount(@NonNull String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APP_LAUNCH_TOTAL_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void addToAppLaunchesDroppedCount(String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        COLUMN_APP_LAUNCH_DROPPED_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void addToTracesTotalCount(@NonNull String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXECUTION_TRACES_TOTAL_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void addToTracesDroppedCount(@NonNull String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        COLUMN_EXECUTION_TRACES_DROPPED_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void addToNetworkLogsTotalCount(@NonNull String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_NETWORK_LOGS_TOTAL_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void addToNetworkLogsDroppedCount(String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        COLUMN_NETWORK_LOGS_DROPPED_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }


    @Override
    public boolean addToUITracesTotalCount(@NonNull String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                int rowCount = addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_UI_TRACES_TOTAL_COUNT,
                        addedCount);
                return rowCount > 0;
            }
        }
        return false;
    }

    @Override
    public void addToUITracesDroppedCount(String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                addToMetric(sqLiteDatabaseWrapper,
                        sessionID,
                        COLUMN_UI_TRACES_DROPPED_COUNT,
                        addedCount);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void setExperimentsTotalCount(String sessionID, int count) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                setMetric(
                        sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXPERIMENTS_COUNT,
                        count
                );
            }
        }
    }

    @Override
    public void addToFragmentSpansTotalCount(@NonNull String sessionID, int count) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(
                        sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_FRAGMENT_SPANS_COUNT,
                        count
                );
            }
        }
    }

    @Override
    public void addToFragmentSpansDroppedCount(@NonNull String sessionID, int count) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(
                        sqLiteDatabaseWrapper,
                        sessionID,
                        InstabugDbContract.APMSessionMetaDataEntry.COLUMN_FRAGMENT_SPANS_DROPPED_COUNT,
                        count
                );
            }
        }
    }

    @Override
    public void addToComposeSpansTotalCount(@NonNull String sessionID, int addedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(
                        sqLiteDatabaseWrapper,
                        sessionID,
                        COLUMN_COMPOSE_SPANS_COUNT,
                        addedCount
                );
            }
        }
    }

    @Override
    public void addToComposeSpansDroppedCount(@NonNull String sessionID, int droppedCount) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                addToMetric(
                        sqLiteDatabaseWrapper,
                        sessionID,
                        COLUMN_COMPOSE_SPANS_DROPPED_COUNT,
                        droppedCount
                );
            }
        }
    }

    @Override
    public void addToWebViewTracesTotalCount(@NonNull String sessionID, int addedCount) {
        addCount(sessionID, addedCount, COLUMN_WEB_VIEW_TRACES_TOTAL_COUNT);
    }

    @Override
    public void addToWebViewTracesDroppedCount(@NonNull String sessionID, int droppedCount) {
        addCount(sessionID, droppedCount, COLUMN_WEB_VIEW_TRACES_DROPPED_COUNT);
    }

    @Override
    public void setAppFlowTotalCount(@NonNull String sessionID, int count) {
        setCount(sessionID, count, COLUMN_APM_APP_FLOW_COUNT);
    }

    @Override
    public void setAppFlowDroppedCount(@NonNull String sessionID, int count) {
        setCount(sessionID, count, COLUMN_APM_APP_FLOW_DROPPED_COUNT);
    }

    @Override
    public void resetAppLaunchesCounts() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APP_LAUNCH_TOTAL_COUNT, 0);
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_APP_LAUNCH_DROPPED_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
                database.close();
            }
        }
    }

    @Override
    public void resetTracesCounts() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXECUTION_TRACES_TOTAL_COUNT, 0);
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_EXECUTION_TRACES_DROPPED_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
                database.close();
            }
        }
    }

    @Override
    public void resetNetworkLogsCounts() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_NETWORK_LOGS_TOTAL_COUNT, 0);
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_NETWORK_LOGS_DROPPED_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
                database.close();
            }
        }
    }

    @Override
    public void resetUITracesCounts() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_UI_TRACES_TOTAL_COUNT, 0);
                contentValues.put(InstabugDbContract.APMSessionMetaDataEntry.COLUMN_UI_TRACES_DROPPED_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
                database.close();
            }
        }
    }

    @Override
    public void resetExperimentsCount() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_EXPERIMENTS_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
            }
        }
    }

    @Override
    public void resetFragmentSpansCounts() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_FRAGMENT_SPANS_COUNT, 0);
                contentValues.put(COLUMN_FRAGMENT_SPANS_DROPPED_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
            }
        }
    }

    @SuppressLint("Range")
    @Override
    public void resetComposeSpansCounts() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_COMPOSE_SPANS_COUNT, 0);
                contentValues.put(COLUMN_COMPOSE_SPANS_DROPPED_COUNT, 0);
                database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
            }
        }
    }

    @Override
    public void resetWebViewTracesCounts() {
        resetCounts(COLUMN_WEB_VIEW_TRACES_TOTAL_COUNT, COLUMN_WEB_VIEW_TRACES_DROPPED_COUNT);
    }

    @Override
    public void resetAppFlowCounts() {
        resetCounts(COLUMN_APM_APP_FLOW_COUNT, COLUMN_APM_APP_FLOW_DROPPED_COUNT);
    }

    @Override
    @Nullable
    public SessionMetaData getSessionMetaData(String sessionID) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            SessionMetaData sessionMetaData = null;
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            String selectionClause = COLUMN_SESSION_ID + " = ?";
            String[] selectionArgs = {sessionID};
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, null);
                if (cursor != null) {
                    if (cursor.moveToNext()) {
                        sessionMetaData = new SessionMetaData();
                        sessionMetaData.setAppLaunchesTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_APP_LAUNCH_TOTAL_COUNT)));
                        sessionMetaData.setExecutionTracesTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_EXECUTION_TRACES_TOTAL_COUNT)));
                        sessionMetaData.setNetworkLogsTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_NETWORK_LOGS_TOTAL_COUNT)));
                        sessionMetaData.setUiTracesTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_UI_TRACES_TOTAL_COUNT)));
                        sessionMetaData.setAppLaunchesDroppedCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_APP_LAUNCH_DROPPED_COUNT)));
                        sessionMetaData.setExecutionTracesDroppedCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_EXECUTION_TRACES_DROPPED_COUNT)));
                        sessionMetaData.setNetworkLogsDroppedCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_NETWORK_LOGS_DROPPED_COUNT)));
                        sessionMetaData.setUiTracesDroppedCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_UI_TRACES_DROPPED_COUNT)));
                        sessionMetaData.setExperimentsTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_EXPERIMENTS_COUNT)));
                        sessionMetaData.setFragmentSpansTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_FRAGMENT_SPANS_COUNT)));
                        sessionMetaData.setFragmentSpansDroppedCountRequestLimit(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_FRAGMENT_SPANS_DROPPED_COUNT)));
                        sessionMetaData.setComposeSpansTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_COMPOSE_SPANS_COUNT)));
                        sessionMetaData.setComposeSpansDroppedCountRequestLimit(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_COMPOSE_SPANS_DROPPED_COUNT)));
                        sessionMetaData.setWebViewTracesTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_WEB_VIEW_TRACES_TOTAL_COUNT)));
                        sessionMetaData.setWebViewTracesDroppedCountRequestLimit(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_WEB_VIEW_TRACES_DROPPED_COUNT)));
                        sessionMetaData.setAppFlowTotalCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_APM_APP_FLOW_COUNT)));
                        sessionMetaData.setAppFlowDroppedCountRequestLimit(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_APM_APP_FLOW_DROPPED_COUNT)));
                    }
                }
                databaseWrapper.close();
                return sessionMetaData;
            } catch (Exception ex) {
                reportAndLogError(ex);
            }catch (OutOfMemoryError oom) {
                reportAndLogError(oom);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    private void addCount(@NonNull String sessionID, int addedCount, @NonNull String column) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                try {
                    SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                    insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                    addToMetric(sqLiteDatabaseWrapper, sessionID, column, addedCount);
                } catch (Exception ex) {
                    reportAndLogError(ex);
                }catch (OutOfMemoryError oom) {
                    reportAndLogError(oom);
                }
            }
        }
    }

    private void setCount(@NonNull String sessionID, int addedCount, @NonNull String column) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                try {
                    SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                    insertRowIfNotExist(sqLiteDatabaseWrapper, sessionID);
                    setMetric(sqLiteDatabaseWrapper, sessionID, column, addedCount);
                } catch (Exception ex) {
                    reportAndLogError(ex);
                } catch (OutOfMemoryError oom) {
                    reportAndLogError(oom);
                }
            }
        }
    }

    private void resetCounts(String... columns) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            synchronized (lock) {
                try {
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    ContentValues contentValues = new ContentValues();
                    for (String column : columns) {
                        if (column != null) contentValues.put(column, 0);
                    }
                    database.update(InstabugDbContract.APMSessionMetaDataEntry.TABLE_NAME, contentValues, null, null);
                } catch (Exception ex) {
                    reportAndLogError(ex);
                } catch (OutOfMemoryError oom) {
                    reportAndLogError(oom);
                }

            }
        }
    }

    private void insertRowIfNotExist(SQLiteDatabaseWrapper sqLiteDatabaseWrapper, String sessionID) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_SESSION_ID, sessionID);
        sqLiteDatabaseWrapper.insertWithOnConflict(TABLE_NAME,
                null,
                contentValues);
    }

    private int addToMetric(SQLiteDatabaseWrapper sqLiteDatabaseWrapper, String sessionID, String metricColumnName, int count) {
        String whereClause = COLUMN_SESSION_ID + " = ?";
        String[] whereValues = {sessionID};
        Cursor cursor = null;
        int updatedRows = 0;
        try {
            cursor = sqLiteDatabaseWrapper.query(TABLE_NAME,
                    null,
                    whereClause,
                    whereValues,
                    null, null, null);
            if (cursor != null) {
                if (cursor.moveToFirst()) {
                    int oldCount = cursor.getInt(cursor.getColumnIndexOrThrow(metricColumnName));
                    cursor.close();
                    int totalCount = oldCount + count;
                    ContentValues contentValues = new ContentValues();
                    contentValues.put(COLUMN_SESSION_ID, sessionID);
                    contentValues.put(metricColumnName, totalCount);
                    updatedRows = sqLiteDatabaseWrapper.update(TABLE_NAME,
                            contentValues,
                            whereClause,
                            whereValues);
                }
            }
        } catch (Exception ex) {
            reportAndLogError(ex);
        } catch (OutOfMemoryError oom) {
            reportAndLogError(oom);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return updatedRows;
    }

    private void setMetric(SQLiteDatabaseWrapper sqLiteDatabaseWrapper, String sessionID, String metricColumnName, int count) {
        String whereClause = COLUMN_SESSION_ID + " = ?";
        String[] whereValues = {sessionID};
        try {
            ContentValues contentValues = new ContentValues();
            contentValues.put(COLUMN_SESSION_ID, sessionID);
            contentValues.put(metricColumnName, count);
            sqLiteDatabaseWrapper.update(TABLE_NAME,
                    contentValues,
                    whereClause,
                    whereValues);
        } catch (Exception ex) {
            reportAndLogError(ex);
        } catch (OutOfMemoryError oom) {
            reportAndLogError(oom);
        }
    }

    private void reportAndLogError(Throwable ex) {
        logger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
        IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
    }

}
