package com.instabug.apm.webview.webview_trace.flow

import com.instabug.apm.webview.webview_trace.model.event.WebViewEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewEventId.Companion.LOAD_URL
import com.instabug.apm.webview.webview_trace.model.event.WebViewEventId.Companion.OVERRIDE_URL_END
import com.instabug.apm.webview.webview_trace.model.event.WebViewEventId.Companion.OVERRIDE_URL_START
import com.instabug.apm.webview.webview_trace.model.event.WebViewOverrideUrlEndEvent


/**
 * Starting new trace flow
 * Here are the valid cases
 * any events -> Load url
 * Any events -> override url loading started -> loadUrl -> override url loading ended, force override is false
 */
class WebViewNewTraceFlow : WebViewEventFlow {

    companion object {
        private const val STATE_NONE: Byte = 0
        private const val STATE_FIRST_EVENT_RECEIVED: Byte = 1
        private const val STATE_OVERRIDE_URL_STARTED: Byte = 2
        private const val STATE_LOAD_URL: Byte = 3
        private const val STATE_FINISHED: Byte = 4
    }

    private var pendingEvent: WebViewEvent? = null
    private var state: Byte = STATE_NONE

    override fun process(event: WebViewEvent): Boolean =
        when (event.id) {
            OVERRIDE_URL_START -> processUrlOverrideStarted()
            LOAD_URL -> processLoadUrlEvent(event)
            OVERRIDE_URL_END -> processOverrideEnded(event)
            else -> processOtherEvent()
        }

    private fun processUrlOverrideStarted(): Boolean {
        state = STATE_OVERRIDE_URL_STARTED
        pendingEvent = null
        return true
    }

    private fun processOverrideEnded(event: WebViewEvent): Boolean {
        return if (
            state == STATE_LOAD_URL &&
            event.isNotForceOverride()
        ) {
            state = STATE_FINISHED
            false
        } else {
            resetState()
            true
        }
    }

    private fun WebViewEvent.isNotForceOverride() =
        this is WebViewOverrideUrlEndEvent &&
                !isForceOverride


    private fun processLoadUrlEvent(event: WebViewEvent): Boolean =
        when (state) {
            STATE_OVERRIDE_URL_STARTED -> {
                state = STATE_LOAD_URL
                pendingEvent = event.copy(isTerminationOverride = true)
                false
            }

            STATE_NONE -> {
                resetState()
                true
            }

            else -> {
                state = STATE_FINISHED
                pendingEvent = event.copy(isTerminationOverride = true)
                false
            }
        }

    private fun processOtherEvent(): Boolean {
        val stateIsAnyThingOtherThanUrlOverrideStartedAndLoadUrl =
            !(state == STATE_OVERRIDE_URL_STARTED || state == STATE_LOAD_URL)
        if (stateIsAnyThingOtherThanUrlOverrideStartedAndLoadUrl) {
            resetState()
        }
        return true
    }

    private fun resetState() {
        state = STATE_FIRST_EVENT_RECEIVED
        pendingEvent = null
    }

    override fun finished(): Boolean {
        return state == STATE_FINISHED
    }

    override fun getPendingEvent(): WebViewEvent? =
        pendingEvent?.takeIf { finished() }
}