package com.instabug.apm.uitrace.handler

import android.app.Activity
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.handler.session.APMSessionObserver
import com.instabug.apm.handler.session.SessionObserverHandler
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks
import com.instabug.apm.uitrace.model.UiTraceInitParams
import com.instabug.apm.uitrace.repo.UiTracesRepo
import com.instabug.apm.uitrace.util.getUiTraceEndParams
import com.instabug.apm.uitrace.util.logAutomaticUiTraceSuccessfullyEnded
import com.instabug.apm.uitrace.util.logAutomaticUiTraceSuccessfullyStarted
import com.instabug.apm.util.device.APMDeviceStateProvider
import com.instabug.apm.util.runOrReportAPMError
import com.instabug.apm.util.view.InstabugViews
import com.instabug.library.model.common.Session
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import java.util.concurrent.Executor

interface CPAutomaticUiTraceHandler {
    fun onScreenChanged(
        screenName: String?,
        timeStampMicro: Long,
        uiTraceId: Long
    )
}

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
class CPAutomaticUiTracesHandlerImpl(
    private val internalTrackingDelegate: InstabugInternalTrackingDelegate,
    private val executor: Executor,
    private val configurationProvider: APMConfigurationProvider,
    private val repo: UiTracesRepo,
    private val logger: Logger,
    private val deviceStateProvider: APMDeviceStateProvider
) : CPAutomaticUiTraceHandler,
    APMUiTraceActivityCallbacks, APMSessionObserver {

    private var currentScreenName = ""

    private val String.canHandleUiTrace
        get() = isNotBlank() && configurationProvider.isUiTraceEnabled

    override fun observeAPMSessions() =
        runOrReportAPMError("error while registering cp ui trace handler as SessionObserver", logger) {
            SessionObserverHandler.register(this)
        }

    override fun stopObservingAPMSessions() =
        runOrReportAPMError("error while unregistering cp ui trace handler as SessionObserver", logger) {
            SessionObserverHandler.unregister(this)
        }

    override fun onScreenChanged(
        screenName: String?,
        timeStampMicro: Long,
        uiTraceId: Long
    ) = executor.execute {
        runOrReportAPMError("Error while handling cp screen changed while starting ui trace for $screenName", logger) {
            screenName?.takeIf { it.canHandleUiTrace }?.let {
                if (currentScreenName.isBlank())
                    startUiTraceIfPossible(screenName, timeStampMicro, uiTraceId)
                else
                    handleScreenChanged(screenName, timeStampMicro, uiTraceId)
                currentScreenName = screenName
            }
        }
    }

    private fun handleScreenChanged(
        screenName: String,
        timestampMicro: Long,
        uiTraceId: Long
    ) {
        endUiTraceIfPossible(timestampMicro)
        startUiTraceIfPossible(screenName, timestampMicro, uiTraceId)
    }

    private fun endUiTraceIfPossible(elapsedTime: Long) {
        doTraceOperationIfPossible("Can not stop tracing the current screen because Activity is null") { activity ->
            endUiTrace(currentScreenName, activity, elapsedTime)
        }
    }

    private fun startUiTraceIfPossible(
        screenName: String,
        timeStampMicros: Long,
        uiTraceId: Long
    ) =
        doTraceOperationIfPossible("Can not trace the current screen because Activity is null") { activity ->
            repo.start(
                screenName,
                getUiTraceInitParams(uiTraceId, timeStampMicros, activity, screenName)
            )
            logger.logAutomaticUiTraceSuccessfullyStarted(screenName)
        }

    private fun getUiTraceInitParams(
        uiTraceId: Long,
        timeStampMicros: Long,
        activity: Activity,
        screenName: String
    ) = UiTraceInitParams(
        uiTraceId,
        timeStampMicros,
        timeStampMicros,
        deviceStateProvider.getBatteryLevel(activity),
        deviceStateProvider.isPowerSaveModeEnabled(activity),
        deviceStateProvider.getScreenOrientation(activity),
        screenName,
        screenName,
        false
    )

    override fun onActivityPaused(activity: Activity, timeMetric: EventTimeMetricCapture) =
        executor.execute {
            runOrReportAPMError("error while handling cp ui trace $currentScreenName at onActivityPaused", logger) {
                currentScreenName
                    .takeIf { it.canHandleUiTrace }
                    ?.takeUnless { InstabugViews.isInstabugActivity(activity) }
                    ?.let { endUiTrace(it, activity, timeMetric.getTimeStampMicro()) }
            }
        }


    private inline fun doTraceOperationIfPossible(
        message: String,
        operation: (Activity) -> Unit
    ) = internalTrackingDelegate
        .currentActivity
        .also { if (it == null) logger.d(message) }
        ?.takeUnless { InstabugViews.isInstabugActivity(it) }
        ?.let { operation(it) }

    private fun endUiTrace(screenName: String, activity: Activity, elapsedTimeMicro: Long) {
        repo.end(screenName, activity.getUiTraceEndParams(deviceStateProvider, elapsedTimeMicro))
            ?.let { logger.logAutomaticUiTraceSuccessfullyEnded(screenName, it) }
    }

    override fun onNewSessionStarted(runningSession: Session, lastSession: Session?) =
        executor.execute {
            runOrReportAPMError("error while handling cp ui traces on new session start", logger) {
                if (configurationProvider.isUiTraceEnabled)
                    repo.setSessionIdAndSaveIfPossible(runningSession.id)
            }
        }
}