package com.instabug.apm.screenloading.di

import android.annotation.SuppressLint
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.apm.cache.model.UiLoadingModel
import com.instabug.apm.di.ContextProvider
import com.instabug.apm.configuration.FeatureAvailabilityChecker
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.di.getOrCreateSingleton
import com.instabug.apm.sanitization.Validator
import com.instabug.apm.screenloading.handler.CPScreenLoadingHandlerImpl
import com.instabug.apm.screenloading.handler.NativeScreenLoadingHandlerFactory
import com.instabug.apm.screenloading.handler.NativeScreenLoadingHandlerImpl
import com.instabug.apm.screenloading.manager.ScreenLoadingManager
import com.instabug.apm.screenloading.manager.ScreenLoadingManagerImpl
import com.instabug.apm.screenloading.repo.NativeScreenLoadingRepo
import com.instabug.apm.screenloading.repo.NativeScreenLoadingRepoFactory
import com.instabug.apm.screenloading.validator.EndScreenLoadingConfigurationValidator
import com.instabug.apm.screenloading.ScreenLoadingFeatureAvailabilityChecker
import com.instabug.apm.screenloading.validator.NativeScreenLoadingStagesValidator
import com.instabug.apm.uitrace.di.UiTracesServiceLocator
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.settings.SettingsManager

object ScreenLoadingServiceLocator {

    private const val SCREEN_LOADING_EXECUTOR_ID = "screen-loading-executor"

    private var _screenLoadingManager: ScreenLoadingManager? = null

    private val isBelowJellybean
        get() = BuildFieldsProvider.provideBuildVersion() < Build.VERSION_CODES.JELLY_BEAN

    val endScreenLoadingConfigurationValidator: Validator<Unit>
        get() = EndScreenLoadingConfigurationValidator(
            ServiceLocator.getApmConfigurationProvider(),
            ServiceLocator.getApmLogger()
        )

    val nativeScreenLoadingStagesValidator: Validator<UiLoadingModel>
        get() = NativeScreenLoadingStagesValidator()

    val ScreenLoadingFeatureAvailabilityChecker: FeatureAvailabilityChecker
        get() = ScreenLoadingFeatureAvailabilityChecker(
            ServiceLocator.getApmConfigurationProvider(),
            ServiceLocator.getApmLogger()
        )

    @JvmStatic
    val manager: ScreenLoadingManager?
        @SuppressLint("NewApi")
        get() = getOrCreateSingleton(
            ::_screenLoadingManager,
            { _screenLoadingManager = it },
            { if (isBelowJellybean) null else createScreenLoadingManager() }
        )

    private val screenLoadingSingleThreadExecutor
        get() = ServiceLocator.getSingleThreadExecutor(SCREEN_LOADING_EXECUTOR_ID)

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    private fun createScreenLoadingManager(): ScreenLoadingManager = ScreenLoadingManagerImpl(
        ServiceLocator.getApmConfigurationProvider(),
        screenLoadingSingleThreadExecutor,
        SettingsManager.getInstance(),
        NativeScreenLoadingFactoryProvider(),
        CPScreenLoadingHandlerProvider(),
        UiTracesServiceLocator.compositeApmUiTraceActivityCallbacks,
        NativeScreenLoadingRepoFactory(),
        ServiceLocator.getUiLoadingMetricCacheHandler(),
        ServiceLocator.getApmLogger(),
        ContextProvider(),
        EndScreenLoadingConfigurationValidatorProvider()
    )

    fun createNativeScreenLoadingHandlerFactory(): ParameterizedFactory<NativeScreenLoadingHandlerImpl, NativeScreenLoadingRepo> =
        NativeScreenLoadingHandlerFactory(
            screenLoadingSingleThreadExecutor,
            MainThreadHandlerProvider(),
            ServiceLocator.getApmConfigurationProvider(),
            ServiceLocator.getApmLogger()
        )

    fun createCpScreenLoadingHandler(): CPScreenLoadingHandlerImpl = CPScreenLoadingHandlerImpl(
        ServiceLocator.getApmConfigurationProvider(),
        ServiceLocator.getUiLoadingMetricCacheHandler(),
        screenLoadingSingleThreadExecutor,
        ServiceLocator.getApmLogger()
    )
}