package com.instabug.apm.handler.fragment

import com.instabug.apm.cache.handler.fragments.FragmentSpansCacheHandler
import com.instabug.apm.cache.handler.fragments.FragmentSpansEventsCacheHandler
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.cache.model.FragmentSpansCacheModel
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.fragment.model.FragmentSpans
import com.instabug.apm.logger.internal.Logger


interface FragmentSpansHandler {
    fun saveFragment(fragmentSpans: FragmentSpans): Boolean
    fun getFragmentsForSession(sessionId: String): List<FragmentSpansCacheModel>
    fun clearFragmentsForSession(sessionId: String)
    fun clearCache()
}

class FragmentSpansHandlerImpl : FragmentSpansHandler {

    private val fragmentSpansCacheHandler: FragmentSpansCacheHandler
        get() = ServiceLocator.getFragmentSpansCacheManager()
    private val fragmentSpansEventsCacheHandler: FragmentSpansEventsCacheHandler
        get() = ServiceLocator.getFragmentSpansEventsCacheHandler()
    private val sessionMetaDataCacheHandler: SessionMetaDataCacheHandler?
        get() = ServiceLocator.getSessionMetaDataCacheHandler()
    private val apmConfigurationProvider: APMConfigurationProvider
        get() = ServiceLocator.getApmConfigurationProvider()
    private val apmLogger: Logger get() = ServiceLocator.getApmLogger()

    override fun saveFragment(fragmentSpans: FragmentSpans): Boolean {
        synchronized(this) {
            var saved = false
            takeIf { canSaveFragment() }?.let {
                fragmentSpansCacheHandler.saveFragment(fragmentSpans)
            }?.takeIf { id ->
                id != -1L
            }?.also { fragmentId ->
                saved = true
                fragmentSpansEventsCacheHandler.saveEvents(fragmentSpans.events, fragmentId)
                sessionMetaDataCacheHandler?.addToFragmentSpansTotalCount(
                    fragmentSpans.sessionId,
                    1
                )
            }?.let {
                val deletedRowsCount = fragmentSpansCacheHandler.trimToLimit(
                    fragmentSpans.sessionId,
                    apmConfigurationProvider.fragmentSpansLimitPerRequest
                )
                fragmentSpansCacheHandler.trimToLimit(apmConfigurationProvider.fragmentSpansStoreLimit)
                deletedRowsCount
            }?.takeIf { deletedRowsCount ->
                deletedRowsCount > 0
            }?.also { deletedRowsCount ->
                sessionMetaDataCacheHandler?.addToFragmentSpansDroppedCount(
                    fragmentSpans.sessionId,
                    deletedRowsCount
                )
                apmLogger.d("Fragment spans dropped count: $deletedRowsCount")
            }
            return saved
        }
    }

    override fun getFragmentsForSession(sessionId: String): List<FragmentSpansCacheModel> {
        synchronized(this) {
            return fragmentSpansCacheHandler.getFragmentsForSession(sessionId)
                .takeIf {
                    it.isNotEmpty()
                }?.onEach { fragment ->
                    fragment.events.addAll(
                        fragmentSpansEventsCacheHandler.getEventsForFragment(
                            fragment.id
                        )
                    )
                } ?: listOf()
        }
    }

    override fun clearFragmentsForSession(sessionId: String) {
        synchronized(this) {
            fragmentSpansCacheHandler.deleteFragmentsForSession(sessionId)
        }
    }

    override fun clearCache() {
        synchronized(this) {
            fragmentSpansCacheHandler.clearAll()
            sessionMetaDataCacheHandler?.resetFragmentSpansCounts()
        }
    }

    private fun canSaveFragment() =
        apmConfigurationProvider.isFragmentSpansEnabled && apmConfigurationProvider.isAPMEnabled
}