package com.instabug.apm.compose.compose_spans.handler

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.compose.compose_spans.model.ComposeSpansCacheModel
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMComposeSpansEntry
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper
import com.instabug.library.map.Mapper
import com.instabug.library.parse.Parser

class ComposeSpansCacheHandlerImpl(
    private val databaseManager: DatabaseManager,
    private val modelContentValuesMapper: Mapper<Pair<ComposeSpansCacheModel, String>, ContentValues>,
    private val cursorParser: Parser<Cursor?, List<ComposeSpansCacheModel>?>,
    private val logger: Logger
): ComposeSpansCacheHandler {

    private val databaseWrapper: SQLiteDatabaseWrapper
        get() = databaseManager.openDatabase()

    override fun insert(model: ComposeSpansCacheModel, sessionId: String): Long =
        runCatching {
            databaseWrapper.insert(
                APMComposeSpansEntry.TABLE_NAME,
                null,
                modelContentValuesMapper.map(Pair(model, sessionId))
            )
        }.onFailure {
            it.reportAndLog()
        }.getOrNull() ?: -1L

    override fun retrieve(sessionId: String): List<ComposeSpansCacheModel>? =
        runCatching {
            val cursor = databaseWrapper.query(
                APMComposeSpansEntry.TABLE_NAME,
                null,
                "${APMComposeSpansEntry.COLUMN_SESSION_ID} = ?",
                arrayOf(sessionId),
                null, null, null
            )
            try {
                cursorParser.parse(cursor)
            } finally {
                cursor?.close()
            }
        }.onFailure {
            it.reportAndLog()
        }.getOrNull()

    override fun trim(sessionId: String, limit: Int): Int =
        kotlin.runCatching {
            val selectByLimitDescendingQuery =
                "SELECT ${APMComposeSpansEntry.COLUMN_ID} " +
                        "FROM ${APMComposeSpansEntry.TABLE_NAME} " +
                        "WHERE ${APMComposeSpansEntry.COLUMN_SESSION_ID} = ? " +
                        "ORDER BY ${APMComposeSpansEntry.COLUMN_ID} DESC " +
                        " LIMIT ?"
            val whereClause =
                "${APMComposeSpansEntry.COLUMN_SESSION_ID} = ? AND " +
                        "${APMComposeSpansEntry.COLUMN_ID} NOT IN ($selectByLimitDescendingQuery)"
            val whereArgs = arrayOf(sessionId, sessionId, limit.toString())
            databaseWrapper.delete(
                APMComposeSpansEntry.TABLE_NAME,
                whereClause,
                whereArgs
            )
        }.onFailure {
            it.reportAndLog()
        }.getOrNull() ?: 0

    override fun trimAll(limit: Int) {
        kotlin.runCatching {
            val selectByLimitDescendingQuery =
                "SELECT ${APMComposeSpansEntry.COLUMN_ID} " +
                        "FROM ${APMComposeSpansEntry.TABLE_NAME} " +
                        "ORDER BY ${APMComposeSpansEntry.COLUMN_ID} DESC " +
                        " LIMIT ?"
            val whereClause =
                "${APMComposeSpansEntry.COLUMN_ID} NOT IN ($selectByLimitDescendingQuery)"
            val whereArgs = arrayOf(limit.toString())
            databaseWrapper.delete(
                APMComposeSpansEntry.TABLE_NAME,
                whereClause,
                whereArgs
            )
        }.onFailure {
            it.reportAndLog()
        }
    }

    override fun clear(sessionId: String) {
        runCatching {
            databaseWrapper.delete(
                APMComposeSpansEntry.TABLE_NAME,
                "${APMComposeSpansEntry.COLUMN_SESSION_ID} = ?",
                arrayOf(sessionId),
            )
        }.onFailure {
            it.reportAndLog()
        }
    }

    override fun clearAll() {
        runCatching {
            databaseWrapper.delete(APMComposeSpansEntry.TABLE_NAME, null, null)
        }.onFailure {
            it.reportAndLog()
        }
    }

    private fun Throwable.reportAndLog() {
        val errorMessage = "ComposeSpans Database error"
        logger.logSDKErrorProtected(errorMessage, this)
        IBGDiagnostics.reportNonFatal(this, errorMessage)

    }
}