package com.instabug.apm.cache.handler.networklog;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.model.APMNetworkLog;

import java.util.List;
import java.util.Map;

public interface NetworkLogCacheHandler {

    /**
     * Inserts network log in cache
     *
     * @param networkLog network log to be cached
     * @return id of cached network log
     */
    long insertNetworkLog(String sessionID, APMNetworkLog networkLog);

    /**
     * Update cached network log values
     *
     * @param networkLog updated network log object
     */
    void updateNetworkLog(APMNetworkLog networkLog);

    /**
     * Removes invalid network logs
     * Invalid network log doesn't have (status code, error code, error message)
     */
    void cleanUp();

    /**
     * Keeps only last x number of records in cache and deletes other records
     *
     * @param limit number of logs to keep
     */
    void trimToLimit(long limit);

    /**
     * Removes oldest network logs till
     * total network logs count for the passed
     * sessionID reaches the limit passed
     *
     * @param sessionID Session id to limit its traces
     * @param limit     number of network logs to be remained
     */
    int trimToLimit(String sessionID, long limit);

    /**
     * Gets list of cached network logs by limit
     *
     * @param limit max count of network logs to be retrieved
     * @return list of cached network logs. Empty list in case of no cached logs
     */
    @Nullable
    List<APMNetworkLog> getNetworkLogsByLimit(long limit);

    /**
     * Removes cached network logs by limit
     *
     * @param limit max count of network logs to be removed
     */
    void deleteNetworkLogsByLimit(long limit);

    /**
     * Removes all cached network logs
     */
    void removeAll();

    /**
     * Removes all graphQl related data
     */
    void removeGraphQlData();

    /**
     * Removes all GRPC related records
     */
    void removeGrpcData();

    /**
     * Gets list of cached ended network logs for passed session id
     *
     * @param sessionID id of the session to get cached network
     *                  logs for
     * @return List of {@link APMNetworkLog}
     */
    @Nullable
    List<APMNetworkLog> getEndedNetworkLogsForSession(String sessionID);

    /**
     * Adds new attribute to specific trace
     *
     * @param traceId   id of the associated trace
     * @param traceName name of the associated trace used for logging
     * @param key       attribute key
     * @param value     attribute value
     */
    void addAttribute(long traceId, @NonNull String traceName, @NonNull String key, @Nullable String value);

    /**
     * Adds new attribute to specific trace
     * @param traceId   id of the associated trace
     * @param traceName name of the associated trace used for logging
     * @param attributes    map of attributes keys and values
     */
    void addAttributes(long traceId, String traceName, @Nullable Map<String, String> attributes);
    /**
     * Get attribute map of specific trace
     *
     * @param traceId   id of the associated trace
     * @return Map of attributes' keys and values
     */
    Map<String, String> getTraceAttributes(long traceId);

    /**
     * Removes all W3C external trace id related data including generated and captured.
     */
    void clearW3CExternalTraceIdCache();

    /**
     * Removes generated W3C trace id.
     */
    void clearGeneratedW3CExternalTraceIdCache();

    /**
     * Removes captured W3C trace id.
     */
    void clearCapturedW3CExternalTraceIdCache();
}
