package com.instabug.apm.appflow.validate

import com.instabug.apm.appflow.configuration.APP_FLOW_ATTRIBUTE_KEY_MAX_LENGTH
import com.instabug.apm.appflow.configuration.APP_FLOW_ATTRIBUTE_VALUE_MAX_LENGTH
import com.instabug.apm.appflow.log.logAttributeKeyExceededMaxLength
import com.instabug.apm.appflow.log.logAttributeKeyIsNullOrEmpty
import com.instabug.apm.appflow.log.logAttributeValueExceededMaxLength
import com.instabug.apm.appflow.log.logAttributeValueIsEmpty
import com.instabug.apm.appflow.model.AppFlowAttribute
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.sanitization.Validator

class AppFlowAttributeValidator(
    private val appFLowNameValidator: Validator<String?>,
    private val logger: Logger
) : Validator<AppFlowAttribute> {

    override fun isValid(item: AppFlowAttribute): Boolean =
        appFLowNameValidator.isValid(item.name) &&
                validateAttributeKey(item) &&
                validateAttributeValue(item)

    private fun validateAttributeKey(item: AppFlowAttribute): Boolean =
        item.key
            ?.trim()
            ?.takeIf { it.isNotEmpty() }
            ?.let { validateKeyLengthAndLogIfInvalid(item.name, item.key) }
            ?: let {
                logger.logAttributeKeyIsNullOrEmpty(item.name)
                false
            }

    private fun validateKeyLengthAndLogIfInvalid(flowName: String?, key: String) =
        (key.length <= APP_FLOW_ATTRIBUTE_KEY_MAX_LENGTH)
            .also { sizeValid ->
                takeUnless { sizeValid }
                    ?.let { logger.logAttributeKeyExceededMaxLength(flowName, key) }
            }

    private fun validateAttributeValue(item: AppFlowAttribute): Boolean =
        item.value
            ?.trim()
            ?.let { trimmedValue ->
                validateValueIsNotEmptyAndLogIfNot(item.name, item.key, trimmedValue) &&
                        validateValueLengthAndLogIfInvalid(item.name, item.key, trimmedValue)
            }
            ?: true

    private fun validateValueIsNotEmptyAndLogIfNot(flowName: String?, key: String?, value: String) =
        value.isNotEmpty()
            .also { isNotEmpty ->
                takeUnless { isNotEmpty }?.let { logger.logAttributeValueIsEmpty(flowName, key) }
            }

    private fun validateValueLengthAndLogIfInvalid(flowName: String?, key: String?, value: String) =
        (value.length <= APP_FLOW_ATTRIBUTE_VALUE_MAX_LENGTH)
            .also { sizeValid ->
                takeUnless { sizeValid }
                    ?.let { logger.logAttributeValueExceededMaxLength(flowName, key) }
            }


}