package com.instabug.apm;

import static com.instabug.apm.constants.ErrorMessages.APM_NOT_ENABLED_FEATURE_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.APP_LAUNCH_NOT_ENABLED_APM_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.APP_LAUNCH_NOT_ENABLED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.APP_LAUNCH_NOT_ENABLED_FEATURE_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.ENABLE_FRAGMENTS_SPANS_NOT_CALLED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.ENABLE_FRAGMENTS_SPANS_NOT_CALLED_FEATURE_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.END_APP_LAUNCH_NOT_CALLED_APM_FEATURE_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.END_APP_LAUNCH_NOT_CALLED_APM_SDK_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_ATTRIBUTE_NOT_ADDED_APM_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_ATTRIBUTE_NOT_ADDED_APM_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_ATTRIBUTE_NOT_ADDED_NETWORK_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_ATTRIBUTE_NOT_REMOVED_APM_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_ATTRIBUTE_NOT_REMOVED_APM_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_ATTRIBUTE_NOT_REMOVED_NETWORK_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.TRACE_NAME_INVALID_LENGTH;
import static com.instabug.apm.constants.ErrorMessages.TRACE_NAME_NULL_OR_EMPTY;
import static com.instabug.apm.constants.ErrorMessages.UI_AUTO_TRACING_NOT_ENABLED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_AUTO_TRACING_NOT_ENABLED_FEATURE_NOT_AVAILABLE;
import static com.instabug.apm.constants.ErrorMessages.UI_HANGS_NOT_ENABLED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_HANGS_NOT_ENABLED_BE_FLAG_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_HANGS_NOT_ENABLED_TRACES_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_LOADING_NOT_ENABLED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_LOADING_NOT_ENABLED_BE_FLAG_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_LOADING_NOT_ENABLED_TRACES_DISABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_END_NOT_STARTED;
import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_NAME_INVALID_LENGTH;
import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_NAME_NULL_OR_EMPTY;
import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_NOT_STARTED_APM_NOT_ENABLED;
import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_NOT_STARTED_BACKGROUND_THREAD;
import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_NOT_STARTED_FEATURE_NOT_AVAILABLE;

import android.app.Activity;
import android.os.Build;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.apm.appflow.di.AppFlowServiceLocator;
import com.instabug.apm.appflow.manager.AppFlowManager;
import com.instabug.apm.appflow.model.AppFlowAttribute;
import com.instabug.apm.attributes.listeners.OnNetworkTraceListener;
import com.instabug.apm.compose.compose_spans.ComposeSpansManager;
import com.instabug.apm.compose.compose_spans.ComposeSpansServiceLocator;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.configuration.APMStateProvider;
import com.instabug.apm.configuration.cp.APMCPConfigurationProvider;
import com.instabug.apm.configuration.cp.APMFeature;
import com.instabug.apm.configuration.cp.FeatureAvailabilityCallback;
import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.constants.DefaultValues;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.eventbus.APMSdkStateEventBus;
import com.instabug.apm.fragment.FragmentSpansHelper;
import com.instabug.apm.handler.applaunch.AppLaunchesHandler;
import com.instabug.apm.handler.executiontraces.ExecutionTracesHandler;
import com.instabug.apm.handler.fragment.FragmentSpansHandler;
import com.instabug.apm.handler.networklog.NetworkLogHandler;
import com.instabug.apm.handler.uitrace.customuitraces.CustomUiTraceHandler;
import com.instabug.apm.lifecycle.AppLaunchLifeCycleCallbacks;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.EventTimeMetricCapture;
import com.instabug.apm.model.ExecutionTrace;
import com.instabug.apm.model.TimeCaptureBoundModel;
import com.instabug.apm.networkinterception.repository.NetworkInterceptionRepository;
import com.instabug.apm.screenloading.di.ScreenLoadingServiceLocator;
import com.instabug.apm.screenloading.manager.ScreenLoadingManager;
import com.instabug.apm.uitrace.di.UiTracesServiceLocator;
import com.instabug.apm.uitrace.manager.UiTracesManager;
import com.instabug.apm.webview.webview_trace.manager.WebViewTraceManager;
import com.instabug.library.BuildFieldsProvider;
import com.instabug.library.logging.listeners.networklogs.NetworkLogListener;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.threading.PoolProvider;

public class APMImplementation {

    private final Logger apmLogger;

    public APMImplementation(Logger apmLogger) {
        this.apmLogger = apmLogger;
    }

    /**
     * Changes APM SDK state
     * APM won't be enabled if APM is disabled by BE feature flag
     *
     * @param enabled true to enable APM, false to disable APM
     */
    public void setEnabled(boolean enabled) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (!apmConfigurationProvider.isAPMFeatureAvailable() && enabled) {
            apmLogger.logSDKError(APM_NOT_ENABLED_FEATURE_NOT_AVAILABLE);
        } else {
            apmConfigurationProvider.setAPMSdkEnabled(enabled);
            APMSdkStateEventBus.INSTANCE.post(enabled);
            if (!enabled) {
                clearCache();
                ServiceLocator.reset();
            }
        }
    }

    /**
     * Clears APM cache
     */
    //TODO: Move this to be called on event bus
    public void clearCache() {
        clearExecutionTracesCache();
        clearAppLaunchesCache();
        clearNetworkCache();
        clearFragmentSpansCache();
        clearApmState();
    }

    public void clearFragmentSpansCache() {
        FragmentSpansHandler fragmentSpansHandler = ServiceLocator.getFragmentSpansHandler();
        if (fragmentSpansHandler != null) {
            fragmentSpansHandler.clearCache();
        }
    }

    public void clearNetworkCache() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR).execute(new Runnable() {
            @Override
            public void run() {
                networkLogHandler.removeAll();
            }
        });
    }

    public void clearGraphQlCache() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        if (networkLogHandler != null) {
            ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR).execute(new Runnable() {
                @Override
                public void run() {
                    networkLogHandler.removeGraphQlData();
                }
            });
        }
    }

    public void clearGrpcCache() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        if (networkLogHandler != null) {
            ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR)
                    .execute(networkLogHandler::removeGrpcData);
        }
    }

    public void clearW3CNetworkExternalTraceIdCache() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR)
                .execute(networkLogHandler::clearW3CExternalTraceIdCache);
    }

    public void clearGeneratedW3CExternalTraceIdCache() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR)
                .execute(networkLogHandler::clearGeneratedW3CExternalTraceIdCache);
    }

    public void clearCapturedW3CExternalTraceIdCache() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR)
                .execute(networkLogHandler::clearCapturedW3CExternalTraceIdCache);
    }

    public void clearExecutionTracesCache() {
        final ExecutionTracesHandler executionTracesHandler = ServiceLocator.getExecutionTracesHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.EXECUTION_TRACES_THREAD_EXECUTOR).execute(new Runnable() {
            @Override
            public void run() {
                executionTracesHandler.removeAll();
            }
        });
    }


    public void clearAppLaunchesCache(@NonNull @AppLaunchType final String type) {
        clearAppLaunchesCache(type, false);
    }

    /**
     * Clear app launch cache data by type, and if clearAppLaunchCounter is passed with true, it
     * will clear stored app launch counters
     */
    public void clearAppLaunchesCache(
            @NonNull @AppLaunchType final String type,
            final boolean clearAppLaunchCounts
    ) {
        final AppLaunchesHandler appLaunchesHandler = ServiceLocator.getAppLaunchesHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.APP_LAUNCH_THREAD_EXECUTOR).execute(new Runnable() {
            @Override
            public void run() {
                appLaunchesHandler.removeAppLaunches(type, clearAppLaunchCounts);
            }
        });
    }

    public void clearAppLaunchesCache() {
        final AppLaunchesHandler appLaunchesHandler = ServiceLocator.getAppLaunchesHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.APP_LAUNCH_THREAD_EXECUTOR).execute(new Runnable() {
            @Override
            public void run() {
                appLaunchesHandler.removeAll();
            }
        });
    }

    public void clearApmState() {
        APMStateProvider stateProvider = ServiceLocator.getApmStateProvider();
        if (stateProvider != null) {
            stateProvider.clearState();
        }
    }

    /**
     * Changes state of cold app launch tracking
     * Cold app launch tracking won't be enabled if APM is disabled
     *
     * @param enabled true to enable cold app launch tracking, false to disable it
     */
    public void setColdAppLaunchEnabled(boolean enabled) {
        validateAndSetAppLaunchEnabled(AppLaunchType.COLD, enabled);
    }

    /**
     * Changes state of hot app launch tracking
     * Hot app launch tracking won't be enabled if APM is disabled
     *
     * @param enabled true to enable hot app launch tracking, false to disable it
     */
    public void setHotAppLaunchEnabled(boolean enabled) {
        validateAndSetAppLaunchEnabled(AppLaunchType.HOT, enabled);
    }

    /**
     * Changes state of warm app launch tracking
     * Warm app launch tracking won't be enabled if APM is disabled
     *
     * @param enabled true to enable warm app launch tracking, false to disable it
     */
    public void setWarmAppLaunchEnabled(boolean enabled) {
        validateAndSetAppLaunchEnabled(AppLaunchType.WARM, enabled);
    }

    private void validateAndSetAppLaunchEnabled(@AppLaunchType String type, boolean enabled) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        String launchType = getPrintableAppLaunchType(type);
        String isEnabled = enabled ? "enabled" : "disabled";
        if (!apmConfigurationProvider.isAPMSdkEnabled()) {
            apmLogger.logSDKError(APP_LAUNCH_NOT_ENABLED_APM_NOT_ENABLED.
                    replace("\"$s1\"", launchType).
                    replace("\"$s2\"", isEnabled));
        } else if (!apmConfigurationProvider.isAPMFeatureAvailable()) {
            apmLogger.logSDKError(APP_LAUNCH_NOT_ENABLED_APM_NOT_AVAILABLE.
                    replace("\"$s1\"", launchType).
                    replace("\"$s2\"", isEnabled));
        } else if (!isAppLaunchFeatureEnabled(type)) {
            apmLogger.logSDKError(APP_LAUNCH_NOT_ENABLED_FEATURE_NOT_AVAILABLE.
                    replace("\"$s1\"", launchType).
                    replace("\"$s2\"", isEnabled));
        } else {
            setAppLaunchSdkEnabled(type, enabled);
            if (!enabled) {
                boolean appLaunchIsDisabled = !apmConfigurationProvider.isAppLaunchesEnabled();
                clearAppLaunchesCache(type, appLaunchIsDisabled);
            }
        }
    }

    private String getPrintableAppLaunchType(@AppLaunchType String type) {
        if (AppLaunchType.COLD.equals(type)) {
            return "Cold";
        }
        if (AppLaunchType.HOT.equals(type)) {
            return "Hot";
        }
        return "Warm";
    }

    private boolean isAppLaunchFeatureEnabled(@AppLaunchType String type) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (AppLaunchType.COLD.equals(type)) {
            return apmConfigurationProvider.isColdAppLaunchesFeatureEnabled();
        }
        if (AppLaunchType.HOT.equals(type)) {
            return apmConfigurationProvider.isHotAppLaunchesFeatureEnabled();
        }
        return apmConfigurationProvider.isWarmAppLaunchFeatureEnabled();
    }

    private void setAppLaunchSdkEnabled(@AppLaunchType String type, boolean enabled) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (AppLaunchType.COLD.equals(type)) {
            apmConfigurationProvider.setColdAppLaunchSDKEnabled(enabled);
        } else if (AppLaunchType.HOT.equals(type)) {
            apmConfigurationProvider.setHotAppLaunchSDKEnabled(enabled);
        } else if (AppLaunchType.WARM.equals(type)) {
            apmConfigurationProvider.setWarmAppLaunchSdkEnabled(enabled);
        }
    }

    /**
     * Creates and starts a new trace with the given name
     *
     * @param name trace name
     * @return created trace instance. Null is returned if trace name is null or empty
     */
    @Nullable
    public ExecutionTrace startExecutionTrace(@NonNull String name) {
        if (name == null || name.trim().isEmpty()) {
            apmLogger.logSDKError(TRACE_NAME_NULL_OR_EMPTY);
            return null;
        }
        String trimmedName = name.trim();
        if (trimmedName.length() > DefaultValues.Traces.TRACE_NAME_LENGTH) {
            trimmedName = trimmedName.substring(0, DefaultValues.Traces.TRACE_NAME_LENGTH);
            apmLogger.logSDKWarning(TRACE_NAME_INVALID_LENGTH.replace("$s", name));
        }
        return new ExecutionTrace(trimmedName);
    }

    public void setAutoUITraceEnabled(boolean enabled) {
        if (BuildFieldsProvider.INSTANCE.provideBuildVersion() < android.os.Build.VERSION_CODES.JELLY_BEAN) {
            apmLogger.logSDKError("Could not enable Auto UI Trace. Feature is supported on API level 16 and up only.");
            return;
        }

        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (!apmConfigurationProvider.isAPMEnabled() && enabled) {
            apmLogger.logSDKError(UI_AUTO_TRACING_NOT_ENABLED_APM_NOT_ENABLED);
        } else if (!apmConfigurationProvider.isUiTraceFeatureEnabled() && enabled) {
            apmLogger.logSDKError(UI_AUTO_TRACING_NOT_ENABLED_FEATURE_NOT_AVAILABLE);
        } else {
            apmConfigurationProvider.setUiTraceSdkEnabled(enabled);
        }
        onAutomaticUiTracesStateChanged();
        onScreenLoadingStateChanged();
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
    void startUiTrace(@NonNull final String name, @Nullable final Looper callerThreadLooper) {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (BuildFieldsProvider.INSTANCE.provideBuildVersion() < android.os.Build.VERSION_CODES.JELLY_BEAN) {
                    return;
                }
                if (name == null || name.trim().isEmpty()) {
                    apmLogger.logSDKError(UI_TRACE_NAME_NULL_OR_EMPTY);
                    return;
                }
                if (callerThreadLooper != Looper.getMainLooper()) {
                    apmLogger.e(UI_TRACE_NOT_STARTED_BACKGROUND_THREAD.replace("$name", name));
                    return;
                }
                APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
                if (!apmConfigurationProvider.isAPMEnabled()) {
                    apmLogger.e(UI_TRACE_NOT_STARTED_APM_NOT_ENABLED.replace("$s", name));
                } else if (!apmConfigurationProvider.isUiHangsFeatureEnabled()) {
                    apmLogger.e(UI_TRACE_NOT_STARTED_FEATURE_NOT_AVAILABLE.replace("$s", name));
                } else {
                    String trimmedName = name.trim();
                    if (trimmedName.length() > DefaultValues.UiTraces.NAME_LENGTH) {
                        trimmedName = trimmedName.substring(0, DefaultValues.UiTraces.NAME_LENGTH);
                        apmLogger.logSDKWarning(UI_TRACE_NAME_INVALID_LENGTH.replace("$s", name));
                    }
                    Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                    if (activity != null) {
                        CustomUiTraceHandler uiTraceHandler = ServiceLocator.getCustomUiTraceHandler();
                        if (uiTraceHandler != null) {
                            uiTraceHandler.startUiTrace(trimmedName, activity, callerThreadLooper);
                        }
                    }
                }
            }
        });
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
    void endUiTrace(final Looper callerThreadLooper) {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (BuildFieldsProvider.INSTANCE.provideBuildVersion() < android.os.Build.VERSION_CODES.JELLY_BEAN) {
                    return;
                }
                CustomUiTraceHandler uiTraceHandler = ServiceLocator.getCustomUiTraceHandler();
                String currentUiTraceName;
                if (uiTraceHandler != null) {
                    currentUiTraceName = uiTraceHandler.getCurrentUiTrace();
                    if (currentUiTraceName != null) {
                        Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                        if (activity != null) {
                            uiTraceHandler.endUiTrace(activity, callerThreadLooper);
                            return;
                        }
                    }
                }
                apmLogger.logSDKError(UI_TRACE_END_NOT_STARTED);
            }
        });
    }

    public void addOnNetworkTraceListener(final OnNetworkTraceListener listener) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (!apmConfigurationProvider.isAPMFeatureAvailable()) {
            apmLogger.logSDKError(NETWORK_ATTRIBUTE_NOT_ADDED_APM_NOT_AVAILABLE);
        } else if (!apmConfigurationProvider.isAPMEnabled()) {
            apmLogger.logSDKError(NETWORK_ATTRIBUTE_NOT_ADDED_APM_DISABLED);
        } else if (!apmConfigurationProvider.isNetworkFeatureEnabled()) {
            apmLogger.logSDKError(NETWORK_ATTRIBUTE_NOT_ADDED_NETWORK_NOT_AVAILABLE);
        } else {
            ServiceLocator.getNetworkTraceAttributesHandler().add(listener);
        }
    }

    public void removeOnNetworkTraceListener(final OnNetworkTraceListener listener) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (!apmConfigurationProvider.isAPMFeatureAvailable()) {
            apmLogger.logSDKError(NETWORK_ATTRIBUTE_NOT_REMOVED_APM_NOT_AVAILABLE);
        } else if (!apmConfigurationProvider.isAPMEnabled()) {
            apmLogger.logSDKError(NETWORK_ATTRIBUTE_NOT_REMOVED_APM_DISABLED);
        } else if (!apmConfigurationProvider.isNetworkFeatureEnabled()) {
            apmLogger.logSDKError(NETWORK_ATTRIBUTE_NOT_REMOVED_NETWORK_NOT_AVAILABLE);
        } else {
            ServiceLocator.getNetworkTraceAttributesHandler().remove(listener);
        }
    }

    public void endAppLaunch() {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (!apmConfigurationProvider.isAPMSdkEnabled()) {
            apmLogger.logSDKError(END_APP_LAUNCH_NOT_CALLED_APM_SDK_DISABLED);
        } else if (!apmConfigurationProvider.isAPMFeatureAvailable()) {
            apmLogger.logSDKError(END_APP_LAUNCH_NOT_CALLED_APM_FEATURE_DISABLED);
        } else {
            AppLaunchLifeCycleCallbacks appLaunchLifeCycleCallbacks = ServiceLocator.getAppLaunchLifeCycleCallbacks();
            if (appLaunchLifeCycleCallbacks != null) {
                appLaunchLifeCycleCallbacks.endAppLaunch();
            }
        }
    }

    private static void onAutomaticUiTracesStateChanged() {
        UiTracesManager uiTracesManager = UiTracesServiceLocator.getManager();
        if (uiTracesManager != null) uiTracesManager.onStateChanged();
    }

    private static void onScreenLoadingStateChanged() {
        ScreenLoadingManager screenLoadingManager = ScreenLoadingServiceLocator.getManager();
        if (screenLoadingManager != null) screenLoadingManager.onStateChanged();
    }

    public void setUIHangEnabled(boolean enabled) {
        if (BuildFieldsProvider.INSTANCE.provideBuildVersion() < android.os.Build.VERSION_CODES.JELLY_BEAN) {
            apmLogger.logSDKError("Could not enable UI Hangs. Feature is supported on API level 16 and up only.");
            return;
        }
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (apmConfigurationProvider == null) {
            apmLogger.logSDKError("Could not enable UI Hangs. apm configuration provider is null");
            return;
        }
        if (!apmConfigurationProvider.isAPMEnabled() && enabled) {
            apmLogger.logSDKError(UI_HANGS_NOT_ENABLED_APM_NOT_ENABLED);
        } else if (!apmConfigurationProvider.isUiHangsFeatureEnabled() && enabled) {
            apmLogger.logSDKError(UI_HANGS_NOT_ENABLED_BE_FLAG_DISABLED);
        } else if (!apmConfigurationProvider.isUiTraceSdkEnabled() && enabled) {
            apmLogger.logSDKError(UI_HANGS_NOT_ENABLED_TRACES_DISABLED);
        } else {
            apmConfigurationProvider.setUiHangsSdkEnabled(enabled);
        }
        onAutomaticUiTracesStateChanged();
    }

    public void setScreenLoadingEnabled(boolean enabled) {
        if (BuildFieldsProvider.INSTANCE.provideBuildVersion() < android.os.Build.VERSION_CODES.JELLY_BEAN) {
            apmLogger.logSDKError("Could not enable UI loading. Feature is supported on API level 16 and up only.");
            return;
        }
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (apmConfigurationProvider == null) {
            apmLogger.logSDKError("Could not enable UI loading. apm configuration provider is null");
            return;
        }
        if (!apmConfigurationProvider.isAPMEnabled() && enabled) {
            apmLogger.logSDKError(UI_LOADING_NOT_ENABLED_APM_NOT_ENABLED);
        } else if (!apmConfigurationProvider.isUiLoadingMetricsFeatureEnabled() && enabled) {
            apmLogger.logSDKError(UI_LOADING_NOT_ENABLED_BE_FLAG_DISABLED);
        } else if (!apmConfigurationProvider.isUiTraceSdkEnabled() && enabled) {
            apmLogger.logSDKError(UI_LOADING_NOT_ENABLED_TRACES_DISABLED);
        } else {
            apmConfigurationProvider.setUiLoadingMetricsSdkEnabled(enabled);
        }
        onScreenLoadingStateChanged();
    }

    /**
     * Mark loading for an activity as finished
     *
     * @param activityClass class of the activity to be marked finished loading
     * @param timeMetric    Captured time metric of the event
     */
    public <ActivityType extends Activity> void endScreenLoading(
            Class<ActivityType> activityClass,
            EventTimeMetricCapture timeMetric
    ) {
        ScreenLoadingManager manager = ScreenLoadingServiceLocator.getManager();
        if (manager != null) manager.endScreenLoading(activityClass, timeMetric);
    }

    public void setFragmentSpansEnabled(boolean enabled) {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if ((!apmConfigurationProvider.isAPMFeatureAvailable() || !apmConfigurationProvider.isFragmentSpansFeatureEnabled()) && enabled) {
            apmLogger.logSDKError(ENABLE_FRAGMENTS_SPANS_NOT_CALLED_FEATURE_NOT_AVAILABLE);
        } else if (!apmConfigurationProvider.isAPMSdkEnabled() && enabled) {
            apmLogger.logSDKError(ENABLE_FRAGMENTS_SPANS_NOT_CALLED_APM_NOT_ENABLED);
        } else {
            apmConfigurationProvider.setFragmentSpansSDKEnabled(enabled);
            onFragmentSpansSdkStateChanged(enabled);
        }
    }

    private void onFragmentSpansSdkStateChanged(boolean enabled) {
        FragmentSpansHelper fragmentSpansHelper = ServiceLocator.getFragmentSpansHelper();
        if (enabled) {
            fragmentSpansHelper.startFragmentsLifecycleCapturing();
        } else {
            fragmentSpansHelper.onFeatureDisabled();
        }
    }

    public void registerNetworkLogsListener(@Nullable NetworkLogListener listener) {
        NetworkInterceptionRepository repository = ServiceLocator.getNetworkInterceptionRepository();
        repository.setNetworkLogListener(listener);
    }

    public void setComposeSpansEnabled(boolean enabled) {
        ComposeSpansManager manager = ComposeSpansServiceLocator.INSTANCE.getComposeSpansManager();
        if (manager != null) manager.setSDKEnabled(enabled);
    }

    public void setWebViewsTrackingEnabled(boolean enabled) {
        WebViewTraceManager manager = ServiceLocator.getWebViewTraceManager();
        if (manager != null) manager.setSDKEnabled(enabled);
    }

    public void startFlow(TimeCaptureBoundModel<String> name) {
        AppFlowManager manager = getAppFlowManager();
        if (manager != null) manager.startFlow(name);
    }

    public void endFlow(TimeCaptureBoundModel<String> name) {
        AppFlowManager manager = getAppFlowManager();
        if (manager != null) manager.endFlow(name);
    }

    public void setFlowAttribute(@NonNull TimeCaptureBoundModel<AppFlowAttribute> attribute) {
        AppFlowManager manager = getAppFlowManager();
        if (manager != null) manager.setAttribute(attribute);
    }

    @Nullable
    private static AppFlowManager getAppFlowManager() {
        return AppFlowServiceLocator.INSTANCE.getManager();
    }

    public void startUiTraceCP(String screenName, long microTimeStamp, long traceId) {
        executeOnCpUiExecutor(() -> {
            ScreenLoadingManager screenLoadingManager = ScreenLoadingServiceLocator.getManager();
            if (screenLoadingManager != null)
                screenLoadingManager.cacheCurrentScreenLoadingCP();
            UiTracesManager uiTracesManager = UiTracesServiceLocator.getManager();
            if (uiTracesManager != null)
                uiTracesManager.onScreenChanged(screenName, microTimeStamp, traceId);
        });
    }

    public void reportScreenLoadingCP(
            long startTimeStampMicro,
            long durationMicro,
            long uiTraceId
    ) {
        executeOnCpUiExecutor(() -> {
            ScreenLoadingManager manager = ScreenLoadingServiceLocator.getManager();
            if (manager != null)
                manager.reportScreenLoadingCP(startTimeStampMicro, durationMicro, uiTraceId);
        });
    }

    public void endScreenLoadingCP(long timeStampMicro, long uiTraceId) {
        executeOnCpUiExecutor(() -> {
            ScreenLoadingManager manager = ScreenLoadingServiceLocator.getManager();
            if (manager != null) manager.endScreenLoadingCP(timeStampMicro, uiTraceId);
        });
    }

    public void isFeatureEnabledCP(@APMFeature String feature, String apiName, FeatureAvailabilityCallback callback) {
        APMCPConfigurationProvider.INSTANCE.getFeatureAvailability(feature, apiName, callback);
    }

    public boolean isFeatureEnabledCP(@APMFeature String feature, String apiName) {
        return APMCPConfigurationProvider.INSTANCE.getFeatureAvailability(feature, apiName);
    }

    private void executeOnCpUiExecutor(Runnable runnable) {
        PoolProvider.postOrderedIOTask(Constants.CP_UI_ORDERED_EXECUTOR_KEY, runnable);
    }
}
