package com.instabug.library.apm_okhttp_event_listener

import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.library.di.APMOkHttpLoggerServiceLocator
import com.instabug.library.util.InstabugSDKLogger
import okhttp3.Call
import okhttp3.Connection
import okhttp3.EventListener
import okhttp3.Handshake
import okhttp3.Protocol
import okhttp3.Request
import okhttp3.Response
import java.io.IOException
import java.net.InetAddress
import java.net.InetSocketAddress
import java.net.Proxy

class InstabugApmOkHttpEventListener(private val listener: EventListener? = null) :
    EventListener() {
    companion object {
        private const val TAG = "InstabugAPMOkhttpEventListener"
    }

    private val captor: NetworkLatencyEventCaptor =
        APMOkHttpLoggerServiceLocator.newNetworkLatencyEventCaptor

    override fun callStart(call: Call) {
        runAndLogFailure("callStart") { listener?.callStart(call) }
        captor.callStart(call)
    }

    override fun dnsStart(call: Call, domainName: String) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("dnsStart") { listener?.dnsStart(call, domainName) }
        captor.dnsStart(call, eventTimeMetric)
    }

    override fun dnsEnd(call: Call, domainName: String, inetAddressList: List<InetAddress>) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("dnsEnd") { listener?.dnsEnd(call, domainName, inetAddressList) }
        captor.dnsEnd(call, eventTimeMetric)
    }

    override fun connectStart(call: Call, inetSocketAddress: InetSocketAddress, proxy: Proxy) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("connectStart") { listener?.connectStart(call, inetSocketAddress, proxy) }
        captor.connectStart(call, eventTimeMetric)
    }

    override fun secureConnectStart(call: Call) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("secureConnectStart") { listener?.secureConnectStart(call) }
        captor.secureConnectStart(call, eventTimeMetric)
    }

    override fun secureConnectEnd(call: Call, handshake: Handshake?) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("secureConnectEnd") { listener?.secureConnectEnd(call, handshake) }
        captor.secureConnectEnd(call, eventTimeMetric)
    }

    override fun connectEnd(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?
    ) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("connectEnd") {
            listener?.connectEnd(
                call,
                inetSocketAddress,
                proxy,
                protocol
            )
        }
        captor.connectEnd(call, eventTimeMetric)
    }

    override fun connectFailed(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?,
        ioe: IOException
    ) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("connectFailed") {
            listener?.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
        }
        captor.onStageFailed(call, eventTimeMetric)
    }

    override fun connectionAcquired(call: Call, connection: Connection) {
        runAndLogFailure("connectionAcquired") { listener?.connectionAcquired(call, connection) }
    }

    override fun connectionReleased(call: Call, connection: Connection) {
        runAndLogFailure("connectionReleased") { listener?.connectionReleased(call, connection) }
    }

    override fun requestHeadersStart(call: Call) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("requestHeadersStart") { listener?.requestHeadersStart(call) }
        captor.requestHeadersStart(call, eventTimeMetric)
    }

    override fun requestHeadersEnd(call: Call, request: Request) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("requestHeadersEnd") { listener?.requestHeadersEnd(call, request) }
        captor.requestHeadersEnd(call, eventTimeMetric)
    }

    override fun requestBodyStart(call: Call) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("requestBodyStart") { listener?.requestBodyStart(call) }
        captor.requestBodyStart(call, eventTimeMetric)
    }

    override fun requestBodyEnd(call: Call, byteCount: Long) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("requestBodyEnd") { listener?.requestBodyEnd(call, byteCount) }
        captor.requestBodyEnd(call, eventTimeMetric)
    }

    override fun requestFailed(call: Call, ioe: IOException) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("requestFailed") { listener?.requestFailed(call, ioe) }
        captor.onStageFailed(call, eventTimeMetric)
    }

    override fun responseHeadersStart(call: Call) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("responseHeadersStart") { listener?.responseHeadersStart(call) }
        captor.responseHeadersStart(call, eventTimeMetric)
    }

    override fun responseHeadersEnd(call: Call, response: Response) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("responseHeadersEnd") { listener?.responseHeadersEnd(call, response) }
        captor.responseHeadersEnd(call, eventTimeMetric)
    }

    override fun responseBodyStart(call: Call) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("responseBodyStart") { listener?.responseBodyStart(call) }
        captor.responseBodyStart(call, eventTimeMetric)
    }

    override fun responseBodyEnd(call: Call, byteCount: Long) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("responseBodyEnd") { listener?.responseBodyEnd(call, byteCount) }
        captor.responseBodyEnd(call, eventTimeMetric)
    }

    override fun responseFailed(call: Call, ioe: IOException) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("responseFailed") { listener?.responseFailed(call, ioe) }
        captor.onStageFailed(call, eventTimeMetric)
    }

    override fun callEnd(call: Call) {
        runAndLogFailure("callEnd") { listener?.callEnd(call) }
        captor.callEnd(call)
    }

    override fun callFailed(call: Call, ioe: IOException) {
        val eventTimeMetric = EventTimeMetricCapture()
        runAndLogFailure("callFailed") { listener?.callFailed(call, ioe) }
        captor.callFailed(call, eventTimeMetric)
    }

    private inline fun runAndLogFailure(callbackName: String, callBack: () -> Unit) = try {
        callBack()
    } catch (t: Throwable) {
        InstabugSDKLogger.e(
            TAG, "Error occurred at okHttp.EventListener.$callbackName() : ${t.message}", t
        )
        throw t
    }

    class Factory(private val factory: EventListener.Factory? = null) : EventListener.Factory {
        override fun create(call: Call): EventListener {
            val originalEventListener = factory?.create(call)
            return if (originalEventListener is InstabugApmOkHttpEventListener)
                originalEventListener
            else
                InstabugApmOkHttpEventListener(originalEventListener)
        }
    }
}