package com.instabug.library.apmokhttplogger;

import static com.instabug.apm.constants.ErrorMessages.ERROR_WHILE_INJECTING_EXTERNAL_TRACE_HEADER;
import static com.instabug.apm.constants.ErrorMessages.NETWORK_REQUEST_STARTED;
import static com.instabug.library.constants.ErrorMessageKt.NETWORK_LOG_NOT_CAPTURED_OKHTTP_VERSION_NOT_SUPPORTED;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.logger.APMLogger;
import com.instabug.apm.networking.ApmNetworkInterceptorHelper;
import com.instabug.library.apm_network_log_repository.APMNetworkLogRepository;
import com.instabug.library.apmokhttplogger.model.OkHttpAPMNetworkLog;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.di.APMOkHttpLoggerServiceLocator;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.networkv2.BodyBufferHelper;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ObjectMapper;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import kotlin.Pair;
import okhttp3.Call;
import okhttp3.Headers;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okio.Buffer;

/**
 * This class allows you to log requests automatically created based on okHttp.
 */

public class InstabugAPMOkhttpInterceptor implements Interceptor {

    private static final String TAG = "InstabugAPMOkhttpInterceptor";
    private static final String CONTENT_TYPE = "content-type";
    private static final String CONTENT_LENGTH = "content-length";
    public static final String APOLLO_GRAPH_QL_QUERY_NAME_KEY = "x-apollo-operation-name";
    public static final String IBG_GRAPH_QL_HEADER = "ibg-graphql-header";
    public static final String APOLLO_GRAPH_QL_ERROR_KEY = "errors";
    public static final String GRAPH_QL_SERVER_SIDE_ERROR = "GraphQLError";

    @NonNull
    @Override
    public Response intercept(@NonNull Chain chain) throws IOException {
        if (canCaptureRequests(chain)) {
            return populateNetworkResponse(chain);
        } else {
            return continueChainWithoutCapturing(chain);
        }
    }

    private boolean canCaptureRequests(Chain chain) {
        return isOkHttpVersionSupported(chain) &&
                ApmNetworkInterceptorHelper.isNativeInterceptionEnabled();
    }

    @NonNull
    private static Response continueChainWithoutCapturing(@NonNull Chain chain) throws IOException {
        return chain.proceed(chain.request());
    }

    private Map<String, String> getResponseHeaders(@NonNull Response response) {
        Map<String, String> responseHeadersMap = new HashMap<>();
        Headers responseHeaders = response.headers();
        for (int i = 0; i < responseHeaders.size(); i++) {
            responseHeadersMap.put(responseHeaders.name(i).toLowerCase(), responseHeaders.value(i));
        }
        return responseHeadersMap;
    }

    private Map<String, String> getRequestHeaders(@NonNull Request request,
                                                  @Nullable RequestBody requestBody) throws IOException {
        Map<String, String> requestHeadersMap = new HashMap<>();
        if (requestBody != null) {
            MediaType contentType = requestBody.contentType();
            if (contentType != null) {
                requestHeadersMap.put(CONTENT_TYPE, contentType.toString());
            }
            if (requestBody.contentLength() != -1L) {
                requestHeadersMap.put(CONTENT_LENGTH, String.valueOf(requestBody.contentLength()));
            }
        }

        Headers requestHeaders = request.headers();
        for (int i = 0; i < requestHeaders.size(); ++i) {
            String name = requestHeaders.name(i);
            requestHeadersMap.put(name.toLowerCase(), requestHeaders.value(i));
        }
        return requestHeadersMap;
    }

    @Nullable
    private static String bodyToString(final RequestBody request) {
        try {
            MediaType contentType = request.contentType();
            String stringContentType = contentType != null ? contentType.toString() : "";
            if (BodyBufferHelper.isMultipartType(stringContentType)) {
                return BodyBufferHelper.MULTIPART_ALERT;
            }
            if (!BodyBufferHelper.isBodySizeAllowed(request.contentLength())) {
                return BodyBufferHelper.MAX_SIZE_ALERT;
            }
            final Buffer buffer = new Buffer();
            request.writeTo(buffer);
            return buffer.readUtf8();
        } catch (final IOException e) {
            InstabugSDKLogger.e(TAG, "Failed to read request body", e);
            return null;
        }
    }

    private Response populateNetworkResponse(@NonNull Chain chain) throws IOException {
        OkHttpAPMNetworkLog networkLog = null;
        Request request = chain.request();
        Map<String, String> requestHeaders = null;
        boolean shouldProcessGraphQlRequest = false;
        boolean isGraphQlRequest = false;
        Response response;
        try {
            networkLog = getNetworkLogRepository().start(chain.call());
            InstabugSDKLogger.v(TAG, "populate network request started");
            RequestBody requestBody = request.body();
            networkLog.setMethod(request.method());
            networkLog.setUrl(request.url().toString());
            requestHeaders = getRequestHeaders(request, requestBody);
            isGraphQlRequest = isGraphQLRequest(requestHeaders);
            shouldProcessGraphQlRequest = shouldProcessGraphQLRequest(isGraphQlRequest);
            if (shouldProcessGraphQlRequest) {
                networkLog.setGraphQlQueryName(getQueryName(requestHeaders));
            }
            if (isGraphQlRequest) {
                request = removeIBGGraphQlHeaderIfPresent(requestHeaders, request);
            }
            networkLog.setRequestHeaders(ObjectMapper.toJson(requestHeaders).toString());
            networkLog.setRequestContentType(requestHeaders.get(CONTENT_TYPE));
            APMLogger.d(NETWORK_REQUEST_STARTED
                    .replace("$method", request.method())
                    .replace("$url", request.url().toString()));
        } catch (Throwable throwable) {
            reportException(throwable);
        }
        if (networkLog != null) {
            networkLog.setStartTimeNanos(System.nanoTime());
            networkLog.setStartTime(System.currentTimeMillis() * 1000);
        }
        try {
            request = injectExternalTraceIdIfPossible(request, chain.call());
            response = chain.proceed(request);
            handleRequestBody(networkLog, request);
        } catch (IOException ex) {
            handleRequestBody(networkLog, request);
            handleExceptionResponse(networkLog, ex, chain.call());
            throw ex;
        }
        handleResponse(
                networkLog,
                shouldProcessGraphQlRequest,
                isGraphQlRequest,
                response,
                chain.call()
        );
        return response;
    }

    Request removeIBGGraphQlHeaderIfPresent(
            @Nullable Map<String, String> headers,
            @NonNull Request originalRequest
    ) {
        if (headers == null || !headers.containsKey(IBG_GRAPH_QL_HEADER)) {
            return originalRequest;
        }
        headers.remove(IBG_GRAPH_QL_HEADER);
        return originalRequest.newBuilder().removeHeader(IBG_GRAPH_QL_HEADER).build();
    }

    private Request injectExternalTraceIdIfPossible(Request originalRequest, Call call) {
        try {
            List<Pair<String, String>> externalTraceIdHeaders =
                    getNetworkLogRepository().getInjectableHeader(call, originalRequest.header("traceparent"));
            if (externalTraceIdHeaders == null) return originalRequest;
            Request.Builder builder = originalRequest.newBuilder();
            for (Pair<String, String> header : externalTraceIdHeaders) {
                builder.addHeader(header.getFirst(), header.getSecond());
            }
            return builder.build();
        } catch (Throwable t) {
            IBGDiagnostics.reportNonFatal(t, "InstabugAPMOkhttpInterceptor " + ERROR_WHILE_INJECTING_EXTERNAL_TRACE_HEADER);
            return originalRequest;
        }
    }

    private void handleRequestBody(
            @Nullable OkHttpAPMNetworkLog networkLog,
            Request request
    ) {
        try {
            RequestBody requestBody = request.body();
            if (networkLog != null && requestBody != null) {
                networkLog.setRequestBodySize(requestBody.contentLength());
                networkLog.setRequestBody(bodyToString(requestBody));
            }
        } catch (Exception exception) {
            InstabugSDKLogger.e(TAG, "Failed to handle Request body", exception);
        }
    }

    @NonNull
    private static APMNetworkLogRepository getNetworkLogRepository() {
        return APMOkHttpLoggerServiceLocator.getNetworkLogRepository();
    }

    private void handleResponse(
            OkHttpAPMNetworkLog networkLog,
            boolean shouldProcessGraphQlRequest,
            boolean isGraphQlRequest,
            Response response,
            Call call
    ) {
        if (networkLog != null && response != null) {
            try {
                updateNetworkLog(networkLog, shouldProcessGraphQlRequest, isGraphQlRequest, response);
                InstabugSDKLogger.v(TAG, "inserting network log");
                getNetworkLogRepository().end(call, null);
            } catch (Throwable throwable) {
                reportException(throwable);
            }
        }
    }

    private void updateNetworkLog(
            @NonNull OkHttpAPMNetworkLog networkLog,
            boolean shouldProcessGraphQlRequest,
            boolean isGraphQlRequest,
            @NonNull Response response
    ) {
        networkLog.setEndTimeNanos(System.nanoTime());
        networkLog.setResponseCode(response.code());
        if (response.code() > 0) {
            networkLog.setErrorMessage(null);
        }
        Map<String, String> responseHeaders = getResponseHeaders(response);
        networkLog.setResponseHeaders(ObjectMapper.toJson(responseHeaders).toString());
        networkLog.setResponseContentType(responseHeaders.get(CONTENT_TYPE));

        String contentLength = responseHeaders.get(CONTENT_LENGTH);
        if (contentLength != null) {
            networkLog.setResponseBodySize(Long.parseLong(contentLength));
        }
        handleGraphQlResponseBody(networkLog, shouldProcessGraphQlRequest, isGraphQlRequest, response);
    }

    private void handleGraphQlResponseBody(
            @NonNull OkHttpAPMNetworkLog networkLog,
            boolean shouldProcessGraphQlRequest,
            boolean isGraphQlRequest,
            @NonNull Response response) {
        try {
            if (isGraphQlRequest && response.body() != null) {
                InstabugAPMOkHttpBuffer buffer = new InstabugAPMOkHttpBuffer(response.body());
                networkLog.setResponseBodySize(buffer.getSize());
                String body = buffer.getBody();
                networkLog.setResponseBody(body);
                if (shouldProcessGraphQlRequest) {
                    String errorMessage = getGraphQlServerSideErrorMessage(body);
                    if (errorMessage != null) {
                        networkLog.setServerSideErrorMessage(errorMessage);
                    }
                }
            }
        } catch (OutOfMemoryError e) {
            InstabugCore.reportError(e, "Not enough memory for saving response");
            InstabugSDKLogger.e(TAG, "Not enough memory for saving response", e);
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Failed to get response body", e);
        }
    }

    private void handleExceptionResponse(
            OkHttpAPMNetworkLog networkLog,
            IOException requestException,
            Call call
    ) {
        if (networkLog != null) {
            try {
                networkLog.setEndTimeNanos(System.nanoTime());
                networkLog.setErrorMessage(requestException.getClass().getSimpleName());
                networkLog.setResponseCode(0);
                getNetworkLogRepository().end(call, requestException);
                InstabugSDKLogger.e(TAG, "Failed to proceed request", requestException);
            } catch (Throwable throwable) {
                reportException(throwable);
            }
        }
    }

    private boolean shouldProcessGraphQLRequest(boolean isGraphQlRequest) {
        return isGraphQlRequest && isGraphQlRequestsInterceptionEnabled();
    }

    private boolean isGraphQLRequest(@Nullable Map<String, String> requestHeaders) {
        if (requestHeaders != null) {
            return requestHeaders.containsKey(APOLLO_GRAPH_QL_QUERY_NAME_KEY) ||
                    requestHeaders.containsKey(IBG_GRAPH_QL_HEADER);
        }
        return false;
    }

    private boolean isGraphQlRequestsInterceptionEnabled() {
        return ApmNetworkInterceptorHelper.isGraphQlEnabled();
    }

    @Nullable
    private String getQueryName(@Nullable Map<String, String> requestHeaders) {
        if (requestHeaders != null) {
            String apolloGqlQueryName = requestHeaders.get(APOLLO_GRAPH_QL_QUERY_NAME_KEY);
            return apolloGqlQueryName != null ? apolloGqlQueryName : requestHeaders.get(IBG_GRAPH_QL_HEADER);
        }
        return null;
    }

    @Nullable
    private String getGraphQlServerSideErrorMessage(@Nullable String body) {
        if (body != null) {
            try {
                JSONObject jsonObject = new JSONObject(body);
                JSONArray errorsJsonArray = jsonObject.optJSONArray(APOLLO_GRAPH_QL_ERROR_KEY);
                if (errorsJsonArray != null && errorsJsonArray.length() != 0) {
                    return GRAPH_QL_SERVER_SIDE_ERROR;
                }
            } catch (JSONException e) {
                // do nothing as the response is not a jsonObject
            }
        }
        return null;
    }

    private void reportException(Throwable throwable) {
        try {
            IBGDiagnostics.reportNonFatal(
                    throwable,
                    "Exception while trying to intercept an OkHttp request"
            );
            InstabugSDKLogger.e(
                    TAG,
                    "Exception while trying to intercept an OkHttp request",
                    throwable
            );
        } catch (Exception exc) {
            exc.printStackTrace();
        }
    }

    private boolean isOkHttpVersionSupported(Chain chain) {
        try {
            chain.call();
            return true;
        } catch (Throwable throwable) {
            APMLogger.e(NETWORK_LOG_NOT_CAPTURED_OKHTTP_VERSION_NOT_SUPPORTED);
            return false;
        }
    }
}
