/*
 * (C) Copyright IBM Corp. 2022, 2023.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.ibm.watson.discovery.v2.model;

import com.google.gson.annotations.SerializedName;
import com.ibm.cloud.sdk.core.service.model.GenericModel;
import java.util.Date;

/** Information about a document classifier model. */
public class DocumentClassifierModel extends GenericModel {

  /** The status of the training run. */
  public interface Status {
    /** training. */
    String TRAINING = "training";
    /** available. */
    String AVAILABLE = "available";
    /** failed. */
    String FAILED = "failed";
  }

  @SerializedName("model_id")
  protected String modelId;

  protected String name;
  protected String description;
  protected Date created;
  protected Date updated;

  @SerializedName("training_data_file")
  protected String trainingDataFile;

  @SerializedName("test_data_file")
  protected String testDataFile;

  protected String status;
  protected ClassifierModelEvaluation evaluation;

  @SerializedName("enrichment_id")
  protected String enrichmentId;

  @SerializedName("deployed_at")
  protected Date deployedAt;

  protected DocumentClassifierModel() {}

  /**
   * Gets the modelId.
   *
   * <p>A unique identifier of the document classifier model.
   *
   * @return the modelId
   */
  public String getModelId() {
    return modelId;
  }

  /**
   * Gets the name.
   *
   * <p>A human-readable name of the document classifier model.
   *
   * @return the name
   */
  public String getName() {
    return name;
  }

  /**
   * Gets the description.
   *
   * <p>A description of the document classifier model.
   *
   * @return the description
   */
  public String getDescription() {
    return description;
  }

  /**
   * Gets the created.
   *
   * <p>The date that the document classifier model was created.
   *
   * @return the created
   */
  public Date getCreated() {
    return created;
  }

  /**
   * Gets the updated.
   *
   * <p>The date that the document classifier model was last updated.
   *
   * @return the updated
   */
  public Date getUpdated() {
    return updated;
  }

  /**
   * Gets the trainingDataFile.
   *
   * <p>Name of the CSV file that contains the training data that is used to train the document
   * classifier model.
   *
   * @return the trainingDataFile
   */
  public String getTrainingDataFile() {
    return trainingDataFile;
  }

  /**
   * Gets the testDataFile.
   *
   * <p>Name of the CSV file that contains data that is used to test the document classifier model.
   * If no test data is provided, a subset of the training data is used for testing purposes.
   *
   * @return the testDataFile
   */
  public String getTestDataFile() {
    return testDataFile;
  }

  /**
   * Gets the status.
   *
   * <p>The status of the training run.
   *
   * @return the status
   */
  public String getStatus() {
    return status;
  }

  /**
   * Gets the evaluation.
   *
   * <p>An object that contains information about a trained document classifier model.
   *
   * @return the evaluation
   */
  public ClassifierModelEvaluation getEvaluation() {
    return evaluation;
  }

  /**
   * Gets the enrichmentId.
   *
   * <p>A unique identifier of the enrichment that is generated by this document classifier model.
   *
   * @return the enrichmentId
   */
  public String getEnrichmentId() {
    return enrichmentId;
  }

  /**
   * Gets the deployedAt.
   *
   * <p>The date that the document classifier model was deployed.
   *
   * @return the deployedAt
   */
  public Date getDeployedAt() {
    return deployedAt;
  }
}
