/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.text.TextUtils;
import android.util.Pair;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAPushConfigs;
import com.hyphenate.chat.adapter.EMAPushManager;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by wei on 2016/12/7.
 */

/**
 * \~chinese
 * 离线消息推送管理类，针对GCM、小米、华为等离线推送以及APNS
 *
 * \~english
 * Offline push message manager class that used for GCM and APNS
 */
public class EMPushManager {
    private static final String TAG = EMPushManager.class.getSimpleName();

    /**
     * \~chinese
     * 推送消息展示样式：SimpleBanner表示展示简单消息，MessageSummary表示展示消息内容
     *
     * \~english
     * push message presentation style: SimpleBanner represents the presentation of a simple message,
     * and MessageSummary represents the presentation of message content
     */
    public enum DisplayStyle {
        SimpleBanner, MessageSummary
    }

    EMClient mClient;
    EMAPushManager emaObject;
    EMPushManager(EMClient client, EMAPushManager pushManager) {
        emaObject = pushManager;
        mClient = client;
    }


    /**
     * \~chinese
     * 开启离线消息推送
     * @throws HyphenateException
     *
     * \~english
     * enable offline message push
     * @throws HyphenateException
     */
    public void enableOfflinePush() throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.enableOfflineNotification(error);
        handleError(error);
    }

    /**
     * \~chinese
     * 在指定的时间段(24小时制)内，不推送离线消息
     * @param start 开始时间
     * @param end 结束时间
     * @throws HyphenateException
     *
     * \~english
     * The offline message is not pushed within the specified time period (24-hour clock)
     * @param start start hour
     * @param end end hour
     * @throws HyphenateException
     */
    public void disableOfflinePush(int start, int end) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.disableOfflineNotification(start, end, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 从缓存获取推送配置信息
     * @return
     *
     * \~english
     * get push configs from cache
     * @return
     */
    public EMPushConfigs getPushConfigs(){
        EMAPushConfigs pushConfigs = emaObject.getPushConfigs();
        if(pushConfigs == null){
            return null;
        }
        return new EMPushConfigs(pushConfigs);
    }

    /**
     * \~chinese
     * 从服务器获取推送配置信息
     * @return
     * @throws HyphenateException
     *
     * \~english
     * get push configs from server
     * @return
     * @throws HyphenateException
     */
    public EMPushConfigs getPushConfigsFromServer() throws HyphenateException{
        EMAError error = new EMAError();
        EMAPushConfigs pushConfigs = emaObject.getPushConfigsFromServer(error);
        handleError(error);

        return new EMPushConfigs(pushConfigs);
    }


    /**
     * \~chinese
     * 设置指定的群组是否接受离线消息推送
     * @param groupIds 要设置的群组列表
     * @param noPush true：不接收离线消息推送， false：接收推送
     * @throws HyphenateException
     *
     * \~english
     * Sets whether the specified group accepts offline message push
     * @param groupIds the list of groups to be set
     * @param noPush true：not accept offline push， false：accept offline push
     * @throws HyphenateException
     */
    public void updatePushServiceForGroup(List<String> groupIds, boolean noPush) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForGroup(groupIds, noPush, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 获取关闭了离线消息推送的群组
     * @return
     *
     * \~english
     * get the group list that disabled offline push
     * @return
     */
    public List<String> getNoPushGroups(){
        return emaObject.getNoPushGroups();
    }

    /**
	 * \~chinese
	 * 更新当前用户的nickname,这样离线消息推送的时候可以显示用户昵称而不是id，
     * 当用户更改昵称时，务必也调用此方法更新到环信服务器，防止显示差异
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param nickname 昵称
	 *
	 * \~english
	 * This method can be used to set a nickname, the nickname will be used to show for APNS and GCM.
     * When the user changes the nickname, be sure to also call this method to update to the hyphenate
     *                 server to prevent the display of differences
	 *
     * Synchronization method will block the current thread
     *
	 * @param nickname
	 */
    public boolean updatePushNickname(String nickname) throws IllegalArgumentException, HyphenateException {
        if (TextUtils.isEmpty(nickname)) {
            throw new IllegalArgumentException("nick name is null or empty");
        }
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushNickname(nickname, error);
        handleError(error);
        return true;
    }

    /**
	 * \~chinese
	 * 更新当前用户的nickname,这样离线消息推送的时候可以显示用户昵称而不是id，
     * 当用户更改昵称时，务必也调用此方法更新到环信服务器，防止显示差异
	 *
	 * @param nickname 昵称
	 *
	 * \~english
	 * This method can be used to set a nickname, the nickname will be used to show for APNS and GCM.
     * When the user changes the nickname, be sure to also call this method to update to the hyphenate
     *                 server to prevent the display of differences
	 *
	 * @param nickname
	 */
    public void asyncUpdatePushNickname(String nickname, EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushNickname(nickname);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
	 * \~chinese
	 * 更新推送消息样式，默认是{@link DisplayStyle#SimpleBanner}
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param style 推送消息样式
	 *
	 * \~english
	 * update push message style, default is {@link DisplayStyle#SimpleBanner}
     *
     * Synchronization method will block the current thread
	 *
	 * @param style push message style
	 */
    public void updatePushDisplayStyle(DisplayStyle style) throws IllegalArgumentException, HyphenateException
    {
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushDisplayStyle(style.ordinal(), error);
        handleError(error);
    }

    /**
	 * \~chinese
	 * 更新推送消息样式，默认是{@link DisplayStyle#SimpleBanner}
	 *
	 * @param style 推送消息样式
	 *
	 * \~english
	 * update push message style, default is {@link DisplayStyle#SimpleBanner}
	 *
	 * @param style push message style
	 */
    public void asyncUpdatePushDisplayStyle(DisplayStyle style, EMCallBack callback)
    {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushDisplayStyle(style);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

}
