/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMMucSharedFile;

import java.util.List;

/**
 *  \~chinese
 *  群组相关的监听器，侦听群组相关的事件，例如加群申请，希望加入某个群，这个群的群主同意或者拒绝，被踢群等事件。 
 *
 *  \~english
 *  Group change listener, listen group related events. Such as receive join group invite, and join group application, etc...
 */
public interface EMGroupChangeListener {
	/**
     * \~chinese
	 * 当前用户收到加入群组邀请
	 * @param groupId 	要加入的群的id
	 * @param groupName 要加入的群的名称
	 * @param inviter 	邀请人的id
	 * @param reason 	邀请加入的reason
     *
     * \~english
	 * when receiving a group invitation
	 *
	 * @param groupId		group id
	 * @param groupName		group's subject
	 * @param inviter		Who invite you join the group
	 * @param reason		Literal message coming with the invitation
	 */
	void onInvitationReceived(String groupId, String groupName, String inviter, String reason);
	
	/**
     * \~chinese
	 * 用户申请加入群
	 * @param groupId 	要加入的群的id
	 * @param groupName 要加入的群的名称
	 * @param applicant 申请人的username
	 * @param reason 	申请加入的reason
     *
     * \~english
     * when the group owner receives a group request from user
	 *
	 * @param groupId		group id
	 * @param groupName		group's name
	 * @param applicant		The applicant want to join the group
	 * @param reason		Literal message coming with the application
	 */
	void onRequestToJoinReceived(String groupId, String groupName, String applicant, String reason);

	/**
     * \~chinese
	 * 加群申请被对方接受
	 * @param groupId	群组的id
	 * @param groupName 群组的名字
	 * @param accepter 	同意人得username
     *
     * \~english
	 * when the group invitation is accepted
	 *
	 * @param groupId 		group id
	 * @param groupName 	group's name
	 * @param accepter 		who approve the application
	 */
	void onRequestToJoinAccepted(String groupId, String groupName, String accepter);

	/**
     * \~chinese
	 * 加群申请被拒绝
	 * @param groupId 	群组id
	 * @param groupName 群组名字
	 * @param decliner 	拒绝人得username
	 * @param reason 	拒绝理由
     *
     * \~english
     * when the group invitation is declined
	 *
	 * @param groupId 		group id
	 * @param groupName 	group name
	 * @param decliner 		decliner's username
	 * @param reason 		reason of declining
	 */
	void onRequestToJoinDeclined(String groupId, String groupName, String decliner, String reason);

	/**
     * \~chinese
	 * 群组邀请被接受
	 * @param groupId
	 * @param invitee
	 * @param reason
     *
     * \~english
     * when the group invitation is accepted
	 *
	 * @param groupId 		group id
	 * @param invitee 
	 * @param reason
	 */
	void onInvitationAccepted(String groupId, String invitee, String reason);
    
	/**
     * \~chinese
	 * 群组邀请被拒绝
	 * @param groupId
	 * @param invitee
	 * @param reason
     *
     * \~english
     * when the group invitation is declined
	 *
	 * @param groupId 		group id
	 * @param invitee
	 * @param reason 		reason of declining
	 */
	void onInvitationDeclined(String groupId, String invitee, String reason);
    
	/**
     * \~chinese
	 * 当前登录用户被管理员移除出群组
	 * @param groupId 
	 * @param groupName
     *
     * \~english
     * current user has been removed from the group
	 *
	 * @param groupId 
	 * @param groupName
	 */
	void onUserRemoved(String groupId, String groupName);
    
    /**
     * \~chinese
     * 群组被解散。
     * sdk 会先删除本地的这个群组，之后通过此回调通知应用，此群组被删除了
     * @param groupId 	群组的ID
     * @param groupName 群组的名称
     *
     * \~english
     * group dissolution
     * SDK will delete the group from local DB and local cache, then notify user that the group is destroyed
	 *
	 * @param groupId		group id
	 * @param groupName 	group name
     */
	void onGroupDestroyed(String groupId, String groupName);
    
    /**
     * \~chinese
     * 自动同意加入群组
     * sdk会先加入这个群组，并通过此回调通知应用
     * 参考{@link com.hyphenate.chat.EMOptions#setAutoAcceptGroupInvitation(boolean value)}
     * 
     * @param groupId
     * @param inviter
     * @param inviteMessage
     *
     * \~english
     * When received group join invitation, will auto accept it and join the group
     * Please refer to {@link com.hyphenate.chat.EMOptions#setAutoAcceptGroupInvitation(boolean value)}
	 *
	 * @param groupId			group id
	 * @param inviter
	 * @param inviteMessage
     */
	void onAutoAcceptInvitationFromGroup(String groupId, String inviter, String inviteMessage);

	/**
	 * \~chinese
	 * 有成员被禁言，此处不同于blacklist
	 *
	 * @param groupId 	产生禁言的群组id
	 * @param mutes 	被禁言的成员列表
	 *              	Map.entry.key 是禁言的成员id，Map.entry.value是禁言动作存在的时间。
	 *
	 * \~english
	 * Callback when group member(s) is muted (added to mute list), and is not allowed to post message temporarily based on muted time duration
	 *
	 * @param groupId		group id
	 * @param mutes 		member(s) added to mute list
	 *	 			        Map.entry.key is muted username. Map.entry.value is the duration of muted time, in milliseconds
	 */
	void onMuteListAdded(String groupId, final List<String> mutes, final long muteExpire);

	/**
	 * \~chinese
	 * 有成员从禁言列表中移除，恢复发言权限，此处不同于blacklist
	 *
	 * @param groupId 	产生禁言的群组id
	 * @param mutes 	有成员从群组禁言列表中移除
	 *
	 * \~english
	 * Callback when group member(s) is unmuted (removed from mute list), and allow to post message now
	 *
	 * @param groupId		group id
	 * @param mutes 		members that be removed from mute list
	 */
	void onMuteListRemoved(String groupId, final List<String> mutes);

	/**
	 * \~chinese
	 * 白名单成员增加
	 *
	 * @param groupId    群组id
	 * @param whitelist     白名单中增加的成员名单
	 *
	 * \~english
	 * Callback when chat room member(s) is added to whitelist
	 *
	 * @param groupId    group id
	 * @param whitelist     member(s) was added to whitelist
	 */
	void onWhiteListAdded(final String groupId, final List<String> whitelist);

	/**
	 * \~chinese
	 * 白名单成员减少
	 *
	 * @param groupId    群组id
	 * @param whitelist     白名单中移除的成员名单
	 *
	 * \~english
	 * Callback when chat room member(s) is removed from whitelist
	 *
	 * @param groupId    group id
	 * @param whitelist     member(s) was removed from whitelist
	 */
	void onWhiteListRemoved(final String groupId, final List<String> whitelist);

	/**
	 * \~chinese
	 * 全员禁言状态的改变
	 *
	 * @param groupId    群组id
	 * @param isMuted       是否开启了全员禁言
	 *
	 * \~english
	 * Callback when chat room all memeber muted or not
	 *
	 * @param groupId    group id
	 * @param isMuted       if the all room memmber is muted or not
	 */
	void onAllMemberMuteStateChanged(final String groupId, final boolean isMuted);

	/**
	 * \~chinese
	 * 添加成员管理员权限
	 *
	 * @param groupId 		添加管理员权限对应的群组
	 * @param administrator 被添加为管理员的成员
	 * @return NA
	 *
	 * \~english
	 * Callback when a member has been changed to admin
	 *
	 * @param groupId		group id
	 * @param administrator member who has been changed to admin
	 */
	void onAdminAdded(String groupId, String administrator);

	/**
	 * \~chinese
	 * 取消某管理员权限
	 * @param groupId 		取消管理员权限事件发生的群id
	 * @param administrator 被取消管理员权限的成员
	 *
	 * \~english
	 * Callback when member is removed from admin
	 *
	 * @param groupId 		group id
	 * @param administrator the member whose admin permission is removed
	 */
	void onAdminRemoved(String groupId, String administrator);

	/**
	 * \~chinese
	 * 转移群组所有者权限
	 * @param groupId 	转移群组所有者权限的群id
	 * @param newOwner 	新的群组所有者
	 * @param oldOwner 	原群组所有者
	 *
	 * \~english
	 * Callback when chat room ownership has been transferred
	 *
	 * @param groupId 		group id
	 * @param newOwner 		new owner
	 * @param oldOwner 		previous owner
	 */
	void onOwnerChanged(String groupId, String newOwner, String oldOwner);
	
	/**
     * \~chinese
     * 群组加入新成员事件
     * 
     * @param groupId
     *          群组id
     * @param member
     *          新成员id
     *
     * \~english
     * Callback when a member join the group.
     * 
     * @param groupId        group id
     * @param member   new member's id
     */
    void onMemberJoined(final String groupId, final String member);

    /**
     * \~chinese
     * 群组成员主动退出事件
     * 
     * @param groupId
     *          群组id
     * @param member
     *          退出的成员的id
     * 
     * \~english
     * Callback when a member exited the group
     * 
     * @param groupId       group id
     * @param member  the member who exited the group
      */
    void onMemberExited(final String groupId,  final String member);

	/**
	 * \~chinese
	 * 群公告更改事件
	 * @param groupId  群组id
	 * @param announcement 更新的公告内容
	 *
	 * \~english
	 * callback when the announcement changed
	 * @param groupId group id
	 * @param announcement changed announcement
	 */
	void onAnnouncementChanged(String groupId, String announcement);


	/**
	 * \~chinese
	 * 群组增加共享文件事件
	 * @param groupId 群组id
	 * @param sharedFile 增加的文件
	 *
	 * \~english
	 * callback when a shared file added
	 * @param groupId group id
	 * @param sharedFile added file
	 */
	void onSharedFileAdded(String groupId, EMMucSharedFile sharedFile);

	/**
	 * \~chinese
	 * 群组删除共享文件事件
	 * @param groupId 群组id
	 * @param fileId 删除文件的id
	 *
	 * \~english
	 * callback when a shared file deleted
	 * @param groupId group id
	 * @param fileId deleted file id
	 */
	void onSharedFileDeleted(String groupId, String fileId);
}
