package com.hyphenate.util;

import android.content.ContentUris;
import android.content.Context;
import android.database.Cursor;
import android.media.MediaMetadataRetriever;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.SystemClock;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.support.v4.provider.DocumentFile;
import android.text.TextUtils;

import java.io.File;
import java.io.IOException;

public class UriUtils {
    private static final String TAG = UriUtils.class.getSimpleName();

    /**
     * 判断文件是否存在
     * @param context
     * @param fileUri
     * @return
     */
    public static boolean isFileExistByUri(Context context, Uri fileUri) {
        if(fileUri == null) {
            return false;
        }
        //target 小于Q
        if(!VersionUtils.isTargetQ(context)) {
            String filePath = getFilePath(context, fileUri);
            if(!TextUtils.isEmpty(filePath)) {
                return new File(filePath).exists();
            }
            return false;
        }
        //target 大于等于Q
        if(uriStartWithFile(fileUri)) {
            String path = fileUri.getPath();
            boolean exists = new File(path).exists();
            long length = new File(path).length();
            EMLog.d(TAG, "file uri exist = "+exists + " file length = "+ length);
            return exists;
        }else if(!uriStartWithContent(fileUri)) {
            return fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists();
        }else {
            DocumentFile documentFile = DocumentFile.fromSingleUri(context, fileUri);
            return documentFile != null && documentFile.exists();
        }
    }

    /**
     * 判断uri是否以file开头
     * @param fileUri
     * @return
     */
    public static boolean uriStartWithFile(Uri fileUri) {
        return TextUtils.equals(fileUri.getScheme(), "file") && fileUri.toString().length() > 7;
    }

    /**
     * 判断uri是否以content开头
     * @param fileUri
     * @return
     */
    public static boolean uriStartWithContent(Uri fileUri) {
        return TextUtils.equals(fileUri.getScheme(), "content");
    }

    /**
     * 获取文件名
     * @param context
     * @param fileUri
     * @return
     */
    public static String getFileNameByUri(Context context, Uri fileUri) {
        if(fileUri == null) {
            return "";
        }
        //target 小于Q
        if(!VersionUtils.isTargetQ(context)) {
            String filePath = getFilePath(context, fileUri);
            if(!TextUtils.isEmpty(filePath) && new File(filePath).exists()) {
                return new File(filePath).getName();
            }
            return "";
        }
        //target 大于Q
        if(uriStartWithFile(fileUri)) {
            File file = new File(fileUri.getPath());
            return file.exists() ? file.getName() : "";
        }
        if(!uriStartWithContent(fileUri)) {
            if(fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists()) {
                return new File(fileUri.toString()).getName();
            }
            return "";
        }

        DocumentFile documentFile = getDocumentFile(context, fileUri);
        if(documentFile == null) {
            return "";
        }
        return documentFile.getName();
    }

    public static long getFileLength(Context context, Uri fileUri) {
        if(fileUri == null) {
            return 0;
        }
        //target 小于Q
        if(!VersionUtils.isTargetQ(context)) {
            String filePath = getFilePath(context, fileUri);
            if(!TextUtils.isEmpty(filePath) && new File(filePath).exists()) {
                return new File(filePath).length();
            }
            return 0;
        }
        //target 大于Q
        if(uriStartWithFile(fileUri) && new File(fileUri.getPath()).exists()) {
            return new File(fileUri.getPath()).length();
        }
        if(!uriStartWithContent(fileUri) && fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists()) {
            return new File(fileUri.toString()).length();
        }
        DocumentFile documentFile = getDocumentFile(context, fileUri);
        if(documentFile == null) {
            return 0;
        }
        return documentFile.length();
    }

    public static String getFileMimeType(Context context, Uri fileUri) {
        if(fileUri == null) {
            return null;
        }
        //target 小于Q
        if(!VersionUtils.isTargetQ(context)) {
            String filePath = getFilePath(context, fileUri);
            if(!TextUtils.isEmpty(filePath)) {
                return FileUtils.getMIMEType(new File(filePath));
            }
            return null;
        }
        //target 大于Q
        if(uriStartWithFile(fileUri)) {
            return FileUtils.getMIMEType(new File(fileUri.getPath()));
        }
        if(!uriStartWithContent(fileUri)) {
            if(fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists()) {
                return FileUtils.getMIMEType(new File(fileUri.toString()));
            }else {
                return null;
            }
        }
        DocumentFile documentFile = getDocumentFile(context, fileUri);
        if(documentFile == null) {
            return null;
        }
        return documentFile.getType();
    }

    public static int getVideoOrAudioDuration(Context context, Uri mediaUri) {
        String[] projection = {MediaStore.Video.Media.DURATION};
        Cursor cursor = context.getContentResolver().query(mediaUri, projection, null,
                null, null);
        long duration = 0;
        if (cursor != null && cursor.moveToFirst()) {
            duration = cursor.getLong(cursor.getColumnIndex(MediaStore.Video.Media.DURATION));
            cursor.close();
            cursor=null;
        }
        if(duration <= 0) {
            MediaMetadataRetriever retriever = null;
            String durationString = "";
            try {
                retriever = new MediaMetadataRetriever();
                retriever.setDataSource(context, mediaUri);
                durationString = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
                duration = Integer.valueOf(durationString);
            } catch (IllegalArgumentException e) {
                e.printStackTrace();
            } catch (SecurityException e) {
                e.printStackTrace();
            } finally {
                if(retriever != null) {
                    retriever.release();
                }
            }
        }
        if(duration <= 0) {
            duration = 0;
        }
        EMLog.d(TAG, "duration:"+duration);
        return (int)duration;
    }

    private static DocumentFile getDocumentFile(Context context, Uri uri) {
        if(uri == null) {
            EMLog.e(TAG, "uri is null");
            return null;
        }
        DocumentFile documentFile = DocumentFile.fromSingleUri(context, uri);
        if(documentFile == null) {
            EMLog.e(TAG, "DocumentFile is null");
            return null;
        }
        return documentFile;
    }

    public static String getUriString(Uri uri) {
        if(uri == null) {
            return null;
        }
        return uri.toString();
    }

    public static Uri getLocalUriFromString(String url){
        if(TextUtils.isEmpty(url)) {
            return null;
        }
        if(url.startsWith("content")) {
            return Uri.parse(url);
        }else if(url.startsWith("file") && url.length() > 7) {
            return Uri.fromFile(new File(Uri.parse(url).getPath()));
        }else if(url.startsWith("/")) {
            return Uri.fromFile(new File(url));
        }
        return null;
    }

    public static String getMimeType(Context context, Uri fileUri) {
        String filePath = getFilePath(context, fileUri);
        if(!TextUtils.isEmpty(filePath) && new File(filePath).exists()) {
            return getMimeType(new File(filePath));
        }
        return context.getContentResolver().getType(fileUri);
    }

    public static String getMimeType(File sourceFile){
        String sourceName=sourceFile.getName();
        return getMimeType(sourceName);
    }

    public static String getMimeType(String fileName) {
        if (fileName.endsWith(".3gp") || fileName.endsWith(".amr")) {
            return "audio/3gp";
        }if(fileName.endsWith(".jpe")||fileName.endsWith(".jpeg")||fileName.endsWith(".jpg")){
            return "image/jpeg";
        }if(fileName.endsWith(".amr")){
            return "audio/amr";
        }if(fileName.endsWith(".mp4")){
            return "video/mp4";
        }if(fileName.endsWith(".mp3")){
            return "audio/mpeg";
        }else {
            return "application/octet-stream";
        }
    }

    public static String getFilePath(Context context, String path) {
        if(TextUtils.isEmpty(path)) {
            return path;
        }
        Uri uri = Uri.parse(path);
        return UriUtils.getFilePath(context, uri);
    }

    /**
     * Get a file path from a Uri. This will get the the path for Storage Access
     * Framework Documents, as well as the _data field for the MediaStore and
     * other file-based ContentProviders.
     *
     * @param context The context.
     * @param uri     The Uri to query.
     * @author paulburke
     */
    public static String getFilePath(final Context context, final Uri uri) {
        if(uri == null) {
            return "";
        }
        final boolean isKitKat = Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT;
        //sdk版本在29之前的
        if(!VersionUtils.isTargetQ(context)) {
            // DocumentProvider
            if (isKitKat && DocumentsContract.isDocumentUri(context, uri)) {
                // ExternalStorageProvider
                if (isExternalStorageDocument(uri)) {
                    final String docId = DocumentsContract.getDocumentId(uri);
                    final String[] split = docId.split(":");
                    final String type = split[0];

                    if ("primary".equalsIgnoreCase(type)) {
                        return Environment.getExternalStorageDirectory() + "/" + split[1];
                    }

                    // TODO handle non-primary volumes
                }
                // DownloadsProvider
                else if (isDownloadsDocument(uri)) {
                    final String id = DocumentsContract.getDocumentId(uri);
                    final Uri contentUri = ContentUris.withAppendedId(
                            Uri.parse("content://downloads/public_downloads"), Long.valueOf(id));

                    return getDataColumn(context, contentUri, null, null);
                }
                // MediaProvider
                else if (isMediaDocument(uri)) {
                    final String docId = DocumentsContract.getDocumentId(uri);
                    final String[] split = docId.split(":");
                    final String type = split[0];

                    Uri contentUri = null;
                    if ("image".equals(type)) {
                        contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                    } else if ("video".equals(type)) {
                        contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                    } else if ("audio".equals(type)) {
                        contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                    }

                    final String selection = "_id=?";
                    final String[] selectionArgs = new String[]{
                            split[1]
                    };

                    return getDataColumn(context, contentUri, selection, selectionArgs);
                }
            }
            else if(isFileProvider(context, uri)) {
                return uri.getPath();
            }
            // MediaStore (and general)
            else if ("content".equalsIgnoreCase(uri.getScheme())) {
                return getDataColumn(context, uri, null, null);
            }
            // File
            else if ("file".equalsIgnoreCase(uri.getScheme())) {
                return uri.getPath();
            }else if(uri.toString().startsWith("/")) {//如果是路径的话，返回路径
                return uri.toString();
            }
        }else {
            //29之后，判断是否是file开头及是否是以"/"开头
            if(uriStartWithFile(uri)) {
                return uri.getPath();
            }
            if(uri.toString().startsWith("/")) {
                return uri.toString();
            }
        }
        return "";
    }

    /**
     * Get the value of the data column for this Uri. This is useful for
     * MediaStore Uris, and other file-based ContentProviders.
     *
     * @param context       The context.
     * @param uri           The Uri to query.
     * @param selection     (Optional) Filter used in the query.
     * @param selectionArgs (Optional) Selection arguments used in the query.
     * @return The value of the _data column, which is typically a file path.
     */
    public static String getDataColumn(Context context, Uri uri, String selection,
                                       String[] selectionArgs) {

        Cursor cursor = null;
        final String column = "_data";
        final String[] projection = {
                column
        };

        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs,
                    null);
            if (cursor != null && cursor.moveToFirst()) {
                final int column_index = cursor.getColumnIndexOrThrow(column);
                return cursor.getString(column_index);
            }
        } finally {
            if (cursor != null)
                cursor.close();
        }
        return null;
    }


    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is ExternalStorageProvider.
     */
    public static boolean isExternalStorageDocument(Uri uri) {
        return "com.android.externalstorage.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is DownloadsProvider.
     */
    public static boolean isDownloadsDocument(Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is MediaProvider.
     */
    public static boolean isMediaDocument(Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }

    public static boolean isFileProvider(Context context, Uri uri) {
        return (context.getApplicationInfo().packageName + ".fileProvider").equals(uri.getAuthority());
    }

}
