package com.hyphenate.util;

import android.content.Context;
import android.database.Cursor;
import android.media.MediaMetadataRetriever;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.SystemClock;
import android.provider.MediaStore;
import android.support.v4.provider.DocumentFile;
import android.text.TextUtils;

import java.io.File;
import java.io.IOException;

public class UriUtils {
    private static final String TAG = UriUtils.class.getSimpleName();

    /**
     * 判断文件是否存在
     * @param context
     * @param fileUri
     * @return
     */
    public static boolean isFileExistByUri(Context context, Uri fileUri) {
        if(fileUri == null) {
            return false;
        }else if(uriStartWithFile(fileUri)) {
            String path = fileUri.getPath();
            boolean exists = new File(path).exists();
            long length = new File(path).length();
            EMLog.d(TAG, "file uri exist = "+exists + " file length = "+ length);
            return exists;
        }else if(!uriStartWithContent(fileUri)) {
            return fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists();
        }else {
            DocumentFile documentFile = DocumentFile.fromSingleUri(context, fileUri);
            return documentFile != null && documentFile.exists();
        }
    }

    /**
     * 判断uri是否以file开头
     * @param fileUri
     * @return
     */
    public static boolean uriStartWithFile(Uri fileUri) {
        return TextUtils.equals(fileUri.getScheme(), "file") && fileUri.toString().length() > 7;
    }

    /**
     * 判断uri是否以content开头
     * @param fileUri
     * @return
     */
    public static boolean uriStartWithContent(Uri fileUri) {
        return TextUtils.equals(fileUri.getScheme(), "content");
    }

    /**
     * 获取文件名
     * @param context
     * @param fileUri
     * @return
     */
    public static String getFileNameByUri(Context context, Uri fileUri) {
        if(fileUri == null) {
            return "";
        }
        if(uriStartWithFile(fileUri)) {
            File file = new File(fileUri.getPath());
            return file.exists() ? file.getName() : "";
        }
        if(!uriStartWithContent(fileUri)) {
            if(fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists()) {
                return new File(fileUri.toString()).getName();
            }
            return "";
        }

        DocumentFile documentFile = getDocumentFile(context, fileUri);
        if(documentFile == null) {
            return "";
        }
        return documentFile.getName();
    }

    public static long getFileLength(Context context, Uri fileUri) {
        if(fileUri == null) {
            return 0;
        }
        if(uriStartWithFile(fileUri) && new File(fileUri.getPath()).exists()) {
            return new File(fileUri.getPath()).length();
        }
        if(!uriStartWithContent(fileUri) && fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists()) {
            return new File(fileUri.toString()).length();
        }
        DocumentFile documentFile = getDocumentFile(context, fileUri);
        if(documentFile == null) {
            return 0;
        }
        return documentFile.length();
    }

    public static String getFileMimeType(Context context, Uri fileUri) {
        if(fileUri == null) {
            return null;
        }
        if(uriStartWithFile(fileUri)) {
            return FileUtils.getMIMEType(new File(fileUri.getPath()));
        }
        if(!uriStartWithContent(fileUri)) {
            if(fileUri.toString().startsWith("/") && new File(fileUri.toString()).exists()) {
                return FileUtils.getMIMEType(new File(fileUri.toString()));
            }else {
                return null;
            }
        }
        DocumentFile documentFile = getDocumentFile(context, fileUri);
        if(documentFile == null) {
            return null;
        }
        return documentFile.getType();
    }

    public static int getVideoOrAudioDuration(Context context, Uri mediaUri) {
        String[] projection = {MediaStore.Video.Media.DURATION};
        Cursor cursor = context.getContentResolver().query(mediaUri, projection, null,
                null, null);
        long duration = 0;
        if (cursor != null && cursor.moveToFirst()) {
            duration = cursor.getLong(cursor.getColumnIndex(MediaStore.Video.Media.DURATION));
            cursor.close();
            cursor=null;
        }
        if(duration <= 0) {
            MediaMetadataRetriever retriever = null;
            String durationString = "";
            try {
                retriever = new MediaMetadataRetriever();
                retriever.setDataSource(context, mediaUri);
                durationString = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
                duration = Integer.valueOf(durationString);
            } catch (IllegalArgumentException e) {
                e.printStackTrace();
            } catch (SecurityException e) {
                e.printStackTrace();
            } finally {
                if(retriever != null) {
                    retriever.release();
                }
            }
        }
        if(duration <= 0) {
            duration = 0;
        }
        EMLog.d(TAG, "duration:"+duration);
        return (int)duration;
    }

    private static DocumentFile getDocumentFile(Context context, Uri uri) {
        if(uri == null) {
            EMLog.e(TAG, "uri is null");
            return null;
        }
        DocumentFile documentFile = DocumentFile.fromSingleUri(context, uri);
        if(documentFile == null) {
            EMLog.e(TAG, "DocumentFile is null");
            return null;
        }
        return documentFile;
    }

    public static String getUriString(Uri uri) {
        if(uri == null) {
            return null;
        }
        return uri.toString();
    }

    public static Uri getLocalUriFromString(String url){
        if(TextUtils.isEmpty(url)) {
            return null;
        }
        if(url.startsWith("content")) {
            return Uri.parse(url);
        }else if(url.startsWith("file") && url.length() > 7) {
            return Uri.fromFile(new File(Uri.parse(url).getPath()));
        }else if(url.startsWith("/")) {
            return Uri.fromFile(new File(url));
        }
        return null;
    }

    public static String getMimeType(Context context, Uri fileUri) {
        String filePath = getFilePath(fileUri);
        if(!TextUtils.isEmpty(filePath) && new File(filePath).exists()) {
            return getMimeType(new File(filePath));
        }
        return context.getContentResolver().getType(fileUri);
    }

    public static String getMimeType(File sourceFile){
        String sourceName=sourceFile.getName();
        return getMimeType(sourceName);
    }

    public static String getMimeType(String fileName) {
        if (fileName.endsWith(".3gp") || fileName.endsWith(".amr")) {
            return "audio/3gp";
        }if(fileName.endsWith(".jpe")||fileName.endsWith(".jpeg")||fileName.endsWith(".jpg")){
            return "image/jpeg";
        }if(fileName.endsWith(".amr")){
            return "audio/amr";
        }if(fileName.endsWith(".mp4")){
            return "video/mp4";
        }if(fileName.endsWith(".mp3")){
            return "audio/mpeg";
        }else {
            return "application/octet-stream";
        }
    }

    public static String getFilePath(String path) {
        if(TextUtils.isEmpty(path)) {
            return path;
        }
        if(path.startsWith("file") && path.length() > 7) {
            return Uri.parse(path).getPath();
        }
        if(path.startsWith("/")) {
            return path;
        }
        return "";
    }

    public static String getFilePath(Uri path) {
        if(path == null) {
            return "";
        }
        if(uriStartWithFile(path)) {
            return path.getPath();
        }
        if(path.toString().startsWith("/")) {
            return path.toString();
        }
        return "";
    }

}
