/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.content.Intent;
import android.hardware.Camera;
import android.hardware.Camera.CameraInfo;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.os.SystemClock;
import android.util.Pair;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.chat.EMCallStateChangeListener.CallError;
import com.hyphenate.chat.EMCallStateChangeListener.CallState;
import com.hyphenate.chat.adapter.EMACallManager;
import com.hyphenate.chat.adapter.EMACallManagerListener;
import com.hyphenate.chat.adapter.EMACallRtcImpl;
import com.hyphenate.chat.adapter.EMACallRtcListenerDelegate;
import com.hyphenate.chat.adapter.EMACallSession;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.EMNoActiveCallException;
import com.hyphenate.exceptions.EMServiceNotReadyException;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.media.EMCallSurfaceView;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.superrtc.mediamanager.EMediaManager;
import com.superrtc.mediamanager.XClientBridger;
import com.superrtc.sdk.RtcConnection;
import com.superrtc.sdk.RtcConnection.RtcStatistics;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;


public class EMCallManager {
    /**
     * Be careful about synchronized usage,
     * there are lots of callback, may lead to concurrent issue
     */
    static final String TAG = "EMCallManager";
    EMACallManager emaObject;
    EMClient mClient;
    List<EMCallStateChangeListener> callListeners = Collections.synchronizedList(new ArrayList<EMCallStateChangeListener>());
    EMACallListenerDelegate delegate = new EMACallListenerDelegate();
    EMCallSession currentSession;
    EMVideoCallHelper callHelper = new EMVideoCallHelper();
    EMCameraDataProcessor processor;
    CallStateUnion callState = new CallStateUnion();
    EMCallOptions callOptions;
    boolean isConnectedFromRinging = false;

    // zhangsong
    private int cameraFacing = -1;

    static {
        EMediaManager.setLoggerDelegate(new XClientBridger.Logcallbackfunc() {
            @Override public void onLog(int i, String s) {
                EMLog.d(TAG + "$RTC", s);
            }
        });
    }

    /**
     * Need a more sophiscated to record more state info
     * @author linan
     *
     */
    static class CallStateUnion {
        /**
         * 0-3 bit:
         *      CALLState.ordinal
         * 4 bit: voice pause | resume
         *      0 resume
         *      1 pause
         * 5 bit: video pause | resume
         *      0 resume
         *      1 pause
         * 6-7: network state
         *     00 NETWORK_NORMAL
         *     01 NETWORK_UNSTABLE
         *     02 NETWORK_DISCONNECTED
         */
        short BIT0 = 0x01;
        short BIT1 = 0x02;
        short BIT2 = 0x04;
        short BIT3 = 0x08;
        short BIT4 = 0x10;
        short BIT5 = 0x20;
        short BIT6 = 0x40;
        short BIT7 = 0x80;

        BitSet callState = new BitSet();

        /**
         * patch: for incoming call, state change flow -> ring -> connected
         * for this case, connected should also be viewed as ringing, for reject action
         */
        boolean ringingToConnected = false;

        void reset() {
            callState.clear();
            ringingToConnected = false;
        }
        void changeState(CallState state) {
            switch (state) {
                case IDLE:
                case RINGING:
                case ANSWERING:
                case CONNECTING:
                case CONNECTED:
                case ACCEPTED:
                case DISCONNECTED:
                    if (isRinging() && state == CallState.CONNECTED) {
                        ringingToConnected = true;
                    }
                    callState.clear(0, 3);
                    int val = state.ordinal();
                    if ((val & BIT0) > 0) {
                        callState.set(0);
                    } else {
                        callState.clear(0);
                    }
                    if ((val & BIT1) > 0) {
                        callState.set(1);
                    } else {
                        callState.clear(1);
                    }
                    if ((val & BIT2) > 0) {
                        callState.set(2);
                    } else {
                        callState.clear(2);
                    }
                    if ((val & BIT3) > 0) {
                        callState.set(3);
                    } else {
                        callState.clear(3);
                    }
                    if (state == CallState.DISCONNECTED) {
                        reset();
                    }
                    break;
                case VOICE_PAUSE:
                    callState.set(4);
                    break;
                case VOICE_RESUME:
                    callState.clear(4);
                    break;
                case VIDEO_PAUSE:
                    callState.set(5);
                    break;
                case VIDEO_RESUME:
                    callState.clear(5);
                    break;
                case NETWORK_NORMAL:
                    callState.clear(6, 7);
                    break;
                case NETWORK_UNSTABLE:
                    callState.set(6);
                    callState.clear(7);
                    break;
                case NETWORK_DISCONNECTED:
                    callState.clear(6);
                    callState.set(7);
                    break;
                default:
                    break;
            }
        }

        CallState getState() {
            int val = 0;
            if (callState.get(0)) {
                val |= BIT0;
            }
            if (callState.get(1)) {
                val |= BIT1;
            }
            if (callState.get(2)) {
                val |= BIT2;
            }
            if (callState.get(3)) {
                val |= BIT3;
            }
            return CallState.values()[val];
        }

        static boolean isMainState(CallState state) {
            return state.ordinal() <= CallState.DISCONNECTED.ordinal();
        }

        boolean isIdle() {
            return (callState.get(0) == false &&
                    callState.get(1) == false &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isRinging_() {
            return (callState.get(0) == true &&
                    callState.get(1) == false &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isRinging() {
            return isRinging_() || (isConnected() && ringingToConnected);
        }

        boolean isAnswering() {
            return (callState.get(0) == false &&
                    callState.get(1) == true &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isConnecting() {
            return (callState.get(0) == true &&
                    callState.get(1) == true &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isConnected() {
            return (callState.get(0) == false &&
                    callState.get(1) == false &&
                    callState.get(2) == true &&
                    callState.get(3) == false);
        }

        boolean isAccepted() {
            return (callState.get(0) == true &&
                    callState.get(1) == false &&
                    callState.get(2) == true &&
                    callState.get(3) == false);
        }

        boolean isDisconnected() {
            return (callState.get(0) == false &&
                    callState.get(1) == true &&
                    callState.get(2) == true &&
                    callState.get(3) == false);
        }

        boolean isVoicePause() {
            return callState.get(4);
        }

        boolean isVideoPause() {
            return callState.get(5);
        }
    }

    RtcConnection mRtcConnection;
    RtcConnection.Listener mRtcListener;

    boolean isVideoCall = true;

    //===================== public interface =====================
    /**
     * Interface for user pre-process camera data
     *
     */
    public interface EMCameraDataProcessor {
        void onProcessData(byte[] data, final Camera camera, final int width, final int height, final int rotateAngle);
    }

    public interface EMCallPushProvider {
        /**
         * Function is called when remote peer is offline, we want to let remote peer known we are calling.
         * For IOS that's go through APNS, for Android that's will go through GCM, MiPush, HuaweiPush.
         * Condition: Only works when EMOptions's configuration setPushCall to be true.
         */
        void onRemoteOffline(final String to);
    }

    EMCallPushProvider mPushProvider;

    final EMCallPushProvider defaultProvider = new EMCallManager.EMCallPushProvider() {

        void updateMessageText(final EMMessage oldMsg, final String to) {
            // update local message text
            EMConversation conv = EMClient.getInstance().chatManager().getConversation(oldMsg.getTo());
            if (conv != null) {
                conv.removeMessage(oldMsg.getMsgId());
            }
        }

        @Override
        public void onRemoteOffline(final String to) {

            //this function should exposed & move to Demo
            EMLog.d(TAG, "onRemoteOffline, to:" + to);

            final EMMessage message = EMMessage.createTxtSendMessage("You have an incoming call", to);
            // set the user-defined extension field
            JSONObject jTitle = new JSONObject();
            try {
                jTitle.put("em_push_title", "You have an incoming call");
            } catch (Exception e) {
                e.printStackTrace();
            }
            message.setAttribute("em_apns_ext", jTitle);

            message.setAttribute("is_voice_call", !isVideoCall);

            message.setMessageStatusCallback(new EMCallBack(){

                @Override
                public void onSuccess() {
                    EMLog.d(TAG, "onRemoteOffline success");
                    updateMessageText(message, to);
                }

                @Override
                public void onError(int code, String error) {
                    EMLog.d(TAG, "onRemoteOffline Error");
                    updateMessageText(message, to);
                }

                @Override
                public void onProgress(int progress, String status) {
                }
            });
            // send messages
            EMClient.getInstance().chatManager().sendMessage(message);
        }
    };

    public static final String IncomingCallAction = ".action.incomingcall";
    public String getIncomingCallBroadcastAction() {
        return EMClient.getInstance().getContext().getPackageName() + IncomingCallAction;
    }

    protected EMCallManager(EMClient client, EMACallManager manager) {
        mClient = client;
        emaObject = manager;
        emaObject.addListener(delegate);
        try {
            if (EMClient.getInstance().getOptions().isUseStereoInput()) {
                EMediaManager.setUseStereoInput(true);
            }
            EMediaManager.initGlobal(EMClient.getInstance().getContext());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Not implemented.
     * This method will be removed in future versions.
     *
     * @param processor
     */
    @Deprecated
    public void setCameraDataProcessor(EMCameraDataProcessor processor) {
    }

    public void setPushProvider(EMCallPushProvider provider) {
        this.mPushProvider = provider;
    }

    public CallState getCallState() {
        return callState.getState();
    }

    public EMVideoCallHelper getVideoCallHelper() {
        return callHelper;
    }

    /**
     * \~chinese
     * 发起呼叫，进行视频通话请求，进行视频呼叫前，请先在Activity.onCreate中先执行setSurfaceView。
     * @param   username        被呼叫的用户id
     * @throws                  EMServiceNotReadyException 呼叫过程中遇到异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate firstly.
     * @param   username        callee's user id
     * @throws                  EMServiceNotReadyException
     */
    public void makeVideoCall(String username) throws EMServiceNotReadyException {
        makeVideoCall(username, "", false, false);
    }

    /**
     * \~chinese
     * 发起呼叫，进行视频通话请求，进行视频呼叫前，请先在Activity.onCreate中先执行setSurfaceView。
     *
     * @param username                      被呼叫的用户id
     * @param ext                           扩展字段，用户可以用来传输自定义的内容
     * @throws EMServiceNotReadyException   呼叫过程中遇到异常
     *
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate first
     *
     * @param   username      callee's user id.
     * @param   ext           extension string, user can transfer self defined content
     * @throws  EMServiceNotReadyException
     */
    public void makeVideoCall(String username, String ext) throws EMServiceNotReadyException {
        makeVideoCall(username, ext, false, false);
    }

    /**
     * \~chinese
     * 发起呼叫，进行视频通话请求，进行视频呼叫前，请先在Activity.onCreate中先执行setSurfaceView。
     *
     * @param username                      被呼叫的用户id
     * @param ext                           扩展字段，用户可以用来传输自定义的内容
     * @param recordOnServer                是否在服务器端录制该通话
     * @param mergeStream                   服务器端录制时是否合并流
     * @throws EMServiceNotReadyException   呼叫过程中遇到异常
     *
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate first
     *
     * @param   username      callee's user id.
     * @param   ext           extension string, user can transfer self defined content
     * @param recordOnServer                if record on server
     * @param mergeStream                   if merge stream when record on server
     * @throws  EMServiceNotReadyException
     */
    public void makeVideoCall(String username, String ext, boolean recordOnServer, boolean mergeStream) throws EMServiceNotReadyException
    {
        EMLog.d(TAG, "makeVideoCall");
        isVideoCall = true;
        if (!EMClient.getInstance().isConnected()) {
            EMLog.d(TAG, "exception isConnected:" + false);
            throw new EMServiceNotReadyException("exception isConnected:" + false);
        }
        if (!NetUtils.hasDataConnection(EMClient.getInstance().getContext())) {
            EMLog.d(TAG, "Has no network connection");
            throw new EMServiceNotReadyException(EMError.NETWORK_ERROR, "Has no network connection");
        }
        if (!callState.isIdle() && !callState.isDisconnected()) {
            EMLog.d(TAG, "exception callState:" + callState);
            throw new EMServiceNotReadyException("exception callState:" + callState.getState().toString());
        }

        EMAError error = new EMAError();
        EMACallSession _session = emaObject.makeCall(username, EMACallSession.Type.VIDEO, ext, error, recordOnServer, mergeStream);
        synchronized (this) {
            if (error.errCode() != EMAError.EM_NO_ERROR) {
                EMLog.d(TAG, "errorCode:" + error.errCode());
                currentSession = null;
                changeState(CallState.DISCONNECTED, getCallError(error));
                throw new EMServiceNotReadyException(error.errCode(), error.errMsg());
            }
            currentSession = new EMCallSession(_session);
            changeState(CallState.CONNECTING);
        }
    }

    /**
     * \~chinese
     * 发起呼叫，进行语音通话请求
     *
     * @param username                      被呼叫的用户id
     * @throws EMServiceNotReadyException   呼叫过程中遇到异常 如果IM没有连接，或者之前通话的连接没有断开，会抛出次异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate first
     *
     * @param username                      callee's user id.
     * @throws EMServiceNotReadyException   if IM is not connected, or previous call doesn't disconnected, will throw EMServiceNotReadyException
     */
    public void makeVoiceCall(String username) throws EMServiceNotReadyException {
        makeVoiceCall(username, "");
    }

    /**
     * \~chinese
     * 发起呼叫，进行语音通话请求
     *
     * @param username                      被呼叫的用户id
     * @param ext                           扩展字段，用户可以用来传输自定义的内容
     * @throws EMServiceNotReadyException   呼叫过程中遇到异常 如果IM没有连接，或者之前通话的连接没有断开，会抛出次异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate first
     *
     * @param username                      callee's user id.
     * @param ext                           extension string, user can transfer self defined content
     * @throws EMServiceNotReadyException   if IM is not connected, or previous call doesn't disconnected, will throw EMServiceNotReadyException
     */
    public void makeVoiceCall(String username, String ext) throws EMServiceNotReadyException
    {
        EMLog.d(TAG, "makeVoiceCall");

        makeVoiceCall(username, ext, false, false);
    }

    /**
     * \~chinese
     * 发起呼叫，进行语音通话请求
     *
     * @param username                      被呼叫的用户id
     * @param ext                           扩展字段，用户可以用来传输自定义的内容
     * @param recordOnServer                是否在服务器端录制该通话
     * @param mergeStream                   服务器端录制时是否合并流
     * @throws EMServiceNotReadyException   呼叫过程中遇到异常 如果IM没有连接，或者之前通话的连接没有断开，会抛出次异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate first
     *
     * @param username                      callee's user id.
     * @param ext                           extension string, user can transfer self defined content
     * @param recordOnServer                if record on server
     * @param mergeStream                   if merge the streams if record on server
     * @throws EMServiceNotReadyException   if IM is not connected, or previous call doesn't disconnected, will throw EMServiceNotReadyException
     */
    public void makeVoiceCall(String username, String ext, boolean recordOnServer, boolean mergeStream) throws EMServiceNotReadyException
    {
        EMLog.d(TAG, "makeVoiceCall");
        isVideoCall = false;
        if (!EMClient.getInstance().isConnected()) {
            EMLog.d(TAG, "exception isConnected:" + false);
            throw new EMServiceNotReadyException("exception isConnected:" + false);
        }
        if (!NetUtils.hasDataConnection(EMClient.getInstance().getContext())) {
            EMLog.d(TAG, "Has no network connection");
            throw new EMServiceNotReadyException(EMError.NETWORK_ERROR, "Has no network connection");
        }
        if (callState.isIdle() == false && callState.isDisconnected() == false) {
            EMLog.d(TAG, "exception callState:" + callState);
            throw new EMServiceNotReadyException("exception callState:" + callState.getState().toString());
        }
        EMAError error = new EMAError();
        EMACallSession _session =  emaObject.makeCall(username, EMACallSession.Type.VOICE, ext, error, recordOnServer, mergeStream);
        synchronized (this) {
            if (error.errCode() != EMAError.EM_NO_ERROR) {
                EMLog.d(TAG, "errorCode:" + error.errCode());
                currentSession = null;

                changeState(CallState.DISCONNECTED, getCallError(error));
                throw new EMServiceNotReadyException(error.errCode(), error.errMsg());
            }
            currentSession = new EMCallSession(_session);
            changeState(CallState.CONNECTING);
        }
    }

    private CallError getCallError(EMAError error) {
        CallError callError = CallError.ERROR_TRANSPORT;
        switch (error.errCode()) {
            case EMError.CALL_INVALID_ID:
                callError = CallError.ERROR_NONE;
                break;
            case EMError.CALL_BUSY:
                callError = CallError.ERROR_BUSY;
                break;
            case EMError.CALL_REMOTE_OFFLINE:
                callError = CallError.ERROR_UNAVAILABLE;
                break;
            case EMError.CALL_CONNECTION_ERROR:
                callError = CallError.ERROR_TRANSPORT;
                break;
            case EMError.SERVICE_ARREARAGES:
                callError = CallError.ERROR_SERVICE_ARREARAGES;
                break;
            case EMError.SERVICE_NOT_ENABLED:
                callError = CallError.ERROR_SERVICE_NOT_ENABLE;
                break;
            case EMError.SERVER_SERVICE_RESTRICTED:
                callError = CallError.ERROR_SERVICE_FORBIDDEN;
                break;
            default:
                break;
        }
        return callError;
    }

    /**
     * \~chinese
     * 设置通话状态监听
     *
     * @param listener
     *
     * \~english
     * register call state change listener
     *
     * @param listener
     */
    public void addCallStateChangeListener(EMCallStateChangeListener listener) {
        synchronized (callListeners) {
            if (!callListeners.contains(listener)) {
                callListeners.add(listener);
            }
        }
    }

    /**
     * \~chinese
     * 移除通话监听
     *
     * @param listener
     *
     * \~english
     * remove call state change listener
     *
     * @param listener
     */
    public void removeCallStateChangeListener(EMCallStateChangeListener listener) {
        synchronized (callListeners) {
            if (callListeners.contains(listener)) {
                callListeners.remove(listener);
            }
        }
    }

    RtcConnection createRtcConnection(String name) {
        RtcConnection rtc = new RtcConnection(name);
        EMCallOptions options = EMClient.getInstance().callManager().getCallOptions();
        if (options.isUserSetAutoResizing) {
            rtc.enableFixedVideoResolution(options.userSetAutoResizing);
        }
        if (options.isUserSetMaxFrameRate) {
            rtc.setMaxVideoFrameRate(options.userSetMaxFrameRate);
        }
        if (options.isEnableExternalVideoData) {
            rtc.setEnableExternalVideoData(options.isEnableExternalVideoData);
        }
        if (options.rotateAngel >= 0) {
            rtc.setRotation(options.rotateAngel);
        }

        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put(RtcConnection.RtcKVMaxAudioKbpsLong, options.maxAudioBitrate);
            jsonObject.put(RtcConnection.RtcKVMaxVideoKbpsLong, options.maxVideoKbps);
            long resolutionWidth = options.getVideoResolutionWidth();
            long resolutionHeight = options.getVideoResolutionHeight();
            if (resolutionWidth != -1 && resolutionHeight != -1) {
                jsonObject.put(RtcConnection.RtcvideowidthLong, resolutionWidth);
                jsonObject.put(RtcConnection.RtcvideoheigthLong, resolutionHeight);
            }
            rtc.setConfigure(jsonObject.toString());
            if (cameraFacing != -1) {
                rtc.setCameraFacing(cameraFacing);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return rtc;
    }


    /**
     * \~chinese
     * 这里一定要在{Activity#onCreate()}方法中调用，防止无法准确获取到 surface 大小
     *
     * @param localSurface 默认本地预览画面控件
     * @param oppositeSurface 默认远端画面控件
     *
     * \~english
     * Must be called when Activity.onCreate, otherwise cannot get surface size accurately
     *
     * @param localSurface default local SurfaceView
     * @param oppositeSurface default remote SurfaceView
     */
    public synchronized void setSurfaceView(EMCallSurfaceView localSurface, EMCallSurfaceView oppositeSurface) {
        if (mRtcConnection == null) {
            mRtcConnection = createRtcConnection("rtc");
        }

//        VideoViewRenderer local = null;
//        VideoViewRenderer opposite = null;
//        if (localSurface != null) {
//            local = localSurface.getRenderer();
//        }
//        if (oppositeSurface != null) {
//            opposite = oppositeSurface.getRenderer();
//        }
        mRtcConnection.setViews(localSurface, oppositeSurface);
    }

    /**
     * \~chinese
     * 接听通话
     *
     * @throws EMNoActiveCallException
     *
     * \~english
     * answer phone call
     *
     * @throws EMNoActiveCallException
     */
    public void answerCall() throws EMNoActiveCallException {
        synchronized (this) {
            if (currentSession == null) {
                throw new EMNoActiveCallException("no incoming active call");
            }
        }
        if (callState.isRinging() == false) {
            throw new EMNoActiveCallException("Current callstate is not ringing callState:" + callState.getState());
        }
        EMAError error = new EMAError();
        emaObject.answerCall(currentSession.getCallId(), error);
        synchronized (this) {
            changeState(CallState.ANSWERING);
            if (error.errCode() != EMAError.EM_NO_ERROR) {
                EMLog.d(TAG, "errorCode:" + error.errCode());
                endCall();
            }
        }
    }

    /**
     * \~chinese
     * 拒绝接听
     *
     * @throws EMNoActiveCallException
     *
     * \~english
     * decline phone call
     *
     * @throws EMNoActiveCallException
     */
    public void rejectCall() throws EMNoActiveCallException {
        final EMCallSession session = currentSession;
        if (session == null) {
            EMLog.e(TAG, "no incoming active call");
            throw new EMNoActiveCallException("no incoming active call");
        }
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                emaObject.endCall(session.getCallId(), callState.isRinging() ?
                        EMACallSession.EndReason.REJECT : EMACallSession.EndReason.HANGUP);

            }
        });
        synchronized (this) {
            // onRecvCallEnded will also trigger changeState(CallState.DISCONNECT)
            //changeState(CallState.DISCONNECTED);
            mRtcConnection = null;
            mRtcListener = null;
        }
    }

    /**
     * \~chinese
     * 挂断通话
     * @throws EMNoActiveCallException
     *
     * \~english
     * hang up phone call
     *
     * @throws EMNoActiveCallException
     */
    public void endCall() throws EMNoActiveCallException {
        String _callId = "";
        synchronized (this) {
            if (currentSession == null) {
                EMLog.e(TAG, "no incoming active call");
                throw new EMNoActiveCallException("no incoming active call");
            }
            _callId = currentSession.getCallId();
        }
        final String callId = _callId;
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                emaObject.endCall(callId, EMACallSession.EndReason.HANGUP);
            }
        });
        synchronized (this) {
            // onRecvCallEnded will also trigger changeState(CallState.DISCONNECT)
            // changeState(CallState.DISCONNECTED);
            mRtcConnection = null;
            mRtcListener = null;
        }
    }

    /**
     * \~chinese
     * 返回当前通话时是否为P2P直连
     *
     * @return
     *
     * \~english
     * check if it's a P2P call with direct connection
     * @return
     */
    public boolean isDirectCall() {
        RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
        if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
            EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
            RtcStatistics statistics = lis.getStatistics();
            if (statistics != null && statistics.connectionType != null) {
                return statistics.connectionType.equals("direct");
            }
        }
        return true;
    }

    /**
     * \~chinese
     * 实时通话时暂停语音数据传输
     *
     * \~english
     * pause real time voice data transfer
     *
     */
    public void pauseVoiceTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.setMute(true);
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.PAUSE_VOICE, error);
            handleError(error);
        }
    }

    /**
     * \~chinese
     * 实时通话时恢复语音数据传输
     *
     * \~english
     * resume real time voice data transfer
     *
     */
    public void resumeVoiceTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.setMute(false);
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.RESUME_VOICE, error);
            handleError(error);
        }
    }

    /**
     * \~chinese
     * 实时通话时停止视频数据传输
     *
     * \~english
     * pause real time video data transfer
     */
    public void pauseVideoTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.stopCapture();
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.PAUSE_VIDEO, error);
            handleError(error);
        }
    }

    /**
     * \~chinese
     * 实时通话时恢复视频数据传输
     *
     * \~english
     * resume real time video data transfer
     */
    public void resumeVideoTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.startCapture();
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.RESUME_VIDEO, error);
            handleError(error);
        }
    }

    /**
     * \~chinese
     * mute远端音频
     *
     * \~english
     * Mute remote audio
     *
     * @param mute
     */
    public void muteRemoteAudio(boolean mute) {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.muteRemoteAudio(mute);
        }
    }

    /**
     * ~\chinese
     * mute远端视频
     *
     * \~english
     * Mute remote video
     *
     * @param mute
     */
    public void muteRemoteVideo(boolean mute) {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.muteRemoteVideo(mute);
        }
    }

    /**
     * \~chinese
     * 开启相机拍摄
     *
     * @param facing        参数可以是CameraInfo.CAMERA_FACING_BACK, 或者CameraInfo.CAMERA_FACING_FRONT
     * @throws HyphenateException
     *  如果cameraIndex不是CameraInfo.CAMERA_FACING_BACK，也不是CameraInfo.CAMERA_FACING_FRONT，会抛出异常.
     *
     * \~english
     * start camera capture
     *
     * @param facing        select CameraInfo.CAMERA_FACING_BACK or CameraInfo.CAMERA_FACING_FRONT
     * @throws HyphenateException
     *  if cameraIndex not in CameraInfo.CAMERA_FACING_BACK, 或者CameraInfo.CAMERA_FACING_FRONT, got the exception.
     */
    public void setCameraFacing(int facing) throws HyphenateException {
        if (facing != CameraInfo.CAMERA_FACING_BACK && facing != CameraInfo.CAMERA_FACING_FRONT) {
            throw new HyphenateException(EMError.CALL_INVALID_CAMERA_INDEX, "Invalid camera index");
        }

        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.setCameraFacing(facing);
        } else {
            cameraFacing = facing;
        }
    }

    public synchronized void switchCamera() {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.switchCamera(null);
        }
    }

    private void handleError(EMAError error)  throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            EMLog.e(TAG, "error code:" + error.errCode() + " errorMsg:" + error.errMsg());
            throw new HyphenateException(error);
        }
    }

    /**
     * \~chinese
     * 获取当前正在使用的摄像头
     *
     * @return 值可以是
     *
     * \~english
     * get the camera currently using
     *
     * @return camera int value
     */
    public int getCameraFacing() {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            return rtc.getCameraFacing();
        }
        return 0;
    }

    //=====================end of public interface=====================
    HandlerThread stateChangeHandlerThread = new HandlerThread("CallStateHandlerThread");
    { stateChangeHandlerThread.start(); }
    Handler stateChangeHandler = new Handler(stateChangeHandlerThread.getLooper()) {

        @Override
        public void handleMessage(Message msg) {
            @SuppressWarnings("unchecked")
            Pair<CallState, CallError> pair = (Pair<CallState, CallError>)msg.obj;
            CallState state = pair.first;
            CallError error = pair.second;
            EMLog.d(TAG, "stateChangeHandler handleMessage BEGIN ---- state:" + state);
            notifyCallStateChanged(state, error);
            EMLog.d(TAG, "stateChangeHandler handleMessage  END  ----");
        }
    };

    private void notifyCallStateChanged(CallState callState, CallError callError){
        synchronized (callListeners) {
            for (EMCallStateChangeListener listener : callListeners) {
                listener.onCallStateChanged(callState, callError);
            }
        }
    }

    void changeState(CallState state) {
        changeState(state, CallError.ERROR_NONE);
    }

    protected void changeState(final CallState state, final CallError callError){
        if (CallStateUnion.isMainState(state) && callState.getState().equals(state)) {
            return;
        }
        EMLog.d(TAG, "changeState:" + state);
        this.callState.changeState(state);
        stateChangeHandler.sendMessage(stateChangeHandler.obtainMessage(0, new Pair<CallState, CallError>(state, callError)));
    }

    void clearStateMessages() {
        stateChangeHandler.removeMessages(0);
    }

    class EMACallListenerDelegate extends EMACallManagerListener {

        @Override
        public void onSendPushMessage(String from, String to) {
            EMCallPushProvider pushProvider = EMCallManager.this.mPushProvider;
            if (pushProvider == null) {
                defaultProvider.onRemoteOffline(to);
            } else {
                pushProvider.onRemoteOffline(to);
            }
        }

        @Override
        public void onRecvCallFeatureUnsupported(EMACallSession session, EMAError error) {
            EMLog.d(TAG, "onRecvCallFeatureUnsupported, callId:" + session.getCallId());
        }

        @Override
        public void onRecvCallIncoming(EMACallSession session) {
            EMLog.d(TAG, "onRecvSessionRemoteInitiate");
            synchronized (EMCallManager.this) {
                currentSession = new EMCallSession(session);
                changeState(CallState.RINGING);
            }

            Intent intent = new Intent(getIncomingCallBroadcastAction());
            intent.putExtra("type", currentSession.getType() == EMCallSession.Type.VIDEO ? "video" : "voice");
            intent.putExtra("from", currentSession.getRemoteName());
            intent.putExtra("to", EMClient.getInstance().getCurrentUser());
            EMClient.getInstance().getContext().sendBroadcast(intent);
        }

        @Override
        public void onRecvCallConnected(EMACallSession session) {
            EMLog.d(TAG, "onRecvSessionConnected");
            synchronized (EMCallManager.this) {
                if (currentSession == null) {
                    currentSession = new EMCallSession(session);
                }
                changeState(CallState.CONNECTED);
            }
        }

        @Override
        public void onRecvCallAccepted(EMACallSession session) {
            EMLog.d(TAG, "onReceiveCallAccepted");
            synchronized (EMCallManager.this) {
                if (currentSession == null) {
                    currentSession = new EMCallSession(session);
                }
                EMLog.d(TAG, "onReceiveCallAccepted");
                changeState(CallState.ACCEPTED);
            }
        }


        CallError endReasonToCallError(EMCallSession.EndReason reason, EMAError emaError){
            CallError error = CallError.ERROR_NONE;
            switch (reason) {
                case HANGUP:
                    error = CallError.ERROR_NONE;
                    break;
                case NORESPONSE:
                    error = CallError.ERROR_NORESPONSE;
                    break;
                case REJECT:
                    error = CallError.REJECTED;
                    break;
                case BUSY:
                    error = CallError.ERROR_BUSY;
                    break;
                case FAIL:
                    error = CallError.ERROR_TRANSPORT;
                    if (emaError.errCode() != EMAError.EM_NO_ERROR) {
                        if (emaError.errCode() == EMAError.CALL_REMOTE_OFFLINE) {
                            error = CallError.ERROR_UNAVAILABLE;
                        } else if (emaError.errCode() == EMAError.CALL_BUSY) {
                            error = CallError.ERROR_UNAVAILABLE;
                        } else if (emaError.errCode() == EMAError.CALL_CONNECTION_FAILED) {
                            error = CallError.ERROR_TRANSPORT;
                        }
                    }
                    break;
                case OFFLINE:
                    error = CallError.ERROR_UNAVAILABLE;
                    break;
                case SERVICE_NOT_ENABLE:
                    error = CallError.ERROR_SERVICE_NOT_ENABLE;
                    break;
                case SERVICE_ARREARAGES:
                    error = CallError.ERROR_SERVICE_ARREARAGES;
                    break;
                case SERVICE_FORBIDDEN:
                    error = CallError.ERROR_SERVICE_FORBIDDEN;
                    break;
                default:
                    break;
            }
            return error;
        }

        @Override
        public void onRecvCallEnded(EMACallSession callSession, int reasonOrdinal, EMAError error) {
            EMLog.d(TAG, "onReceiveCallTerminated, reasonOrdinal: " + reasonOrdinal);
            synchronized (EMCallManager.this) {
                if (currentSession != null) {
                    currentSession = null;
                }
                changeState(CallState.DISCONNECTED, endReasonToCallError(EMCallSession.getEndReason(reasonOrdinal), error));
                mRtcConnection = null;
                mRtcListener = null;
            }
        }

        @Override
        public void onRecvCallNetworkStatusChanged(EMACallSession callSession, int toStatus) {
            EMLog.d(TAG, "onRecvCallNetworkStatusChanged, callId: " + callSession.getCallId() + " toStatus:" + toStatus);
            CallState callState = CallState.DISCONNECTED;

            if (toStatus == EMACallSession.NetworkStatus.CONNECTED.ordinal()) {
                callState = CallState.NETWORK_NORMAL;
            } else if (toStatus == EMACallSession.NetworkStatus.UNSTABLE.ordinal()) {
                callState = CallState.NETWORK_UNSTABLE;
            } else if (toStatus == EMACallSession.NetworkStatus.DISCONNECTED.ordinal()) {
                callState = CallState.NETWORK_DISCONNECTED;
            } else {
                try {
                    throw new HyphenateException("onRecvCallNetworkStatusChanged invalid toStatus:" + toStatus);
                } catch (HyphenateException e) {
                    e.printStackTrace();
                }
            }

            if (EMCallManager.this.callState.getState() == callState) {
                EMLog.d(TAG, "onRecvCallNetworkStatusChanged toStatus equals to current callState");
                return;
            }
            changeState(callState);
        }

        @Override
        public void onRecvCallStateChanged(EMACallSession callSession, int streamControlType) {
            EMLog.d(TAG, "onRecvCallStateChanged, callId: " + callSession.getCallId() + " StreamControlType:" + streamControlType);

            CallState callState = CallState.DISCONNECTED;

            if (streamControlType == EMACallSession.StreamControlType.PAUSE_VIDEO.ordinal()) {
                callState = CallState.VIDEO_PAUSE;
            } else if (streamControlType == EMACallSession.StreamControlType.PAUSE_VOICE.ordinal()) {
                callState = CallState.VOICE_PAUSE;
            } else if (streamControlType == EMACallSession.StreamControlType.RESUME_VIDEO.ordinal()) {
                callState = CallState.VIDEO_RESUME;
            } else if (streamControlType == EMACallSession.StreamControlType.RESUME_VOICE.ordinal()) {
                callState = CallState.VOICE_RESUME;
            } else {
                try {
                    throw new HyphenateException("onRecvCallStateChanged invalid streamControlType:" + streamControlType);
                } catch (HyphenateException e) {
                    e.printStackTrace();
                }
            }
            if (EMCallManager.this.callState.getState() == callState) {
                EMLog.d(TAG, "onRecvCallStateChanged toStatus equals to current callState");
                return;
            }
            changeState(callState);
        }


        @Override
        public void onNewRtcConnection(String callId, String to,
                                       RtcConnection.Listener listener, EMACallRtcImpl rtcImpl ) {

            EMLog.d(TAG, "onNewRtcConnection, remoteName: " + to);

            if (rtcImpl == null || listener == null) {
                return;
            }

            synchronized (EMCallManager.this) {
                if (mRtcConnection == null) {
                    mRtcConnection = createRtcConnection("rtc:" + to);
                }
                mRtcListener = listener;
                mRtcConnection.setListener(listener);
                rtcImpl.setRtcConnection(EMCallManager.this, mRtcConnection);
            }
        }
    }

    void onLogout() {
        stateChangeHandler.removeMessages(0);
    }

    void printStackTrace() {
        try {
            throw new Exception();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public EMCallOptions getCallOptions() {
        if (callOptions == null) {
            callOptions = new EMCallOptions(emaObject);
        }
        return callOptions;
    }

    public EMCallSession getCurrentCallSession() {
        return currentSession;
    }

    /**
     * \~chinese
     * 从外部直接输入视频数据, 仅支持NV21数据格式
     *
     * @param data
     * @param width
     * @param height
     * @param rotation
     *
     * \~english
     * input video data from external data source, only support NV21 format.
     *
     * @param data
     * @param width
     * @param height
     * @param rotation
     *
     * Use {@link #inputExternalVideoData(byte[], RtcConnection.FORMAT, int, int, int)} for instead.
     */
    @Deprecated
    public void inputExternalVideoData(byte[] data, int width, int height, int rotation) {
        inputExternalVideoData(data, RtcConnection.FORMAT.NV21, width, height, rotation);
    }

    /**
     * \~chinese
     * 从外部直接输入视频数据
     *
     * @param data
     * @param width
     * @param height
     * @param rotation
     *
     * \~english
     * input video data from external data source
     *
     * @param data
     * @param width
     * @param height
     * @param rotation
     *
     */
    public void inputExternalVideoData(byte[] data, RtcConnection.FORMAT format, int width, int height, int rotation) {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            final long captureTimeNs = TimeUnit.MILLISECONDS.toNanos(SystemClock.elapsedRealtime());
            rtc.inputExternalVideoData(data, format, width, height, rotation, captureTimeNs);
        }
    }
}
