/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package internal.org.apache.http.entity.mime.content;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

public class StringBody extends AbstractContentBody {
	private final byte[] content;
	private final Charset charset;

	public static StringBody create(String text, String mimeType, Charset charset) throws IllegalArgumentException {
		try {
			return new StringBody(text, mimeType, charset);
		} catch (UnsupportedEncodingException ex) {
			throw new IllegalArgumentException("Charset " + charset + " is not supported", ex);
		}
	}

	public static StringBody create(String text, Charset charset) throws IllegalArgumentException {
		return create(text, null, charset);
	}

	public static StringBody create(String text) throws IllegalArgumentException {
		return create(text, null, null);
	}

	public StringBody(String text, String mimeType, Charset charset) throws UnsupportedEncodingException {
		super(mimeType);
		if (text == null) {
			throw new IllegalArgumentException("Text may not be null");
		}
		if (charset == null) {
			charset = Charset.forName("US-ASCII");
		}
		this.content = text.getBytes(charset.name());
		this.charset = charset;
	}

	public StringBody(String text, Charset charset) throws UnsupportedEncodingException {
		this(text, "text/plain", charset);
	}

	public StringBody(String text) throws UnsupportedEncodingException {
		this(text, "text/plain", null);
	}

	public Reader getReader() {
		return new InputStreamReader(new ByteArrayInputStream(this.content), this.charset);
	}

	public void writeTo(OutputStream out) throws IOException {
		if (out == null) {
			throw new IllegalArgumentException("Output stream may not be null");
		}
		InputStream in = new ByteArrayInputStream(this.content);
		byte[] tmp = new byte[4096];
		int l;
		while ((l = in.read(tmp)) != -1) {
			out.write(tmp, 0, l);
		}
		out.flush();
	}

	public String getTransferEncoding() {
		return "8bit";
	}

	public String getCharset() {
		return this.charset.name();
	}

	public long getContentLength() {
		return this.content.length;
	}

	public String getFilename() {
		return null;
	}

}
