package com.hyphenate.push;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.support.annotation.NonNull;
import android.util.Pair;

import com.hyphenate.EMError;
import com.hyphenate.chat.EMClient;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.push.common.PushUtil;
import com.hyphenate.push.platform.IPush;
import com.hyphenate.push.platform.fcm.EMFCMPush;
import com.hyphenate.push.platform.hms.EMHMSPush;
import com.hyphenate.push.platform.meizu.EMMzPush;
import com.hyphenate.push.platform.mi.EMMiPush;
import com.hyphenate.push.platform.normal.EMNormalPush;
import com.hyphenate.push.platform.oppo.EMOppoPush;
import com.hyphenate.push.platform.vivo.EMVivoPush;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;

import org.apache.http.HttpStatus;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Random;

public class EMPushHelper {
    private static final String TAG = "EMPushHelper";

    private static final int TIMES_RETRY = 3;
    private static final int MSG_WHAT_REGISTER = 0;
    private static final int MSG_WHAT_UNREGISTER = 1;

    private Context context;
    private EMPushConfig pushConfig;
    private IPush pushClient;
    private Handler handler;

    private EMPushType pushType;
    private String pushToken;
    private boolean unregisterSuccess;

    private final Object bindLock = new Object();
    private final Object unbindLock = new Object();

    private PushListener pushListener;

    private EMPushHelper() {
        final HandlerThread handlerThread = new HandlerThread("token-uploader");
        handlerThread.start();

        handler = new Handler(handlerThread.getLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case MSG_WHAT_REGISTER:
                        synchronized (bindLock) { // 确保task在执行时没有新的tasks加入，避免该task执行完后把新加入的tasks清空。
                            String token = (String) msg.obj;
                            boolean registerSuccess = uploadTokenInternal(pushClient.getNotifierName(), token);
                            if (registerSuccess) {
                                removeMessages(MSG_WHAT_REGISTER);
                                return;
                            }
                            // 当所有task执行完且还没有上传成功时，更换为NORMAL push type。
                            boolean allTasksExecuted = !hasMessages(MSG_WHAT_REGISTER);
                            if (allTasksExecuted) {
                                EMPushHelper.this.onErrorResponse(pushType, EMError.PUSH_BIND_FAILED);
                                register(EMPushType.NORMAL);
                            }
                        }
                        break;
                    case MSG_WHAT_UNREGISTER:
                        unregisterSuccess = uploadTokenInternal(pushClient.getNotifierName(), "");
                        if (!unregisterSuccess) {
                            EMPushHelper.this.onErrorResponse(pushType, EMError.PUSH_UNBIND_FAILED);
                        }

                        synchronized (unbindLock) {
                            unbindLock.notifyAll();
                        }
                        break;
                    default:
                        super.handleMessage(msg);
                        break;
                }
            }
        };
    }

    public static EMPushHelper getInstance() {
        return InstanceHolder.INSTANCE;
    }

    public void setPushListener(PushListener callback) {
        this.pushListener = callback;
    }

    public void init(Context context, EMPushConfig config) {
        if (!PushUtil.isMainProcess(context)) {
            return;
        }

        EMLog.e(TAG, TAG + " init, config: " + config.toString());

        this.context = context.getApplicationContext();
        this.pushConfig = config;
    }

    public void register() {
        if (!PushUtil.isMainProcess(context)) {
            return;
        }

        EMPushType pushType = getPreferPushType(pushConfig);
        EMLog.e(TAG, TAG + " register, prefer push type: " + pushType);
        register(pushType);
    }

    public boolean unregister(boolean unbindToken) {
        EMLog.e(TAG, TAG + " unregister, unbind token: " + unbindToken);

        if (pushClient != null) pushClient.unregister(context);

        // 停止Token上传操作。
        handler.removeMessages(MSG_WHAT_REGISTER);

        if (!unbindToken) {
            pushType = EMPushType.NORMAL;
            return true;
        }

        deleteToken();
        // Wait for token delete result.
        synchronized (unbindLock) {
            try {
                unbindLock.wait();
            } catch (InterruptedException e) {
            }
        }
        if (unregisterSuccess) {
            pushType = EMPushType.NORMAL;
        }
        EMLog.e(TAG, "Push type after unregister is " + pushType);
        return unregisterSuccess;
    }

    public void onReceiveToken(EMPushType type, final String token) {
        EMLog.e(TAG, "onReceiveToken: " + type + " - " + token);
        this.pushToken = token;
        uploadToken(token);
    }

    public void onErrorResponse(EMPushType type, long resultCode) {
        EMLog.e(TAG, "onErrorResponse: " + type + " - " + resultCode);
        if (resultCode == EMError.PUSH_NOT_SUPPORT) {
            register(EMPushType.NORMAL);
        }

        if (pushListener != null) pushListener.onError(type, resultCode);
    }

    public EMPushType getPushType() {
        return pushType;
    }

    public String getPushToken() {
        return pushToken;
    }

    public String getFCMPushToken() {
        return EMPreferenceUtils.getInstance().getFCMPushToken();
    }

    public void setFCMPushToken(String token) {
        EMPreferenceUtils.getInstance().setFCMPushToken(token);
    }

    private void register(@NonNull EMPushType pushType) {
        if (this.pushType == pushType) {
            EMLog.e(TAG, "Push type " + pushType + " no change, return. ");
            return;
        }

        if (this.pushClient != null) {
            EMLog.e(TAG, pushClient.getPushType() + " push already exists, unregister it and change to " + pushType + " push.");
            pushClient.unregister(context);
        }

        this.pushType = pushType;

        switch (pushType) {
            case FCM:
                pushClient = new EMFCMPush();
                break;
            case MIPUSH:
                pushClient = new EMMiPush();
                break;
            case OPPOPUSH:
                pushClient = new EMOppoPush();
                break;
            case VIVOPUSH:
                pushClient = new EMVivoPush();
                break;
            case MEIZUPUSH:
                pushClient = new EMMzPush();
                break;
            case HMSPUSH:
                pushClient = new EMHMSPush();
                break;
            case NORMAL:
            default:
                pushClient = new EMNormalPush();
                break;
        }

        pushClient.register(context, pushConfig);
    }

    private void uploadToken(String token) {
        // Cancel all previous upload tasks first.
        handler.removeMessages(MSG_WHAT_REGISTER);

        synchronized (bindLock) { // 确保把tasks放入queue时没有task正在执行。以避免task执行完后把刚放进的tasks清空。
            for (int i = -1; i < TIMES_RETRY; i++) {
                Message msg = handler.obtainMessage(MSG_WHAT_REGISTER, token);
                if (i == -1) {
                    handler.sendMessage(msg);
                } else {
                    int delaySeconds = randomDelay(i);
                    EMLog.i(TAG, "Retry upload after " + delaySeconds + "s if failed.");
                    handler.sendMessageDelayed(msg, delaySeconds * 1000);
                }
            }
        }
    }

    private void deleteToken() {
        handler.obtainMessage(MSG_WHAT_UNREGISTER).sendToTarget();
    }

    private boolean uploadTokenInternal(String notifierName, String token) {
        String remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl() + "/users/"
                + EMClient.getInstance().getCurrentUser();
        try {
            EMLog.e(TAG, "uploadTokenInternal, token=" + token + ", url=" + remoteUrl
                    + ", notifier name=" + notifierName);

            DeviceUuidFactory deviceFactory = new DeviceUuidFactory(EMClient.getInstance().getContext());

            JSONObject json = new JSONObject();
            json.put("device_token", token);
            json.put("notifier_name", notifierName);
            json.put("device_id", deviceFactory.getDeviceUuid().toString());

            Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl,
                    json.toString(), EMHttpClient.PUT);
            int statusCode = response.first;
            String content = response.second;

            if (statusCode == HttpStatus.SC_OK) {
                EMLog.e(TAG, "uploadTokenInternal success.");
                return true;
            }

            EMLog.e(TAG, "uploadTokenInternal failed: " + content);
        } catch (Exception e) {
            EMLog.e(TAG, "uploadTokenInternal exception: " + e.toString());
        }

        return false;
    }

    private EMPushType getPreferPushType(EMPushConfig pushConfig) {
        EMPushType[] supportedPushTypes = new EMPushType[]{
                EMPushType.FCM,
                EMPushType.MIPUSH,
                EMPushType.HMSPUSH,
                EMPushType.MEIZUPUSH,
                EMPushType.OPPOPUSH,
                EMPushType.VIVOPUSH,
        };

        ArrayList<EMPushType> enabledPushTypes = pushConfig.getEnabledPushTypes();

        for (EMPushType pushType : supportedPushTypes) {
            if (enabledPushTypes.contains(pushType) && isSupportPush(pushType, pushConfig)) {
                return pushType;
            }
        }

        return EMPushType.NORMAL;
    }

    private boolean isSupportPush(EMPushType pushType, EMPushConfig pushConfig) {
        boolean support;
        if (pushListener != null) {
            support = pushListener.isSupportPush(pushType, pushConfig);
        } else {
            support = PushUtil.isSupportPush(pushType, pushConfig);
        }
        EMLog.i(TAG, "isSupportPush: " + pushType + " - " + support);
        return support;
    }

    public int randomDelay(int attempts) {
        if (attempts == 0) {
            return new Random().nextInt(5) + 1;
        }

        if (attempts == 1) {
            return new Random().nextInt(54) + 6;
        }

        return new Random().nextInt(540) + 60;
    }

    private static class InstanceHolder {
        static EMPushHelper INSTANCE = new EMPushHelper();
    }
}
