/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of emaObject information or reproduction of emaObject material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package com.hyphenate.chat;

import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.EMCallBack;
import com.hyphenate.chat.adapter.EMACallback;
import com.hyphenate.chat.adapter.message.EMACmdMessageBody;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMALocationMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessage.EMADirection;
import com.hyphenate.chat.adapter.message.EMAMessage.EMAMessageStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMATextMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.chat.adapter.message.EMAVoiceMessageBody;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * \~chinese
 * 代表一条发送或接收到的消息
 * 
 * <p>
 * 构造一条文本发送消息
 * <pre>
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * </pre>
 * 
 * <p>
 * 构造一条图片消息
 * <pre>
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * </pre>
 * 
 * \~english
 * represent a sent/recv message
 *<p>
 * construct a new send text message
 * <pre>
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * </pre>
 * 
 * <p>
 * construct a new recv text message
 * <pre>
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * </pre>
 */
public class EMMessage extends EMBase<EMAMessage> implements Parcelable, Cloneable{
    static final String ATTR_ENCRYPTED = "isencrypted";
    
    private final static String TAG = "msg";

    /**
     * \~chinese
     * 消息类型：文本，图片，视频，位置，语音，文件,透传消息
     * 
     * \~english
     * message type:TXT,IMAGE,VIDOE,LOCATION,VOICE,FILE,CMD
     */
    public enum Type {
        TXT, IMAGE, VIDEO, LOCATION, VOICE, FILE, CMD
    }
    
    /**
     * \~chinese
     * 消息的方向类型：区分是发送消息还是接收到的消息
     * 
     * \~english
     * message direction,send direction or receive direction
     */
    public enum Direct {
        SEND, RECEIVE
    }
    
    /**
     * \~chinese
     * 消息的发送/接收状态：成功，失败，发送/接收过程中，创建成功待发送
     * 
     * \~english
     * message status
     */
    public enum Status {
        SUCCESS, FAIL, INPROGRESS, CREATE
    }
    
    EMMessageBody body;
    
    /**
     * \~chinese
     * 消息的发送/接收状态：成功，失败，发送/接收过程中，创建成功待发送
     * 
     * \~english
     * message status
     */
    public Status status() {
    	EMAMessage.EMAMessageStatus st = emaObject._status();
    	Status s = Status.CREATE;
    	if (st == EMAMessageStatus.NEW) {
    		s = Status.CREATE;
    	} else if (st == EMAMessageStatus.SUCCESS) {
    		s = Status.SUCCESS;
    	} else if (st == EMAMessageStatus.FAIL) {
    		s = Status.FAIL;
    	} else if (st == EMAMessageStatus.DELIVERING) {
    		s = Status.INPROGRESS;
    	}
    	return s;
    }
    
    /**
     * \~chinese
     * 聊天类型：单聊，群聊，聊天室
     *
     *\~english
     * chat type : private chat, group chat, chat room
     */
    public enum ChatType {
        /**
         * 单聊
         * private single chat
         */
        Chat, 
        /**
         * 群聊
         * group chat
         */
        GroupChat,
        /**
         * 聊天室
         * chat room
         */
        ChatRoom
    }
    
    public EMMessage(EMAMessage message) {
        emaObject = message;
    }
    
    /**
     * \~chinese
     * 设置消息的状态
     * 
     * \~english
     * set the status of the message
     *
     */
    public void setStatus(Status status) {
    	EMAMessage.EMAMessageStatus s = EMAMessageStatus.SUCCESS;
    	if (status == Status.CREATE) {
    		s = EMAMessageStatus.NEW;
    	} else if (status == Status.SUCCESS) {
    		s = EMAMessageStatus.SUCCESS;
    	} else if (status == Status.FAIL) {
    		s = EMAMessageStatus.FAIL;
    	} else if (status == Status.INPROGRESS) {
    		s = EMAMessageStatus.DELIVERING;
    	}
    	emaObject.setStatus(s.ordinal());
    }
    
    /**
     * \chinese
     * 获取消息类型
     * @return
     * 
     * \~english
     * get message chat type
     * @return
     */
    public Type getType() {
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		int type = bodies.get(0).type();
    		//Log.d(TAG, "type:" + type);
    		if (type == Type.TXT.ordinal()) {
    			return Type.TXT;
    		} else if (type == Type.IMAGE.ordinal()) {
    			return Type.IMAGE;
    		} else if (type == Type.CMD.ordinal()) {
    			return Type.CMD;
    		} else if (type == Type.FILE.ordinal()) {
    			return Type.FILE;
    		} else if (type == Type.VIDEO.ordinal()) {
    			return Type.VIDEO;
    		} else if (type == Type.VOICE.ordinal()) {
    			return Type.VOICE;
    		} else if (type == Type.LOCATION.ordinal()) {
    			return Type.LOCATION;
    		}
    	}
        return Type.TXT;
    }
    
    /**
     * \~chinese
     * 获取消息的body
     * @return
     * 
     * \~english
     * get message body
     * @return
     */
    public EMMessageBody getBody() {
        if (body != null) {
            return body;
        }
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		EMAMessageBody emaBody = bodies.get(0);
    		if (emaBody instanceof EMATextMessageBody) {
    			body = new EMTextMessageBody((EMATextMessageBody)emaBody);
    		} else if (emaBody instanceof EMACmdMessageBody) {
    			body = new EMCmdMessageBody((EMACmdMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVideoMessageBody) {
    			body = new EMVideoMessageBody((EMAVideoMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVoiceMessageBody) {
    			body = new EMVoiceMessageBody((EMAVoiceMessageBody)emaBody);
    		} else if (emaBody instanceof EMAImageMessageBody) {
    			body = new EMImageMessageBody((EMAImageMessageBody)emaBody);
    		} else if (emaBody instanceof EMALocationMessageBody) {
    			body = new EMLocationMessageBody((EMALocationMessageBody)emaBody);
    		} else if (emaBody instanceof EMAFileMessageBody) {
    			body = new EMNormalFileMessageBody((EMAFileMessageBody)emaBody);
    		}
        	return body;
    	}

    	return null;
    }
    
    /**
     * \~chinese
     * 获取消息的时间(server time)
     * @return
     * 
     * \~english
     * get message time stamp(server time)
     * @return
     */
    public long getMsgTime() {
    	return emaObject.timeStamp();
    }
    
    
    /**
     * \~chinese
     * 设置消息时间(server time)
     * @param msgTime
     * 
     * \~english
     * set message time stamp(server time)
     * @param msgTime
     */
    public void setMsgTime(long msgTime) {
		emaObject.setTimeStamp(msgTime);
	}
    
    /**
     * \~chinese
     * 获取消息的本地接收时间
     * @return
     * 
     * \~english
     * get local receive time
     * @return
     */
    public long localTime() {
        return emaObject.getLocalTime();
    }
    
    /**
     * \~chinese
     * 设置消息的服务器时间
     * @param serverTime
     * 
     * \~english
     * set message local receive time
     * @param serverTime
     */
    public void setLocalTime(long serverTime) {
        emaObject.setLocalTime(serverTime);
    }

	/**
	 * \~chinese
     * 创建一个发送消息
     * @param type 消息类型
     * @return 消息对象
     * 
     * \~english
     * create a new send message
     * @param type the message type
     * @return the message object
     */
    public static EMMessage createSendMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createSendMessage(self(), "", null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个接收消息
     * @param type 消息类型
     * @return 消息对象
     * 
     * \~english
     * create a new recv message
     * @param type message type
     * @return the message object
     */
    public static EMMessage createReceiveMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        msg.setTo(EMSessionManager.getInstance().currentUser.getUsername());
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个文本发送消息
     * @param content 文本内容
     * @param username 消息接收人或群id
     * @return
     * 
     * \~english
     * @param content text content
     * @param username the recipient(user or group) id
     * @return
     */
    public static EMMessage createTxtSendMessage(String content, String username){
        if (content.length() > 0) {
        	EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
        	EMTextMessageBody txtBody = new EMTextMessageBody(content);
        	msg.addBody(txtBody);
        	msg.setTo(username);
            return msg;
        }
        EMLog.e(TAG, "text content size must be greater than 0");
        return null;
    }
    
    /**
     * \~chinese
     * 创建一个语音发送消息
     * @param filePath 语音文件路径
     * @param timeLength 语音时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return
     * 
     * \~english
     * create a voice send message
     * @param filePath the path of the voice file
     * @param timeLength the time length of the voice(unit s)
     * @param username  the recipient id
     * @return
     */
    public static EMMessage createVoiceSendMessage(String filePath, int timeLength, String username){
        if (!(new File(filePath).exists())) {
            EMLog.e(TAG, "voice file does not exsit");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.VOICE);
        // 如果是群聊，设置chattype,默认是单聊
        EMVoiceMessageBody body = new EMVoiceMessageBody(new File(filePath), timeLength);
        message.addBody(body);
        
        message.setTo(username);
        return message;
    }
    
    /**
     * \~chinese
     * 创建一个图片发送消息
     * @param filePath 图片路径
     * @param sendOriginalImage 是否发送原图(默认大于100K的图片sdk会进行压缩)
     * @param username 消息接收人或群id
     * @return
     * 
     * \~english
     * create a image send message
     * @param filePath the path of the image
     * @param sendOriginalImage whether to send the original(if image greater than 100k sdk will be compressed)
     * @param username the recipient id
     * @return
     */
    public static EMMessage createImageSendMessage(String filePath, boolean sendOriginalImage, String username){
        if (!(new File(filePath).exists())) {
            EMLog.e(TAG, "image file does not exsit");
            return null;
        }
        // create and add image message in view
        final EMMessage message = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
        message.setTo(username);
        EMImageMessageBody body = new EMImageMessageBody(new File(filePath));
        // 默认超过100k的图片会压缩后发给对方，可以设置成发送原图
        body.setSendOriginalImage(sendOriginalImage);
        message.addBody(body);
        return message;
    }
    
    /**
     * \~chinese
     * 创建一个视频发送消息
     * @param videofilePath 视频文件路径
     * @param imageThumbPath 视频第一帧图缩略图
     * @param timeLength 视频时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return
     * 
     * \~english
     * create a video send message
     * @param videofilePath the path of the video file
     * @param imageThumbPath the path of the thumbnail
     * @param timeLength the length of the video time, unit s
     * @param username the recipient id
     * @return
     */
    public static EMMessage createVideoSendMessage(String videofilePath, String imageThumbPath, int timeLength, String username){
        File videoFile = new File(videofilePath);
        if (!videoFile.exists()) {
            EMLog.e(TAG, "video file does not exist");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.VIDEO);
        message.setTo(username);
        EMVideoMessageBody body = new EMVideoMessageBody(videofilePath, imageThumbPath, timeLength, videoFile.length());
        message.addBody(body);
        return message;
    }
    
    /**
     * \~chinese
     * 创建一个位置发送消息
     * @param latitude 纬度
     * @param longitude 经度
     * @param locationAddress 位置详情
     * @param username 消息接收人或群id
     * @return
     * 
     * \~english
     * create a location send message
     * @param latitude the latitude
     * @param longitude the longitude
     * @param locationAddress location details
     * @param username the recipient id
     * @return
     */
    public static EMMessage createLocationSendMessage(double latitude, double longitude, String locationAddress, String username){
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.LOCATION);
        EMLocationMessageBody locBody = new EMLocationMessageBody(locationAddress, latitude, longitude);
        message.addBody(locBody);
        message.setTo(username);
        
        return message;
    }
    
    /**
     * \~chinese
     * 创建一个普通文件发送消息
     * @param filePath 文件路径
     * @param username 消息接收人或群id
     * @return
     * 
     * \~english
     * create a normal file send message
     * @param filePath the path of the file
     * @param username the recipient id
     * @return
     */
    public static EMMessage createFileSendMessage(String filePath, String username){
        File file = new File(filePath);
        if(!file.exists()){
            EMLog.e(TAG, "file does not exist");
            return null;
        }
        // 创建一个文件消息
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.FILE);

        message.setTo(username);
        // add message body
        EMNormalFileMessageBody body = new EMNormalFileMessageBody(new File(filePath));
        message.addBody(body);
        return message;
    }
    
    /**
     * \~chinese
     * 添加消息体. （现在只支持添加一个 MesageBody)
     * @param body 消息体
     * 
     * \~english
     * add a message body,only support add one now
     * @param body the message body
     */
    public void addBody(EMMessageBody body) {
        this.body = body;
        emaObject.addBody(body.emaObject);
    }
    
    /**
     * \~chinese
     * 获取消息发送者的用户名
     * get sender name
     * @return 用户名
     * 
     * \~english
     * get the sender id
     * @return user id 
     */
    public String getFrom() {
        return emaObject.from();
    }
    
    /**
     * \~chinese
     * 设置消息发送者
     * @param from
     * 
     * \~english
     * set sender id
     * @param from
     */
    public void setFrom(String from) {
		emaObject.setFrom(from);
		if (!self().equals(from)) {
			emaObject.setConversationId(from);
		}
	}

    /**
     * \~chinese
     * 设置消息的接收者
     * @param to
     * 
     * \~english
     * set receiver name
     * @param to
     */
	public void setTo(String to) {
		emaObject.setTo(to);
		if (!self().equals(to)) {
			emaObject.setConversationId(to);
		}
	}

	/**
	 * \~chinese
     * 获取消息接收者的用户名
     * @return
     * 
     * \~english
     * get receiver name
     * @return
     */
    public String getTo() {
        return emaObject.to();
    }
    
    /**
     * \~chinese
     * 获取消息的id
     * @return id
     * 
     * \~english
     * get message id
     * @return id
     */
    public String getMsgId() {
        return emaObject.msgId();
    }
    
    /**
     * \~chinese
     * 设置消息id
     * @param msgId
     * 
     * \~english
     * set local message id
     * @param msgId
     */
    public void setMsgId(String msgId){
    	emaObject.setMsgId(msgId);
    }

    /**
     * EMXXXListener doesn't need EMCallbackHolder, user will be responsible for add, remove listener reference.
     * Message's callback function is different from listener, user add callback, and won't remove the reference.
     * Old implementation for Message's callback is simply hold the reference, which leads to memory leak.
     * SDK caches hold reference of message, message hold callback, callback hold Activity's this reference, so activities will never be released.
     *
     * Solution:
     * 1. change the message's callback to weak reference.
     * 2. when sendMessage or downloadMessageAttachment make the reference be strong one.
     * 3. when callback be executed, onSuccess or onError, release the reference.
     *
     * Declaim:
     * 1. don't set message be strong when user call setMessageStatusCallback, user may just attach the callback, there will not be onError or onSuccess to be called
     */
    static class EMCallbackHolder implements EMCallBack {

        EMCallbackHolder(EMCallBack callback) {
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void update(EMCallBack callback) {
            if (strong != null) {
                strong = callback;
                return;
            }
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void makeItStrong() {
            if (strong != null) {
                return;
            }
            if (weak != null && weak.get() != null) {
                strong = weak.get();
            }
        }

        synchronized void release() {
            if (strong == null) {
                return;
            }
            weak = new WeakReference<EMCallBack>(strong);
            strong = null;
        }

        synchronized EMCallBack getRef() {
            if (strong != null) {
                return strong;
            }
            if (weak != null) {
                EMCallBack callBack = weak.get();
                if (callBack == null) {
                    EMLog.d(TAG, "getRef weak:" + callBack);
                }
                return callBack;
            }
            return null;
        }

        public void onSuccess() {
            if (innerCallback != null) {
                innerCallback.onSuccess();
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onSuccess();
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onError(int code, String error) {
            if (innerCallback != null) {
                innerCallback.onError(code, error);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onError(code, error);
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onProgress(int progress, String status) {
            if (innerCallback != null) {
                innerCallback.onProgress(progress, status);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onProgress(progress, status);
            }
        }

        // strong & weak are used for external callback reference
        // inner callback is used by chat manager sendMessage, to handle changes of msgId and msg time
        private EMCallBack strong;
        private WeakReference<EMCallBack> weak;
        EMCallBack innerCallback = null;
    }

    EMCallbackHolder messageStatusCallBack;

    synchronized void setInnerCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.innerCallback = callback;
        } else {
            messageStatusCallBack = new EMCallbackHolder(null);
            messageStatusCallBack.innerCallback = callback;
        }
        setCallback(messageStatusCallBack);
    }

    /**
     * \~chinese
     * 设置消息状态改变的回调
     * @param callback
     * 
     * \~english
     * set message status callback, your app should set emaObject callback to get message status and then refresh the ui accordingly
     * @param callback
     */
    public synchronized void setMessageStatusCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.update(callback);
        } else {
            messageStatusCallBack = new EMCallbackHolder(callback);
        }
        setCallback(messageStatusCallBack);
    }


    void setCallback(final EMCallbackHolder holder) {
        this.emaObject.setCallback(new EMACallback(new EMCallBack(){

            @Override
            public void onSuccess() {
                if(holder != null){
                    holder.onSuccess();
                }
            }

            @Override
            public void onProgress(int progress, String status) {
                if(holder != null){
                    holder.onProgress(progress, status);
                }
            }

            @Override
            public void onError(int code, String error) {
                if(holder != null){
                    holder.onError(code, error);
                }
            }
        }));
    }

    void makeCallbackStrong() {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.makeItStrong();
        }
    }


    public String toString() {
        String sb = "msg{from:" + getFrom() +
                ", to:" + getTo() +
                " body:" + getBody();
        return sb;
    }
    
    /**
     * \~chinese
     * 设置消息的boolean 类型扩展属性
     * @param attribute 属性名
     * @param value, 属性值
     * 
     * \~english
     * set a boolean type extra attributes of the message
     * @param attribute attribute key
     * @param value attribute value
     */
    public void setAttribute(String attribute, boolean value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 设置消息的int 类型扩展属性
     * @param attribute 属性名
     * @param value, 属性值
     * 
     * \~english
     * set a int type extra attributes of the message
     * @param attribute attribute key
     * @param value attribute value
     */
    public void setAttribute(String attribute, int value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }


    /**
     * \~chinese
     * 设置消息的long 类型扩展属性
     * @param attribute 属性名
     * @param value, 属性值
     * 
     * \~english
     * set a long type extra attributes of the message
     * @param attribute attribute key
     * @param value attribute value
     */
    public void setAttribute(String attribute, long value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    
    /**
     * \~chinese
     * 设置消息的 JSONObject 类型扩展属性
     * @param attribute 属性名
     * @param value, 属性值
     * 
     * \~english
     * set a jsonobject type extra attributes of the message
     * @param attribute attribute key
     * @param value attribute value
     */
    public void setAttribute(String attribute, JSONObject value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的 JSONArray 类型扩展属性
     * @param attribute 属性名
     * @param value, 属性值
     * 
     * \~english
     * set a jsonarray type extra attributes of the message
     * @param attribute attribute key
     * @param value attribute value
     */
    public void setAttribute(String attribute, JSONArray value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的 string 类型扩展属性
     * @param attribute 属性名
     * @param value, 属性值
     * 
     * \~english
     * set a string type extra attributes of the message
     * @param attribute attribute key
     * @param value attribute value
     */
    public void setAttribute(String attribute, String value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 获取 boolean 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * get a boolean type extra attribute
     * @param attribute the attribute name
     * return
     * @throws HyphenateException
     */
    public boolean getBooleanAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取 boolean 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * get a boolean type extra attribute
     * @param attribute the attribute name
     * @param defaultValue the default value you want
     * @return
     */
    public boolean getBooleanAttribute(String attribute, boolean defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取 int 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * get a int type extra attribute
     * @param attribute the attribute name
     * @param defaultValue the default value you want
     * @return 
     */
    public int getIntAttribute(String attribute,int defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取 long 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * get a long type extra attribute
     * @param attribute the attribute name
     * @param defaultValue the default value you want
     * @return 
     */
    public long getLongAttribute(String attribute,long defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.longValue();
    }
    
    /**
     * \~chinese
     * 获取 int 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * get a int type extra attribute
     * @param attribute the attribute name
     * @return the value
     * @throws HyphenateException
     */
    public int getIntAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取 long 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * get long type extra attribute
     * @param attribute the attribute name
     * @return the value
     * @throws HyphenateException
     */
    public long getLongAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.longValue();
    }
    
    /**
     * \~chinese
     * 获取 String 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * get a string type extra attribute
     * @param attribute the attribute name
     * @return the value
     * @throws HyphenateException
     */
    public String getStringAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.toString();
    }
    
    /**
     * \~chinese
     * 获取 String 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * get a string type extra attribute
     * @param attribute the attribute name
     * @param defaultValue the default value you want
     * @return 
     */
    public String getStringAttribute(String attribute, String defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            return defaultValue;
        }        
        return val.toString();
    }
    
    
    /**
     * \~chinese
     * 获取 JSONObject 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * get a jsonobject type attribute
     * @param attribute
     * @return the value
     * @throws HyphenateException
     */
    public JSONObject getJSONObjectAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            return null;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "{}", val);
        if (exists) {
            try {
                JSONObject jsonObj = new JSONObject(val.toString());
                return jsonObj;
            } catch (JSONException e) {
            }
        }
        throw new HyphenateException("attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取 JSONArray 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * get a jsonarray type attribute
     * @param attribute
     * @return the value
     * @throws HyphenateException
     */
    public JSONArray getJSONArrayAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            return null;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "[]", val);
        if (exists) {
            try {
                JSONArray jsonArray = new JSONArray(val.toString());
                return jsonArray;
            } catch (JSONException e) {
            }
        }
        throw new HyphenateException("attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取聊天类型
     * @return ChatType
     * 
     * \~english
     * get chat type
     *  @return ChatType
     */
    public ChatType getChatType() {
    	EMAMessage.EMAChatType t = emaObject.chatType();
    	ChatType type = ChatType.Chat;
    	if (t == EMAMessage.EMAChatType.SINGLE) {
    		type = ChatType.Chat;
    	} else if (t == EMAMessage.EMAChatType.GROUP) {
    		type = ChatType.GroupChat;
    	} else {
    		type = ChatType.ChatRoom;
    	}
    	return type;
    }
    
    /**
     * \~chinese
     * 设置聊天类型， 缺省为单聊  ChatType.Chat
     * @param chatType
     * @see ChatType
     * 
     * \~english
     * set chat type, the default is single chat ChatType.Chat
     * @param chatType
     * @see ChatType
     */
    public void setChatType(ChatType chatType) {
    	EMAMessage.EMAChatType t = EMAMessage.EMAChatType.SINGLE;
    	if (chatType == ChatType.Chat) {
    		t = EMAMessage.EMAChatType.SINGLE;
    	} else if (chatType == ChatType.GroupChat) {
    		t = EMAMessage.EMAChatType.GROUP;
    	} else {
    		t = EMAMessage.EMAChatType.CHATROOM;
    	}
    	emaObject.setChatType(t);
    }
    

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(getMsgId());
    }
    
    public static final Parcelable.Creator<EMMessage> CREATOR
        = new Parcelable.Creator<EMMessage>() {
        public EMMessage createFromParcel(Parcel in) {
            EMMessage msg = null;
            try {
                msg = new EMMessage(in);
            } catch (HyphenateException e) {
                e.printStackTrace();
            }
            return msg;
        }

        public EMMessage[] newArray(int size) {
            return new EMMessage[size];
        }
    };
    
	@SuppressWarnings("CloneDoesntCallSuperClone")
    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    /**
     * \~chinese
     * 对方是否已读
     * 
     * \~english
     * Read Ack
     */
	public boolean isAcked() {
		return emaObject.isAcked();
	}

    /**
     * \~chinese
     * 设置对方是否已读
     * 
     * \~english
     * Sets whether the other has been read,
     * not suppposed to be called by app
     */
	public void setAcked(boolean isAcked) {
		emaObject.setIsAcked(isAcked);
	}

    /**
     * \~chinese
     * 获取消息是否已到达对方
     * 
     * \~english
     * Delivery Ack, check if the peer has received the message
     *
     */
	public boolean isDelivered() {
		return emaObject.isDeliverAcked();
	}

	public void setDelivered(boolean isDelivered) {
		emaObject.setIsDeliverAcked(isDelivered);
	}

    /**
     * unread flag
     */
	public boolean isUnread() {
		return !emaObject.isRead();
	}

	public void setUnread(boolean unread) {
		emaObject.setIsRead(!unread);
	}

	/**
	 * \~chinese
	 * 获取语音是否已听
	 * @return
	 * 
	 * \~english
	 * get whether the message is listen
	 */
	public boolean isListened() {
		return emaObject.isListened();
	}

	/**
	 * \~chinese
	 * 设置语音是否已听
	 * @param isListened
	 * 
	 * \~english
	 * Sets whether the other has been listen
	 * @param isListened
	 */
	public void setListened(boolean isListened) {
	    emaObject.setListened(isListened);
	}
	
	/**
	 * \~chinese
	 * 获取通话对象
	 * @return
	 * 
	 * \~english
	 * get the peer's id
	 */
	public String getUserName() {
		String username = "";
		if (getFrom() != null && getFrom().equals(EMClient.getInstance().getCurrentUser())) {
			username = getTo();
		} else {
			username = getFrom();
		}
		return username;
	}
	
	public void setDeliverAcked(boolean isDeliverAcked) {
	    emaObject.setIsDeliverAcked(isDeliverAcked);
	}

    /**
     * \~chinese
     * 消息包含附件的上传或者下载进度，值的范围0-100。
     * 消息附件的缩略图不涉及进度信息
     *
     * \~english
     * indicate message attachment upload or download progress, value ranges between 0-100.
     * for message attachment's thumbnail, it doesn't has progress information
     * @return
     */
	public int progress() {
	    return emaObject.progress();
	}

    /**
     * \~chinese
     * 对于app开发者来说，通常不需要主动设置进度，
     *
     * \~english
     * for app developing, it doesn't need set progress.
     * @param progress
     */
    public void setProgress(int progress) {
        emaObject.setProgress(progress);
    }

	/**
	 * \~chinese
	 * 消息方向
	 * 
	 * \~english
	 * the message direction
	 */
	public Direct direct() {
	    if (emaObject.direction() == EMADirection.SEND) {
	        return Direct.SEND;
	    } else {
	        return Direct.RECEIVE;
	    }
	}
	
	/**
	 * \~chinese
	 * 设置消息的方向
	 * @param dir
	 * 
	 * \~english
	 * set message direction
	 * @param dir
	 */
	public void setDirection(Direct dir) {
	    emaObject.setDirection(dir.ordinal());
	}
	
	public String conversationId() {
	    return emaObject.conversationId();
	}

    static String self() {
        String myself = EMClient.getInstance().getCurrentUser();
        if (myself == null) {
            //currentUser is only set after login successfully
            //since it is not when not logedin, use last login user instead
            myself = EMSessionManager.getInstance().getLastLoginUser();
        }
        return myself;
    }

    private EMMessage(Parcel in) throws HyphenateException {
        String msgId = in.readString();
        EMMessage msg = EMClient.getInstance().chatManager().getMessage(msgId);
        if (msg == null) {
            throw new HyphenateException("EMMessage constructed from parcel failed");
        }
        emaObject = msg.emaObject;
    }

    /**
     * \~chinese
     * 获取消息包含的全部扩展字段，并以Map形式返回
     *
     * @return 消息的扩展字段，以Map形式返回，返回Entry.value的具体类型有Boolean, Integer, Long, Float, Double, String
     *   输入JsonObject的属性，map结构返回时属性是String
     *
     * \~english
     * get message extension, return type is Map&#60;String, Object&#62;
     *
     * @return return message extension, return type is Map&#60;String, Object&#62;
     *     Object can be Boolean, Integer, Long, Float, Double, String
     *     If the input is JsonObject or JsonArray, which use setAttribute(String attribute, JSONObject json) to passed in, the Map.Entry.value type is String.
     */
    public Map<String, Object> ext() {
        return emaObject.ext();
    }
}
