/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.chat.EMMessage.ChatType;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAConversation.EMASearchDirection;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * \~chinese
 * EMConversation 代表和一个用户的对话，包含发送和接收的消息
 * 下面的示例会取得对话中未读的消息数：
 * 
 * EMConversation conversation = EMClient.getInstance().chatManager().getConversation(&quot;user1&quot;);
 * int unread = conversation.getUnreadMsgCount();
 *
 * @version 3.0
 * 
 * \~english
 * EMConversation represent a conversation chating with a user, containing chat messages.
 * Example illustrate how to get unread message count
 * 
 * EMConversation conversation = EMClient.getInstance().chatManager().getConversation(&quot;user1&quot;);
 * int unread = conversation.getUnreadMsgCount();
 *
 * @version 3.0
*/

public class EMConversation extends EMBase<EMAConversation> {

    private static final String TAG = "conversation";
    private static final int LIST_SIZE = 512;

    EMConversation(EMAConversation conversation) {
        emaObject = conversation;
    }

    /**
     * \~chinese
     * 会话类型: 包含单聊，群聊，聊天室，讨论组(暂时不可用)，客服
     *
     * \~english
     * Conversation type
     */
    public enum EMConversationType {
        /**
         *\~chinese 
         * 单聊会话
         *
         *\~english
         * one-to-one chat
         */
        Chat,

        /**
         *\~chinese
         * 群聊会话
         *
         *\~english
         * group chat  
         */
        GroupChat,

        /**
         *\~chinese
         *聊天室会话
         *\~english
         *chat room
         */
        ChatRoom,

        /**
         *\~chinese
         * 目前没有使用
         *\~english
         * Not current used
         */
        DiscussionGroup,
        HelpDesk
    }

    /**
     *  \~chinese
     *  消息搜索方向
     *
     *  \~english
     *  Message search direction
     */
    public enum EMSearchDirection {
        /**
         *\~chinese
         *  向上搜索 
         *
         *\~english
         *  Search older messages 
         */
        UP,     

        /**
         *\~chinese
         *  向下搜索
         *
         *\~english
         *  Search newer messages
         */
        DOWN    
    }

    /**
     * \~chinese
     * 会话ID
     * 对于单聊类型，会话ID同时也是对方用户的名称。
     * 对于群聊类型，会话ID同时也是对方群组的ID，并不同于群组的名称。
     * 对于聊天室类型，会话ID同时也是聊天室的ID，并不同于聊天室的名称。
     * 对于HelpDesk类型，会话ID与单聊类型相同，是对方用户的名称。
     *
     * \~english
     * conversation ID
     * For single chat，conversation ID is to chat user's name
     * For group chat, conversation ID is groupID(), different with getGroupName()
     * For chat room, conversation ID is chatroom ID, different with chat room name()
     * For help desk, it is same with single chat, conversationID is also chat user's name
     */
    public String conversationId() {
        return emaObject.conversationId();
    }

    /**
     * \~chinese
     * 获取会话类型
     *  
     * \~english
     * get conversation type
     */
    public EMConversationType getType() {
        EMAConversationType t = emaObject._getType();
        if (t == EMAConversationType.CHAT) {
            return EMConversationType.Chat;
        }
        if (t == EMAConversationType.GROUPCHAT) {
            return EMConversationType.GroupChat;
        }
        if (t == EMAConversationType.CHATROOM) {
            return EMConversationType.ChatRoom;
        }
        if (t == EMAConversationType.DISCUSSIONGROUP) {
            return EMConversationType.DiscussionGroup;
        }
        if (t == EMAConversationType.HELPDESK) {
            return EMConversationType.HelpDesk;
        }
        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 获取此对话中未读取的消息数量
     *
     * \~english
     * get unread message count
     */
    public int getUnreadMsgCount() {
        return emaObject.unreadMessagesCount();
    }

    /**
     *  \~chinese
     *  将所有未读消息设置为已读
     *
     *  \~english
     *  Mark all messages as read
     *
     */
    public void markAllMessagesAsRead() {
        emaObject.markAllMessagesAsRead(true);
    }

    /**
     * \~chinese
     * 获取本地存储会话的全部消息数目
     *
     * \~english
     * get all messages count in this conversation
     */
    public int getAllMsgCount() {
        return emaObject.messagesCount();
    }

    /**
     * \~chinese
     * 根据传入的参数从db加载startMsgId之前(存储顺序)指定数量的message，
     * 加载到的messages会加入到当前conversation的messages里
     *
     * @param startMsgId    加载这个id之前的message
     * @param pageSize      加载多少条
     * @return              消息列表
     *
     * \~english
     * load message from database, message id starting from param startMsgId, messages will also store in to memory cache
     * so next time calling getAllMessages(), result will contain those messages
     *
     * @param startMsgId    message storage time will before this ID, if startMsgId is "" or null, will load last messages in database.
     * @param pageSize      message count to be loaded.
     * @return              messages
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize) {
        List<EMAMessage> msgs = emaObject.loadMoreMessages(startMsgId, pageSize, EMASearchDirection.UP);
        List<EMMessage> result = new ArrayList<EMMessage>();
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(result);
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的message。
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @return           消息列表
     *
     * \~english
     * search message from database according the parameter
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param timeStamp  the timestamp for search
     * @param maxCount   the max number of message to search
     * @return           the list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(long timeStamp, int maxCount, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(timeStamp, maxCount, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的message。
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param type       消息类型，文本、图片、语音等等
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @param from       搜索来自某人的消息，适用于搜索群组里的消息
     * @return           消息列表
     *
     * \~english
     * search message from database according the parameter
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param type       message type, TXT、VOICE、IMAGE etc.
     * @param timeStamp  the timestamp for search
     * @param maxCount   the max number of message to search
     * @param from       who the message from, used to search in group
     * @return           the list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(EMMessage.Type type, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的message。
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param keywords   搜索消息中的关键词
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @param from       搜索来自某人的消息，适用于搜索群组里的消息
     * @return           消息列表
     *
     * \~english
     * search message from database based the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param keywords   the keywords in message.
     * @param timeStamp  the timestamp for search
     * @param maxCount   the max number of message to search
     * @param from       who the message from, used to search in group
     * @return           the list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的message。
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param startTimeStamp   搜索的起始时间
     * @param endTimeStamp     搜索的结束时间
     * @param maxCount         搜索结果的最大条数
     * @return                 消息列表
     *
     * \~english
     * search message from database based the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param startTimeStamp    start timestamp for search in.
     * @param startTimeStamp    end timestamp for search
     * @param maxCount          the max number of message to search
     * @return                  the list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(long startTimeStamp, long endTimeStamp, int maxCount) {
        List<EMAMessage> msgs = emaObject.searchMessages(startTimeStamp, endTimeStamp, maxCount);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据msgid获取消息
     * 
     * @param  messageId    需要获取的消息id
     * @param  markAsRead   是否获取消息的同时标记消息为已读
     * @return              获取到的message实例
     *
     * \~english
     * get message by message ID, if the message already loaded into memory cache, will directly return the message, otherwise load message from database, and put it into cache.
     *
     * @param  messageId
     * @param  markAsRead    if true, will mark the message as read and send read ack to server
     * @return message
     */
    public EMMessage getMessage(String messageId, boolean markAsRead) {
        EMMessage msg = getCache().getMessage(messageId);
        if (msg == null) {
            EMAMessage emaMsg = emaObject.loadMessage(messageId);
            if (emaMsg == null) {
                return null;
            }
            msg = new EMMessage(emaMsg);
        }
        emaObject.markMessageAsRead(messageId, markAsRead);
        return msg;
    }

    /**
     * \~chinese
     * 加载一组消息，如果缓存不存在会去DB查询并加载
     *
     * @param msgIds 一组消息ID
     * @return 返回一组消息如果找到，否者返回null
     *
     * \~english
     * load messages, if those message not exists in memory in cache, will load message from database.
     *
     * @param  msgIds   messages to be loaded.
     * @return
     */
    @Deprecated
    public List<EMMessage> loadMessages(List<String> msgIds) {

        List<EMMessage> msgs = new ArrayList<EMMessage>();

        for (String msgId : msgIds) {
            EMAMessage msg = emaObject.loadMessage(msgId);
            if (msg != null) {
                msgs.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(msgs);
        return msgs;
    }

    /**
     * \~chinese
     * 设置某条消息为已读
     *
     * @param messageId 消息ID
     *
     * \~english
     * mark the message as read, and send read as to server.
     *
     * @param messageId     messageID
     */
    public void markMessageAsRead(String messageId) {
        emaObject.markMessageAsRead(messageId, true);
    }

    /**
     * \~chinese
     * 获取此conversation当前内存所有的message。如果内存中为空，再从db中加载。
     *
     * @return
     *
     * \~english
     * get all message in local cache, if empty, then get from local database.
     *
     */
    public List<EMMessage> getAllMessages() {
        if (getCache().isEmpty()) {
            EMAMessage lastMsg = emaObject.latestMessage();
            List<EMMessage> msgs = new ArrayList<EMMessage>();
            if (lastMsg != null) {
                msgs.add(new EMMessage(lastMsg));
            }
            getCache().addMessages(msgs);
        }
        return getCache().getAllMessages();
    }

    /**
     * \~chinese
     * 删除一条指定的消息
     *
     * @param messageId     待删除消息的ID
     *
     * \~english
     * delete a message
     *
     * @param messageId     message to be deleted
     */
    public void removeMessage(String messageId) {
        EMLog.d(TAG, "remove msg from conversation: " + messageId);
        emaObject._removeMessage(messageId);
        getCache().removeMessage(messageId);
    }

    /**
     * \~chinese
     * 获取队列中的最后一条消息 （此操作不会改变未读消息计数）
     *
     * @return
     *
     * \~english
     * get last message from conversation
     */
    public EMMessage getLastMessage() {
        if (getCache().isEmpty()) {
            EMAMessage _msg = emaObject.latestMessage();
            EMMessage msg = _msg == null ? null : new EMMessage(_msg);
            getCache().addMessage(msg);
            return msg;
        } else {
            return getCache().getLastMessage();
        }
    }

    /**
     * \~chinese
     * 获取会话接收到的最后一条消息
     *
     * \~english
     * Get received latest message from conversation.
     */
    public EMMessage getLatestMessageFromOthers() {
        EMAMessage _msg = emaObject.latestMessageFromOthers();
        EMMessage msg = _msg == null ? null : new EMMessage(_msg);
        getCache().addMessage(msg);
        return msg;
    }

    /**
     * \~chinese
     * 清除对话中的所有消息，只清除内存的，不清除db的消息
     * 在退出会话的时候清除内存缓存，减小内存消耗
     *
     * \~english
     * clear messages in this conversation's from cache, but will NOT clear local database
     */
    public void clear() {
        getCache().clear();
    }

    /**
     * \~chinese
     * 删除该会话所有消息，同时清除内存和数据库中的消息
     *
     * \~english
     * Delete all messages of the conversation from memory cache and local database
     */
    public void clearAllMessages() {
        emaObject.clearAllMessages();
        getCache().clear();
    }

    /**
     * \~chinese
     * 用户可以自行定义会话扩展字段，该字段只保存在本地，不进行网络同步
     *
     * @param ext       会话对应扩展字段的内容
     *
     * \~english
     * set Conversation extension. extend field only stored in local database, not sync to network server.
     *
     * @param ext       extension string
     */
    public void setExtField(String ext) {
        emaObject._setExtField(ext);
    }

    /**
     * \~chinese
     * 获取用户可以自行定义会话扩展字段
     * 该字段只保存在本地，不进行网络同步
     *
     * @return      会话对应扩展字段的内容
     *
     * \~english
     * get conversation's extend field, extend field only stored in local database, not sync to network server.
     *
     * @return extension object
     */
    public String getExtField() {
        return emaObject.extField();
    }

    /**
     * \~chinese
     * 从消息类型到会话类型的转化
     *
     * @param       id   消息Id，用来区分客服和单聊，对于其他类型，这个参数没有影响。
     * @param       type 消息类型
     * @return      会话类型
     *
     * \~english
     * provide transformation from message type to conversation type
     *
     * @param       id   message Id, used to distinguish single chat and help desk, has no effect on other chat type.
     * @param       type message type
     * @return      conversation type
     */
    public static EMConversationType msgType2ConversationType(String id, EMMessage.ChatType type) {
        if (type == ChatType.Chat) {
//			if(EMCustomerService.getInstance().isCustomServiceAgent(id)){
//				return EMConversationType.HelpDesk;
//			}
            return EMConversationType.Chat;
        } else if (type == ChatType.GroupChat) {
            return EMConversationType.GroupChat;
        } else if (type == ChatType.ChatRoom) {
            return EMConversationType.ChatRoom;
        }

        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 群组和聊天室类型都会返回true
     *
     * @return 群组和聊天室类型都会返回true, 其他类型返回false.
     *
     * \~english
     * group chat and chatroom chat will both return true
     *
     * @return group chat and chatroom chat will both return true, otherwise return false.
     */
    public boolean isGroup() {
        EMConversationType type = getType();
        return type == EMConversationType.GroupChat ||
                type == EMConversationType.ChatRoom;
    }

    /**
     *  \~chinese
     *  插入一条消息，消息的conversationId应该和会话的conversationId一致，消息会被插入DB，并且更新会话的latestMessage等属性
     *
     *  @param msg 消息实例
     *
     *  \~english
     *  Insert a message to a conversation in local database.
     *  ConversationId of the message should be the same as conversationId of the conversation in order to insert the message into the conversation correctly.
     *  The inserting message will be inserted based on timestamp.
     *
     *  @param msg Message
     */
    public void insertMessage(EMMessage msg) {
        emaObject.insertMessage(msg.emaObject);
        getCache().addMessage(msg);
    }

    /**
     *  \~chinese
     *  插入一条消息到会话尾部，消息的conversationId应该和会话的conversationId一致，消息会被插入DB，并且更新会话的latestMessage等属性
     *
     *  @param msg 消息实例
     *
     *  \~english
     *  Insert a message to the end of a conversation in local database.
     *  ConversationId of the message should be the same as conversationId of the conversation in order to insert the message into the conversation correctly.
     *
     *  @param msg Message
     */
	public void appendMessage(EMMessage msg) {
	    emaObject.appendMessage(msg.emaObject);
	    getCache().addMessage(msg);
	}

    /**
     *  \~chinese
     *  更新本地的消息，不能更新消息ID，消息更新后，会话的latestMessage等属性进行相应更新
     *
     *  @param msg 要更新的消息
     *
     *  \~english
     *  Update a message in local database. latestMessage of the conversation and other properties will be updated accordingly. messageId of the message cannot be updated
     *
     *  @param msg Message
     */
    public boolean updateMessage(EMMessage msg) {
        return emaObject.updateMessage(msg.emaObject);
    }

    /**
     * \~chinese
     * 返回会话对应的附件存储路径，该方法适用于清理该会话磁盘存储，不确保该路径一定存在，请在删除对应路径前加以判断，并加上异常保护
     *
     * \~english
     * return conversation associated attachment path, can be used to erase conversation related downloaded files from local storage.
     * not ensure the return path exists, please handle IOException when deleting directory
     *
     * @return
     */
    public String getMessageAttachmentPath() {
        String downloadPath = EMClient.getInstance().getChatConfigPrivate().getDownloadPath();
        return downloadPath + "/" + EMClient.getInstance().getCurrentUser()
                + "/" + conversationId();
    }


    // ====================================== Message cache ======================================

    MessageCache getCache() {
        MessageCache cache;
        synchronized (EMClient.getInstance().chatManager().caches) {
            cache = EMClient.getInstance().chatManager().caches.get(emaObject.conversationId());
            if (cache == null) {
                cache = new MessageCache();
            }
            EMClient.getInstance().chatManager().caches.put(emaObject.conversationId(), cache);
        }
        return cache;
    }

    static class MessageCache {

        TreeMap<Long, EMMessage> sortedMessages = new TreeMap<Long, EMMessage>(new MessageComparator());
        Map<String, EMMessage> messages = new HashMap<String, EMMessage>();
        Map<String, Long> idTimeMap = new HashMap<String, Long>();

        final boolean sortByServerTime = EMClient.getInstance().getChatConfigPrivate().getOptions().isSortMessageByServerTime();

        class MessageComparator implements Comparator<Long> {

            @Override
            public int compare(Long time0, Long time1) {
                long val = time0 - time1;
                if (val > 0) {
                    return 1;
                } else if (val == 0) {
                    return 0;
                } else {
                    return -1;
                }
            }
        }

        public synchronized EMMessage getMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return null;
            }
            return messages.get(msgId);
        }

        public synchronized void addMessages(List<EMMessage> msgs) {
            for (EMMessage msg : msgs) {
                addMessage(msg);
            }
        }

        public synchronized void addMessage(EMMessage msg) {
            if (msg == null || msg.emaObject == null || msg.getMsgTime() == 0 || msg.getMsgTime() == -1 || msg.getMsgId() == null
                    || msg.getMsgId().isEmpty() || msg.getType() == Type.CMD) {
                return;
            }
            String id = msg.getMsgId();
            long time = sortByServerTime ? msg.getMsgTime() : msg.localTime();
            sortedMessages.put(time, msg);
            messages.put(id, msg);
            idTimeMap.put(id, time);
        }

        public synchronized void removeMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            EMMessage msg = messages.get(msgId);
            if (msg != null) {
                Long time = idTimeMap.get(msgId);
                if (time != null) {
                    sortedMessages.remove(time);
                    idTimeMap.remove(msgId);
                }
                messages.remove(msgId);
            }
        }

        public synchronized List<EMMessage> getAllMessages() {
            List<EMMessage> list = new ArrayList<EMMessage>();
            list.addAll(sortedMessages.values());
            return list;
        }

        public synchronized EMMessage getLastMessage() {
            if (sortedMessages.isEmpty()) {
                return null;
            }
            return sortedMessages.lastEntry().getValue();
        }

        public synchronized void clear() {
            //no need to keep the last message
            sortedMessages.clear();
            messages.clear();
            idTimeMap.clear();
        }

        public synchronized boolean isEmpty() {
            return sortedMessages.isEmpty();
        }
    }
}
