/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2015 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.graphics.BitmapFactory.Options;
import android.util.Log;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConversationListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMessageListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.EMConversation.EMConversationType;
import com.hyphenate.chat.EMMessage.Status;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAChatManager;
import com.hyphenate.chat.adapter.EMAChatManagerListener;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.cloud.EMCloudOperationCallback;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.ImageUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import static com.hyphenate.chat.EMMessage.self;

/**
 * send a message
 * EMMessage msg = EMMessage;
 * EMClient.getInstance().chatManager().sendMessage(msg);
 */
public class EMChatManager {

	EMAChatManager emaObject;

	private static final String TAG = "EMChatManager";

	private final static String INTERNAL_ACTION_PREFIX = "em_";

	EMClient mClient;

	Map<String, EMConversation.MessageCache> caches = new Hashtable<String, EMConversation.MessageCache>();

	protected EMChatManager(){}

	protected EMChatManager(EMClient client, EMAChatManager manager) {
		mClient = client;
		emaObject = manager;

		emaObject.addListener(chatManagerListenerImpl);
	}

	private List<EMMessageListener> messageListeners = Collections.synchronizedList(new ArrayList<EMMessageListener>());
	private List<EMConversationListener> conversationListeners = Collections.synchronizedList(new ArrayList<EMConversationListener>());

	EMAChatManagerListener chatManagerListenerImpl = new EMAChatManagerListener() {

        List<EMMessageListener> cloneSyncedList(List<EMMessageListener> list) {
            if (list == null) {
                return new ArrayList<>();
            } else {
                synchronized (list) {
                    return list.subList(0, list.size());
                }
            }
        }

		@Override
		public void onReceiveMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

					List<EMMessage> remainingMsgs = new ArrayList<EMMessage>();
					for (EMMessage msg : msgs) {
				    	/*if(msg.getChatType() == EMMessage.ChatType.ChatRoom){
                            EMChatRoom room = EMClient.getInstance().chatroomManager().getChatRoom(msg.conversationId());
                            if(room == null){
                            	continue;
                            }
				    	}*/

	                    EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getFrom(), msg.getChatType()), false);
	                    if(conv == null){
	                    	continue;
	                    }
	                    
	                    // CMD not put into cache
	                    if (msg.getType() != Type.CMD) {
	                        conv.getCache().addMessage(msg);
	                    }
	                    remainingMsgs.add(msg);
				    }
				    
				    if(remainingMsgs.size() <= 0){
				    	return;
				    }

                    try {
                        for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                            l.onMessageReceived(remainingMsgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveCmdMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						String action = ((EMCmdMessageBody)message.getBody()).action();
						if(isAdvanceDebugMessage(action)) {
							EMAdvanceDebugManager.getInstance().handleDebugMessage(message, EMAdvanceDebugManager.Type.valueOf(action));
						} else {
							msgs.add(message);
						}
					}
                    try {
                        for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                            l.onCmdMessageReceived(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

				}
			});
		}

		@Override
		public void onMessageStatusChanged(final EMAMessage message, final EMAError error) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    try {
                        EMMessage msg = new EMMessage(message);
                        for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                            l.onMessageChanged(msg, null);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
	    public void onMessageAttachmentsStatusChanged(final EMAMessage message, final EMAError error) {
            try {
                EMMessage msg = new EMMessage(message);
                for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                    l.onMessageChanged(msg, null);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
		}
		
		
        @Override
        public void onReceiveRecallMessages(final List<EMAMessage> messages) {
            mClient.executeOnMainQueue(new Runnable() {

                @Override
                public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                        getConversation(msg.conversationId()).getCache().removeMessage(msg.msgId());
                    }

                    try {
                        for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                            l.onMessageRecalled(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
		@Override
		public void onReceiveHasReadAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                    }

                    try {
                        for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                            l.onMessageRead(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveHasDeliveredAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

                    try {
                        for (EMMessageListener l : cloneSyncedList(messageListeners)) {
                            l.onMessageDelivered(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onUpdateConversationList(final List<EMAConversation> conversations) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateConversationList");
                    synchronized (conversationListeners) {
	                    try {
	                        for (EMConversationListener l : conversationListeners) {
	                            l.onCoversationUpdate();
	                        }
	                    } catch (Exception e) {
		                    e.printStackTrace();
	                    }
                    }
				}
			});
		}

	};

	private boolean isAdvanceDebugMessage(String action){
		if(action.startsWith(INTERNAL_ACTION_PREFIX)){
			try {
				EMAdvanceDebugManager.Type.valueOf(action);
				return true;
			} catch (Exception e) {
				//if it is not a field of Type, throws exception
				e.printStackTrace();
			}
		}
		return false;
	}

	/**
	 * \~chinese
	 * 异步发送消息 如果是语音，图片类有附件的消息，sdk 会自动上传附件
	 *
	 * @param msg					待发送消息对象
	 *
	 * \~english
	 * send message asynchronously and you can set this message status callback through EMMessage.setMessageStatusCallback
	 *
	 * @param msg
	 */
	public void sendMessage(final EMMessage msg) {
		msg.makeCallbackStrong();

		boolean createConv = msg.getType() != Type.CMD;
		final EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getTo(), msg.getChatType()), createConv);

		//add the message at first
		if (conv != null) {
			boolean exists = conv.getCache().getMessage(msg.getMsgId()) != null;
			if (!exists) {
				// sent message time is lastMsgTime + 1ms
				long lastMsgTime = System.currentTimeMillis();
				EMMessage lastMsg = conv.getLastMessage();
				if (lastMsg != null) {
					lastMsgTime = lastMsgTime < lastMsg.getMsgTime() ? lastMsg.getMsgTime() : lastMsgTime;
				}
				msg.setMsgTime(lastMsgTime + 1);

				conv.getCache().addMessage(msg);
			}
		}

		class HandleError {
			HandleError(int code, String desc) {
				EMMessage.EMCallbackHolder holder = msg.messageStatusCallBack;
				if (holder != null) {
					holder.onError(code, desc);
				}
			}
		}

    mClient.executeOnSendQueue(new Runnable() {
            
        @Override
        public void run() {
				try {
					String originPath = null;
					// If message body is image, check scale request, set size, and file name
					if (msg.getType() == Type.IMAGE) {
						//the default status is fail status,
						//which lead to message show fail and then show success in need scale image
						msg.setStatus(Status.INPROGRESS);
						EMImageMessageBody body = (EMImageMessageBody) msg.getBody();
						if (body == null) {
							new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
							return;
						}

						String filePath = body.getLocalUrl();
						File file = new File(filePath);
						if (!file.exists() || !file.canRead()) {
							new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
							return;
						}
						if (!body.isSendOriginalImage()) {
							String scaledImagePath = ImageUtils.getScaledImage(mClient.getContext(), filePath);
							if (!scaledImagePath.equals(filePath)) {
								originPath = filePath;
								File attachment = new File(scaledImagePath);
								long originalSize = new File(filePath).length();
								long scaledSize = attachment.length();
								if (originalSize == 0) {
									EMLog.d(TAG, "original image size:" + originalSize);
									new HandleError(EMError.FILE_INVALID, "original image size is 0");
									return;
								}
								EMLog.d(TAG, "original image size:" + originalSize + " scaled image size:" + scaledSize
										+ " ratio:" + (int) (scaledSize / originalSize) + "%");
								filePath = scaledImagePath;
								body.setLocalUrl(filePath);
							}
						}
						// get image width and height
						Options options = ImageUtils.getBitmapOptions(filePath);
						int width = options.outWidth;
						int height = options.outHeight;
						body.setSize(width, height);
						body.setFileName(new File(filePath).getName());
					}else if (msg.getType() == Type.VIDEO){
						msg.setStatus(Status.INPROGRESS);
						EMVideoMessageBody body = (EMVideoMessageBody) msg.getBody();
						if (body == null) {
							new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
							return;
						}
						String filePath = body.getLocalUrl();
						File file = new File(filePath);
						if (!file.exists() || !file.canRead()) {
							new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
							return;
						}
						// get video width and height
						Options options = ImageUtils.getBitmapOptions(body.getLocalThumb());
						int width = options.outWidth;
						int height = options.outHeight;
						body.setThumbnailSize(width, height);
					}

					String oldId = msg.getMsgId();
					//set callback to replace old id
					setMessageSendCallback(msg, conv, oldId, originPath);
					emaObject.sendMessage(msg.emaObject);
				} catch (Exception e) {
					e.printStackTrace();
					new HandleError(EMError.GENERAL_ERROR, "send message failed");
				}
			}
		});
	}

	private void setMessageSendCallback(final EMMessage msg,
										final EMConversation conv,
										final String oldId,
										final String originImagePath) {
		if (msg == null) {
			return;
		}

		msg.setInnerCallback(new EMCallBack(){

			@Override
			public void onSuccess() {
				if (conv != null) {
					conv.getCache().removeMessage(oldId);
					conv.getCache().addMessage(msg);
				}
				if (originImagePath != null){
					if (msg.getBody() instanceof EMImageMessageBody) {
						String scaleImagePath = ((EMImageMessageBody)msg.getBody()).getLocalUrl();
						EMLog.d(TAG, "origin: + " + originImagePath + ", scale:" + scaleImagePath);
						if(scaleImagePath != null && !scaleImagePath.equals(originImagePath)){
							File scaleFile = new File(scaleImagePath);
							if(scaleFile.exists()){
								scaleFile.delete();
							}
						}
						((EMImageMessageBody)msg.getBody()).setLocalUrl(originImagePath);
						updateMessage(msg);
					}
				}
			}

			@Override
			public void onProgress(int progress, String status) {
			}

			@Override
			public void onError(int code, String error) {
			}
		});
	}

	/**
	 * \~chinese
	 * 发送消息已读回执
	 *
	 * @param to			接收方的用户名
	 * @param messageId		消息的ID
	 * @throws HyphenateException
	 *
	 * \~english
	 * send read ack to server
	 * @param to			the recipient id
	 * @param messageId		message id
	 * @throws HyphenateException
	 */
	public void ackMessageRead(String to, String messageId) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			emaObject.sendReadAckForMessage(msg);
		} else { // just for Xinju since there is no local storage
			EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, EMMessage.ChatType.Chat.ordinal());
			_msg.setMsgId(messageId);
			_msg.setFrom(to);

			emaObject.sendReadAckForMessage(_msg);
		}
	}
	
    /**
     * \~chinese
     * 撤回发送成功的消息对象
     *
     * @param message 消息对象
     * @param message message
     * @return
     */
    public void recallMessage(EMMessage message) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.recallMessage(message.emaObject, error);
        handleError(error);
        getConversation(message.getTo()).getCache().removeMessage(message.getMsgId());
    }

    public void aysncRecallMessage(final EMMessage message,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    recallMessage(message);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 获取指定ID 的消息对象
	 *
	 * @param messageId		消息ID
	 * @return
	 *
	 * \~english
	 * get message through message id
	 *
	 * @param messageId 	message id
	 * @return
	 */
	public EMMessage getMessage(String messageId) {
		synchronized (caches) {
			for(EMConversation.MessageCache cache : caches.values()) {
				EMMessage msg = cache.getMessage(messageId);
				if (msg != null) {
					return msg;
				}
			}
		}
		EMAMessage message =  emaObject.getMessage(messageId);
		if (message == null) {
			return null;
		}
		EMMessage msg = new EMMessage(message);
		return msg;
	}


	/**
	 * \~chinese
	 * 获取会话，没有则返回null, 没找到则返回空
	 * @param id user id or group id
	 * @return 会话
	 *
	 * \~english
	 * get conversation by id
	 *
	 * @param id user id, group id or chatroom id
	 * @return EMConversation the existing conversation found by conversation, null if not found
	 */
	public EMConversation getConversation(String id){
		EMAConversation conversation = emaObject.conversationWithType(id, EMAConversationType.CHAT, false);
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.DISCUSSIONGROUP, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.HELPDESK, false);
		}

		return conversation == null ? null : new EMConversation(conversation);
	}

	/**
	 * \~chinese
	 * 根据用户或群组id以及会话类型获取会话，没有找到则返回空
	 *
	 * @param id 		用户或群组id
	 * @param type 		会话类型
	 * @return          会话
	 *
	 * \~english
	 * get conversation by conversation id and conversation type, create a empty conversation if not found
	 *
	 * @param id 			user id, group id or chatroom id
	 * @param type @see  	EMConversationType the conversation type
	 * @return 				EMConversation the conversation found according to the id and type, create a empty conversation if not found
	 */
	public EMConversation getConversation(String id, EMConversationType type) {
		return getConversation(id, type, false);
	}

	/**
	 * \~chinese
	 * 根据用户或群组id以及会话类型获取会话
	 *
	 * @param username 用户或群组id
	 * @param type 会话类型
	 * @param createIfNotExists 没找到相应会话时是否自动创建
	 * @return
	 *
	 * \~english
	 * get conversation by conversation id and conversation type
	 *
	 * @param username user id, group id or chatroom id
	 * @param type @see  EMConversationType the conversation type
	 * @param createIfNotExists create conversation if not exists
	 * @return
	 */
	public EMConversation getConversation(String username, EMConversationType type, boolean createIfNotExists) {
		EMAConversationType t = EMAConversationType.CHAT;
		if (type == EMConversationType.Chat) {
			t = EMAConversationType.CHAT;
		} else if (type == EMConversationType.GroupChat) {
			t = EMAConversationType.GROUPCHAT;
		} else if (type == EMConversationType.ChatRoom) {
			t = EMAConversationType.CHATROOM;
		} else if (type == EMConversationType.DiscussionGroup) {
			t = EMAConversationType.DISCUSSIONGROUP;
		} else if (type == EMConversationType.HelpDesk) {
			t = EMAConversationType.HELPDESK;
		}
		EMAConversation conversation = emaObject.conversationWithType(username, t, createIfNotExists);
		if (conversation == null) {
			return null;
		}
		Log.d(TAG, "convID:" + conversation.conversationId());
		return new EMConversation(conversation);
	}


	/**
	 * \~chinese
	 * 把所有的会话都设成已读
	 *
	 * \~english
	 * mark all messages as read
	 */
	public void markAllConversationsAsRead() {
		List<EMAConversation> conversations = emaObject.loadAllConversationsFromDB();
		for (EMAConversation conversation : conversations) {
			conversation.markAllMessagesAsRead(true);
		}
	}

	/**
	 * \~chinese
	 * 获取未读消息计数
	 * @deprecated  使用 {@link EMChatManager#getUnreadMessageCount()} 替代
	 * @return
	 *
	 * \~english
	 * get unread message count
	 * @deprecated  use {@link EMChatManager#getUnreadMessageCount()} instead
	 * @return
	 */
	@Deprecated
	public int getUnreadMsgsCount() {
		return getUnreadMessageCount();
	}

	/**
	 * \~chinese
	 * 获取未读消息计数
	 * @return
	 *
	 * \~english
	 * get unread message count
	 * @return
	 */
	public int getUnreadMessageCount()
	{
		List<EMAConversation> conversations = emaObject.getConversations();
		int unreadCount = 0;
		for (EMAConversation conversation : conversations) {
			if (conversation._getType() != EMAConversationType.CHATROOM) {
				unreadCount += conversation.unreadMessagesCount();
			}
		}
		return unreadCount;
	}

	/**
	 * \~chinese
	 * 保存用户app 生成的消息，比如系统提示 消息会存到内存中的conversation 和数据库
	 * CMD类型数据不保存在本地
	 *
	 * @param message	待存储的消息
	 *
	 * \~english
	 * save the message to memory and local database
	 * CMD type message to will stored in database
	 *
	 * @param message
	 */
	public void saveMessage(EMMessage message) {
		EMMessage.ChatType type = message.getChatType();
		EMConversationType t = EMConversationType.Chat;
		switch (type) {
			case Chat:
				t = EMConversationType.Chat;
				break;
			case GroupChat:
				t = EMConversationType.GroupChat;
				break;
			case ChatRoom:
				t = EMConversationType.ChatRoom;
				break;
		}
		String convId = message.getTo();
		//for group, chatroom, conversation id is group id for both receive and send message
		if (t == EMConversationType.Chat && message.direct() == EMMessage.Direct.RECEIVE) {
			convId = message.getFrom();
		}
		if (message.getType() == Type.CMD) {
			return;
		}
		EMConversation conv = getConversation(convId, t, true);
		// when send message out, appendMessage will update time to lastMsgTime + 1ms
		conv.insertMessage(message);
	}

	/**
	 * \~chinese
	 * 更新消息，消息的内容会被保存到本地
	 *
	 * @param message
	 *
	 * \~english
	 * update the message
	 *
	 * @param message
	 */
	public boolean updateMessage(EMMessage message) {
		String id = message.direct() == EMMessage.Direct.RECEIVE ? message.getFrom() : message.getTo();
		if (message.getType() == Type.CMD) {
			return false;
		}
		EMConversation conv = getConversation(message.conversationId(), EMConversation.msgType2ConversationType(id, message.getChatType()), true);
		return conv.updateMessage(message);
	}

	/**
	 * \~chinese
	 * 下载消息的附件，未成功下载的附件，可调用此方法再次下载
	 *
	 * @param msg
	 *
	 * \~english
	 * download the message attachement
	 *
	 * @param msg 	message to be downloaded
	 */
	public void downloadAttachment(final EMMessage msg) {
        if (msg == null) {
            return;
        }
        msg.makeCallbackStrong();
		emaObject.downloadMessageAttachments(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 下载消息的缩略图
	 *
	 * @param msg
	 *
	 * \~english
	 * download the thumbnail if not downloaded before
	 *
	 * @param msg  	the message to be downloaded
	 */
	public void downloadThumbnail(final EMMessage msg) {
        msg.makeCallbackStrong();
		emaObject.downloadMessageThumbnail(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 向消息数据库导入多条聊天记录
	 * 在调用次函数时要保证，消息的发送方或者接收方是当前用户
	 * 已经对函数做过速度优化， 推荐一次导入1000条数据
	 *
	 * @param msgs 需要导入数据库的消息
	 *
	 * \~english
	 * import messages to local database. Make sure the meessage's sender or receiver is current user
	 * Recommend import up to 1000 messages per operation
	 *
	 * @param msgs the message list of be imported
	 */
	public synchronized void importMessages(List<EMMessage> msgs) {
		List<EMAMessage> dummy = new ArrayList<EMAMessage>();
		for (EMMessage msg : msgs) {
			dummy.add(msg.emaObject);
		}
		EMClient.getInstance().getChatConfigPrivate().importMessages(dummy);
	}

	/**
	 * \~chinese
	 * 获取某个类型所有的会话
	 *
	 * @return 会话list
	 *
	 * \~english
	 * get list of conversations by conversation type
	 *
	 * @param type	converation type
	 * @return 		list of conversation in specified type
	 */
	public List<EMConversation> getConversationsByType(EMConversationType type) {
		List<EMAConversation> conversations = emaObject.getConversations();
		List<EMConversation> result = new ArrayList<EMConversation>();
		for (EMAConversation conv : conversations) {
			if (type.ordinal() == conv._getType().ordinal()) {
				result.add(new EMConversation(conv));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器下载文件，推荐使用downloadAttachment(EMMessage msg)替代此函数
	 *
	 * @param remoteUrl 服务器上的远程文件
	 * @param localFilePath 本地要生成的文件
	 * @param headers Http Request Header
	 * @param callback EMCallBack
	 *
	 * \~english
	 * download the file from server. Recommend using downloadAttachment(EMMessage msg) instead of this function.
	 *
	 * @param remoteUrl 		the remote file url
	 * @param localFilePath 	the local file path
	 * @param headers 			Http Request Headers
	 * @param callback 			EMCallBack
	 */
	@Deprecated
	public void downloadFile(final String remoteUrl,
							 final String localFilePath,
							 final Map<String, String> headers,
							 final EMCallBack callback) {
		/*
		if (!headers.containsKey("Authorization")) {
			headers.put("Authorization", EMChatConfig.getInstance().getAccessToken());
		}
		int responseCode = EMARHttpAPI.download(remoteUrl, localFilePath, headers);

		if (200 <= responseCode && responseCode <= 299) {
			if (callback != null) {
				callback.onSuccess();
			}
		} else {
			callback.onError(responseCode, "");
		}
		*/

		EMHttpClient.getInstance().downloadFile(remoteUrl, localFilePath, headers, new EMCloudOperationCallback(){

			@Override
			public void onSuccess(String result) {
				if(callback != null){
					callback.onSuccess();
				}
			}

			@Override
			public void onError(String msg) {
				if(callback != null){
					callback.onError(EMError.GENERAL_ERROR, msg);
				}
			}

			@Override
			public void onProgress(int progress) {
				if(callback != null){
					callback.onProgress(progress, null);
				}
			}

		});
	}

	/**
	 * \~chinese
	 * 获取当前所有的会话
	 *
	 * @return
	 *
	 * \~english
	 * get all conversations in the cache
	 *
	 * @return conversations which is a map with key&#45;&#62;conversation id : value&#45;&#62;EMConversation
	 */
	public Map<String, EMConversation> getAllConversations() {
		List<EMAConversation> conversations = emaObject.getConversations();
		Hashtable<String, EMConversation> result = new Hashtable<String, EMConversation>();
		for (EMAConversation conversation : conversations) {
			/*if (conversation._getType() != EMAConversationType.CHATROOM) {
				result.put(conversation.conversationId(), new EMConversation(conversation));
			}*/
			result.put(conversation.conversationId(), new EMConversation(conversation));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 同步加载所有的会话，并且每条会话读入EMChatOptions.getNumberOfMessagesLoaded()条消息, 默认是20条以保持兼容
	 *
	 * \~english
	 * load all conversatons from local database
	 */
	public void loadAllConversations() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 删除和指定用户或者群聊的对话(包括删除本地的聊天记录)
	 *
	 * @param username 			用户名或者群聊id
	 * @param deleteMessages	是否删除消息
	 * @return 					删除失败或者不存在此user的conversation返回false
	 *
	 * \~english
	 * delete conversation and messages from local database
	 *
	 * @param username 			user id or group id
	 * @param deleteMessages 	whether delete messages
	 * @return 					return true if delete successfully
	 */
	public boolean deleteConversation(String username, boolean deleteMessages) {
		EMConversation conv = getConversation(username);
		if (conv == null) {
			return false;
		}
		if (!deleteMessages) {
			conv.clear();
		} else {
			conv.clearAllMessages();
		}
		emaObject.removeConversation(username, deleteMessages);
		return true;
	}

	/**
	 * \~chinese
	 * 注册消息监听
	 * @param listener
	 *
	 * \~english
	 * add message listener @see EMMessageListener
	 *
	 * @param listener the message listener which is used to monitor the incoming messages
	 */
	public void addMessageListener(EMMessageListener listener) {
		if(listener == null){
			return;
		}

		if(!messageListeners.contains(listener)){
			messageListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除消息监听
	 *
	 * @param listener 要移除的监听
	 *
	 * \~english
	 * remove the message listener
	 *
	 * @param listener the message listener set before
	 */
	public void removeMessageListener(EMMessageListener listener) {
		if(listener == null){
			return;
		}

		messageListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 注册会话监听
	 *
	 * @param listener
	 *
	 * \~english
	 * add conversation listener @see EMConversationListener
	 * refresh the conversation list when receiving the conversation refresh event
	 *
	 * @param listener the conversation listener
	 */
	public void addConversationListener(EMConversationListener listener){
		if(!conversationListeners.contains(listener)){
			conversationListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除会话监听
	 *
	 * @param listener
	 *
	 * \~english
	 * remove the conversation listener
	 *
	 * @param listener the conversation listener set before
	 */
	public void removeConversationListener(EMConversationListener listener){
		if(listener == null){
			return;
		}

		conversationListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 设置消息为已听，一般用于语音消息
	 *
	 * @deprecated  使用{@link EMChatManager#setVoiceMessageListened(EMMessage)}替代
	 *
	 * \~english
	 * set message to listened, used for voice message
	 *
	 * @deprecated  use {@link EMChatManager#setVoiceMessageListened(EMMessage)} instead
	 */
	@Deprecated
	public void setMessageListened(EMMessage message) {
		setVoiceMessageListened(message);
	}

	/**
	 * \~chinese
	 * 设置消息为已听，一般用于语音消息
	 *
	 * \~english
	 * set message to listened, used for voice message
	 */
	public void setVoiceMessageListened(EMMessage message)
	{
		message.setListened(true);
		updateMessage(message);
	}

	void onLogout() {
		caches.clear();
	}

	synchronized void  loadAllConversationsFromDB() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 将数据库中的某个联系人相关信息变更成另外一个联系人
	 * 与变更相关的表单包含消息表单，会话表单，联系人表单，黑名单表单
	 * 注意：该操作不会更新内存中数据
	 *
	 * @param from
	 * @param changeTo
	 * @return 返回更新结果，任何表单更新失败，都会返回false
	 *
	 * \~english
	 * update database records, change username 'from' to 'changeTo', take effect on message table, conversation table, contacts, blacklist table
	 * note: this operation does not update data stored in memory cache.
	 *
	 * @param from
	 * @param changeTo
	 * @return operation result, if any update operations on those tables failed, result is false
	 */
	public boolean updateParticipant(String from, String changeTo) {
		return emaObject.updateParticipant(from, changeTo);
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息
	 * @param conversationId 会话名称
	 * @param type 会话类型
	 * @param pageSize 获取的页面大小
	 * @param startMsgId 漫游消息的开始消息id，如果为空，从最新的消息向前开始获取
	 * @return 返回消息列表和用于继续获取历史消息的Cursor
	 *
	 * \~english
	 * fetch conversation roam messages from server.
	 * @param conversationId the conversation id which select to fetch roam message.
	 * @param type the conversation type which select to fetch roam message.
	 * @param pageSize the page size.
	 * @param startMsgId the start search roam message, if empty start from the server lastest message.
	 * @return server return messages and cursor for next fetch action.
	 */
	public EMCursorResult<EMMessage> fetchHistoryMessages(String conversationId, EMConversationType type, int pageSize,
	                                 String startMsgId) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAMessage> _cursorResult = emaObject.fetchHistoryMessages(conversationId,
				type.ordinal(), pageSize, startMsgId, error);
		handleError(error);
		EMCursorResult<EMMessage> cursorResult = new EMCursorResult<>();
		cursorResult.setCursor(_cursorResult.getCursor());
		List<EMMessage> msgs = new ArrayList<>();
		for (EMAMessage _msg : _cursorResult.getData()) {
			msgs.add(new EMMessage(_msg));
		}
		cursorResult.setData(msgs);
		return cursorResult;
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息
	 * @param conversationId 会话名称
	 * @param type 会话类型
	 * @param pageSize 获取的页面大小
	 * @param startMsgId 漫游消息的开始消息id，如果为空，从最新的消息向前开始获取
	 * @param callBack 返回消息列表和用于继续获取历史消息的Cursor
	 *
	 * \~english
	 * fetch conversation roam messages from server.
	 * @param conversationId the conversation id which select to fetch roam message.
	 * @param type the conversation type which select to fetch roam message.
	 * @param pageSize the page size.
	 * @param startMsgId the start search roam message, if empty start from the server lastest message.
	 * @param callBack server return messages and cursor for next fetch action.
	 */
	public void asyncFetchHistoryMessage(final String conversationId, final EMConversationType type, final int pageSize,
	                                final String startMsgId, final EMValueCallBack<EMCursorResult<EMMessage>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private void handleError(EMAError error) throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}

}
