package com.hyphenate.chat.core;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;
import android.util.Pair;

import com.hyphenate.chat.EMClient;
import com.hyphenate.chat.EMConversation;
import com.hyphenate.chat.EMConversation.EMConversationType;
import com.hyphenate.chat.EMMessage;
import com.hyphenate.chat.EMMessage.ChatType;
import com.hyphenate.chat.EMMessage.Direct;
import com.hyphenate.chat.MessageEncoder;
import com.hyphenate.chat.adapter.EMAGroupSetting;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

@SuppressWarnings("unused")
public class EMDBManager {
	private static String TAG = "EMDBManager";
	public static final String DATABASE_NAME = "_emmsg.db";
	private static final int DATABASE_VERSION = 13;

	private static final String CHAT_TABLE_NAME = "chat";
	private static final String COLUMN_ID = "_id";
	private static final String COLUMN_MSG_ID = "msgid";
	private static final String COLUMN_MSG_TIME = "msgtime";
	private static final String COLUMN_MSG_DIR = "msgdir";
	private static final String COLUMN_PARTICIPANT = "participant";
	public static final String COLUMN_MSG_BODY = "msgbody";
	private static final String COLUMN_MSG_GROUP = "groupname";
	private static final String COLUMN_MSG_ISACKED = "isacked";
	private static final String COLUMN_MSG_ISDELIVERED = "isdelivered";
	public static final String COLUMN_MSG_STATUS = "status";
	private static final String COLUMN_MSG_ISLISTENED = "islistened";
	private static final String COLUMN_MSG_TYPE = "msgtype";

	private static final String GROUP_TABLE_NAME = "emgroup";
	private static final String COLUMN_GROUP_NAME = "name";
	private static final String COLUMN_GROUP_NICK = "nick";
	private static final String COLUMN_GROUP_DESC = "desc";
	private static final String COLUMN_GROUP_OWNER = "owner";
	private static final String COLUMN_GROUP_MEMBERS = "members";
	private static final String COLUMN_GROUP_MEMBERS_SIZE = "members_size";
	private static final String COLUMN_MODIFIED_TIME = "modifiedtime";
	private static final String COLUMN_GROUP_JID = "jid";
	private static final String COLUMN_GROUP_PUBLIC = "ispublic";
	private static final String COLUMN_GROUP_IS_BLOCKED = "isblocked";
	private static final String COLUMN_GROUP_MAX_USERS = "max_users";
	
	private static final String CHAT_ROOM_TABLE_NAME = "chatroom";
    private static final String COLUMN_CHAT_ROOM_ID = "name";
    private static final String COLUMN_CHAT_ROOM_NICK = "nick";
    private static final String COLUMN_CHAT_ROOM_DESC = "desc";
    private static final String COLUMN_CHAT_ROOM_OWNER = "owner";
    private static final String COLUMN_CHAT_ROOM_MEMBERS = "members";
    private static final String COLUMN_CHAT_ROOM_MEMBERS_SIZE = "members_size";
    private static final String COLUMN_CHAT_ROOM_IS_BLOCKED = "isblocked";
    private static final String COLUMN_CHAT_ROOM_MAX_USERS = "max_users";

	private static final String UNREAD_TABLE_NAME = "unreadcount";
	private static final String COLUMN_UNREAD_USERNAME = "username";
	private static final String COLUMN_UNREAD_COUNT = "count";
	
	private static final String TOKEN_TABLE_NAME = "token";
	private static final String COLUMN_TOKEN_USERNAME = "username";
	private static final String COLUMN_TOKEN_VALUE = "value";
	private static final String COLUMN_TOKEN_SAVED_TIME = "saved_time";
	
	private static final String CONTACT_TABLE_NAME = "contact";
	private static final String COLUMN_CONTACT_JID = "jid";
	private static final String COLUMN_CONTACT_USERNAME = "username";
	private static final String COLUMN_CONTACT_NICK = "nick";
	
	private static final String BLACK_LIST_TABLE_NAME = "black_list";
	private static final String COLUMN_BLACK_LIST_USERNAME = "username";
	
	private static final String CONVERSATION_LIST_TABLE_NAME = "conversation_list";
	private static final String COLUMN_CONVERSATION_LIST_USERNAME = "username";
	private static final String COLUMN_CONVERSATION_LIST_GROUP = "groupname";
	private static final String COLUMN_CONVERSATION_LIST_EXT = "ext";
	private static final String COLUMN_CONVERSATION_LIST_TYPE = "conversation_type";
	
	private static final String MIGRATE_ONESDK_TABLE_NAME = "porting";
	private static final String COLUMN_MIGRATE_ONESDK_DONE = "done";
	
	private static final String CREATE_CHAT_TABLE = "create table " + CHAT_TABLE_NAME + " (" + COLUMN_ID
			+ " integer primary key autoincrement, " + COLUMN_MSG_ID + " text, " + COLUMN_MSG_TIME + " integer, " + COLUMN_MSG_DIR
			+ " integer, " + COLUMN_MSG_ISACKED + " integer, " + COLUMN_MSG_ISDELIVERED + " integer, " + COLUMN_MSG_STATUS + " integer," + COLUMN_PARTICIPANT + " text not null, "
			+ COLUMN_MSG_ISLISTENED + " integer, "
			+ COLUMN_MSG_BODY + " text not null,"
			+ COLUMN_MSG_TYPE + " integer, "
			+ COLUMN_MSG_GROUP + " text);";

	private static final String CREATE_GROUP_TABLE = "create table " + GROUP_TABLE_NAME + " (" + COLUMN_GROUP_NAME + " text primary key, "
			+ COLUMN_GROUP_JID + " text not null, " + COLUMN_GROUP_NICK + " text not null, " + COLUMN_GROUP_OWNER + " text not null, "
			+ COLUMN_MODIFIED_TIME + " integer, " + COLUMN_GROUP_PUBLIC + " integer, " + COLUMN_GROUP_DESC + " text, "
			+ COLUMN_GROUP_MEMBERS_SIZE + " integer, " + COLUMN_GROUP_IS_BLOCKED + " integer, " + COLUMN_GROUP_MEMBERS + " text, "
			+ COLUMN_GROUP_MAX_USERS + " integer);";
	
	private static final String CREATE_CHAT_ROOM_TABLE = "create table " + CHAT_ROOM_TABLE_NAME + " ("
	        + COLUMN_CHAT_ROOM_ID + " text primary key, "
	        + COLUMN_CHAT_ROOM_NICK + " text, "
	        + COLUMN_CHAT_ROOM_OWNER + " text, "
	        + COLUMN_GROUP_DESC + " text, "
	        + COLUMN_GROUP_MEMBERS_SIZE + " integer, "
	        + COLUMN_GROUP_IS_BLOCKED + " integer, "
	        + COLUMN_GROUP_MEMBERS + " text, "
	        + COLUMN_GROUP_MAX_USERS + " integer);";

	private static final String CREATE_UNREAD_COUNT_TABLE = "create table " + UNREAD_TABLE_NAME + " (" + COLUMN_UNREAD_USERNAME
			+ " text primary key, " + COLUMN_UNREAD_COUNT + " integer);";
	
	private static final String CREATE_TOKEN_TABLE = "create table " + TOKEN_TABLE_NAME + " (" + COLUMN_TOKEN_USERNAME
			+ " text primary key, " + COLUMN_TOKEN_VALUE + " text, " + COLUMN_TOKEN_SAVED_TIME + " integer);";

	private static final String CREATE_CONTACT_TABLE = "create table " +  CONTACT_TABLE_NAME + " ("
            + COLUMN_CONTACT_JID + " text primary key, "
            + COLUMN_CONTACT_USERNAME + " text, " + COLUMN_CONTACT_NICK + " );";
			
	private static final String CREATE_BLACK_LIST_TABLE = "create table " + BLACK_LIST_TABLE_NAME + " ("
			+ COLUMN_BLACK_LIST_USERNAME + " text primary key);"; 
	
	private static final String CREATE_CONVERSATION_LIST_TABLE = "create table if not exists " + CONVERSATION_LIST_TABLE_NAME + " ("
	        + COLUMN_CONVERSATION_LIST_USERNAME + " text primary key, "
	        + COLUMN_CONVERSATION_LIST_GROUP + " text, "
	        + COLUMN_CONVERSATION_LIST_EXT + " text, "
	        + COLUMN_CONVERSATION_LIST_TYPE + " integer);";
	
	private static final String CREATE_MIGRATE_ONESDK_TABLE = "create table if not exists " + MIGRATE_ONESDK_TABLE_NAME + " ("
			+ COLUMN_MIGRATE_ONESDK_DONE + " integer)"; 
			
	private static EMDBManager instance = null;
	private boolean isDBClosed = true;

	/**
	 * current logined username
	 */
	private String currentUserName = null;
	private Context appContext;
	private boolean updateFrom9To10 = false;
	private EMChatConfigPrivate config = null;

	private EMDBManager() {
	}

	public synchronized static void initDB(String username, EMChatConfigPrivate config) {
		EMLog.d(TAG, "initDB : " + username);
		if (instance != null) {
			if(instance.currentUserName != null && instance.currentUserName.equals(username)){
				return;
			}

			instance.closeDatabase();
		}
		
		if(instance == null){
			instance = new EMDBManager();
			instance.config = config;
			instance.appContext = EMClient.getInstance().getContext();
		}

		instance.currentUserName = username;
		instance.isDBClosed = false;
	}

	public synchronized static EMDBManager getInstance() {
		if (instance == null) {
			EMLog.e(TAG, "Please login first!");
			throw new IllegalStateException("Please login first!");
		}
		
		return instance;
	}
	
	public synchronized void closeDatabase() {
		try{
			EMChatDBOpenHelper.closeDB();
			EMLog.d(TAG, "close msg db");
		}catch(Exception e){
			e.printStackTrace();
		}
	}
	
	public void tryMigrationToOneSDK() {
		// Migration check is done in onUpgrade
		SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
		instance.closeDatabase();
	}
	
	/**
	 * 查询会话消息总数
	 */
	public long getConversationMessageCount(String username, boolean isGroup) {
		SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
		try {
			String field = !isGroup ? COLUMN_PARTICIPANT : COLUMN_MSG_GROUP;
			Cursor cursor = db.rawQuery("select count(*) as msgCount from chat where " + field + " = ?", new String[] { username });
			if (!cursor.moveToFirst()) {
				cursor.close();
				return 0;
			}
			long count = cursor.getLong(0);
			cursor.close();
			return count;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return 0;
	}

	private static EMMessage loadMsgFromCursor(Cursor cursor) {
		String jsonString = cursor.getString(cursor.getColumnIndex(COLUMN_MSG_BODY));
		EMMessage msg = MessageEncoder.getMsgFromJson(jsonString);
		
		msg.setMsgId(cursor.getString(cursor.getColumnIndex(COLUMN_MSG_ID)));
		msg.setMsgTime(cursor.getLong(cursor.getColumnIndex(COLUMN_MSG_TIME)));
		int dir = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_DIR));
		msg.setDirection(dir == 0 ? Direct.SEND : Direct.RECEIVE);
		int status = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_STATUS));
		if (status == EMMessage.Status.CREATE.ordinal()) {
			msg.setStatus(EMMessage.Status.CREATE);

		} else if (status == EMMessage.Status.INPROGRESS.ordinal()) {
			msg.setStatus(EMMessage.Status.INPROGRESS);
		} else if (status == EMMessage.Status.SUCCESS.ordinal()) {
			msg.setStatus(EMMessage.Status.SUCCESS);
		} else if (status == EMMessage.Status.FAIL.ordinal()) {
			msg.setStatus(EMMessage.Status.FAIL);
		}
		

		int isAcked = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_ISACKED));
		msg.setAcked(isAcked != 0);
		
		int isDelivered = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_ISDELIVERED));
		msg.setDeliverAcked(isDelivered != 0);
		
		int isListened = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_ISLISTENED));
		msg.setListened(isListened == 1);
		
		msg.setUnread(false);
		String groupId = cursor.getString(cursor.getColumnIndex(COLUMN_MSG_GROUP));
		if (groupId == null) {
			msg.setChatType(ChatType.Chat);
		} else {
		    int type = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_TYPE));
			msg.setChatType(ChatType.GroupChat);
			if(type == ChatType.ChatRoom.ordinal())
			    msg.setChatType(ChatType.ChatRoom);
			msg.setTo(groupId);
		}

		return msg;
	}

    private static void importGroupFromCursor(Cursor cursor) throws Exception {

        String username = cursor.getString(cursor.getColumnIndex(COLUMN_GROUP_NAME));
        int style = cursor.getInt(cursor.getColumnIndex(COLUMN_GROUP_PUBLIC)) == 0 ?
				EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE :
				EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL;
        String subject = cursor.getString(cursor.getColumnIndex(COLUMN_GROUP_NICK));
        String ownerName = cursor.getString(cursor.getColumnIndex(COLUMN_GROUP_OWNER));
        String description = cursor.getString(cursor.getColumnIndex(COLUMN_GROUP_DESC));
        boolean isBlocked = cursor.getInt(cursor.getColumnIndex(COLUMN_GROUP_IS_BLOCKED)) == 0 ? false : true;
        int maxUsers = cursor.getInt(cursor.getColumnIndex(COLUMN_GROUP_MAX_USERS));

        String memeberString = cursor.getString(cursor.getColumnIndex(COLUMN_GROUP_MEMBERS));
        StringTokenizer st = new StringTokenizer(memeberString, ",");
        List<String> members = new ArrayList<String>(); 
        while (st.hasMoreTokens()) {
            String member = st.nextToken();
            members.add(member);
        }

        instance.config.importGroup(username, style, ownerName, subject, description, members, isBlocked, maxUsers);
    }

    private static void importChatRoomFromCursor(Cursor cursor) throws Exception{
        String username = cursor.getString(cursor.getColumnIndex(COLUMN_CHAT_ROOM_ID));
        String subject = cursor.getString(cursor.getColumnIndex(COLUMN_CHAT_ROOM_NICK));
        String ownerName = cursor.getString(cursor.getColumnIndex(COLUMN_CHAT_ROOM_OWNER));
        String description = cursor.getString(cursor.getColumnIndex(COLUMN_CHAT_ROOM_DESC));
//        room.setAffiliationsCount(cursor.getInt(cursor.getColumnIndex(COLUMN_CHAT_ROOM_MEMBERS_SIZE)));
//        room.setMsgBlocked(cursor.getInt(cursor.getColumnIndex(COLUMN_CHAT_ROOM_IS_BLOCKED)) == 0 ? false : true);
        int maxUsers = cursor.getInt(cursor.getColumnIndex(COLUMN_CHAT_ROOM_MAX_USERS));

        String memeberString = cursor.getString(cursor.getColumnIndex(COLUMN_CHAT_ROOM_MEMBERS));

        StringTokenizer st = new StringTokenizer(memeberString, ",");
        List<String> members = new ArrayList<String>(); 
        while (st.hasMoreTokens()) {
            String member = st.nextToken();
            members.add(member);
        }
        instance.config.importChatRoom(username, ownerName, subject, description, members, maxUsers);
    }
    
    private static void importConversation(String convId, int type, String ext) {
        instance.config.importConversation(convId, type, ext);
    }
    
    private static void reloadAll() {
        instance.config.reloadAll();
    }

	/**
	 * get user or group unread message count
	 * 
	 * @param username
	 * @return
	 */
	public int getConversationUnreadCount(String username) {
		try {
			SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();

			int count = 0;
			Cursor cursor = db.rawQuery("select " + COLUMN_UNREAD_COUNT + " from " + UNREAD_TABLE_NAME + " where " + COLUMN_UNREAD_USERNAME
					+ " = ?", new String[] { username });
			if (cursor.moveToFirst()) {
				count = cursor.getInt(cursor.getColumnIndex(COLUMN_UNREAD_COUNT));
			}
			cursor.close();
			if(count < 0)
				return 0;
			return count;
		} catch (Exception e) {
			return 0;
		}
	}

	/**
	 * 获取有未读数的conversations
	 *
	 * get list of user id or group id which have the unread messages
	 * @return
	 */
	public List<String> getConversationsUnread() {
		ArrayList<String> conversations = new ArrayList<String>();
		try {
			SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();

			Cursor cursor = db.rawQuery("select * from " + UNREAD_TABLE_NAME, new String[] {});
			if (!cursor.moveToFirst()) {
				cursor.close();
				return conversations;
			}
			do {
				String userOrName = cursor.getString(0);
				int count = cursor.getInt(1);
				if(count > 0)
					conversations.add(userOrName);
			} while (cursor.moveToNext());
			cursor.close();
		} catch (Exception e) {
		}
		return conversations;
	}

	public boolean isUpdateFrom9To10() {
		return updateFrom9To10;
	}
	
	/**
	 * db open helper
	 */
	private static class EMChatDBOpenHelper extends SQLiteOpenHelper {
		private static EMChatDBOpenHelper instance = null;
		private String username;

		private EMChatDBOpenHelper(Context context, String username) {
			super(context, username + DATABASE_NAME, null, DATABASE_VERSION);
			this.username = username;
			EMLog.d(TAG, "created chatdb for :" + username);
		}

		public synchronized static EMChatDBOpenHelper getInstance(Context context, String username) {
			if (instance == null) {
				instance = new EMChatDBOpenHelper(context, username);
			}
			return instance;
		}

		public synchronized static void closeDB() {
			if (instance != null) {
				try {
					SQLiteDatabase db = instance.getWritableDatabase();
					db.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
				instance = null;
			}
		}

		@Override
		public void onCreate(SQLiteDatabase db) {
//			db.execSQL(CREATE_CHAT_TABLE);
//			db.execSQL(CREATE_GROUP_TABLE);
//			db.execSQL(CREATE_UNREAD_COUNT_TABLE);
//			db.execSQL(CREATE_TOKEN_TABLE);
//			db.execSQL(CREATE_CONTACT_TABLE);
//			db.execSQL(CREATE_BLACK_LIST_TABLE);
//			db.execSQL(CREATE_CONVERSATION_LIST_TABLE);
//			db.execSQL(CREATE_CHAT_ROOM_TABLE);
		}

		@Override
		public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
			Log.w(TAG, "Upgrading from version " + oldVersion + " to " + newVersion);
			if (oldVersion < 2) {
				db.execSQL(CREATE_UNREAD_COUNT_TABLE);
			}
			
			if(oldVersion < 3) {
				db.execSQL(CREATE_TOKEN_TABLE);
				db.execSQL(CREATE_CONTACT_TABLE);
			}
			
			if(oldVersion < 4) {
				try {
					db.delete(TOKEN_TABLE_NAME, COLUMN_TOKEN_USERNAME + " = ?", new String[] {username});
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			
			if (oldVersion < 5) {
				//add isDelivered column
				try {
				    db.execSQL("ALTER TABLE "+ CHAT_TABLE_NAME +" ADD COLUMN "+ 
				        COLUMN_MSG_ISDELIVERED + " integer ;");
				    EMLog.d(TAG,"db upgrade to vervison 5");
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			
			if(oldVersion < 6){
				db.execSQL(CREATE_BLACK_LIST_TABLE);
				db.execSQL("ALTER TABLE "+ CHAT_TABLE_NAME +" ADD COLUMN "+ 
				        COLUMN_MSG_ISLISTENED + " integer ;");
			}
			
			if(oldVersion < 7){
			    db.execSQL("ALTER TABLE " + GROUP_TABLE_NAME + " ADD COLUMN " + 
			            COLUMN_GROUP_MEMBERS_SIZE + " INTEGER ;");
			}
			
			if(oldVersion < 8){
			    db.execSQL("ALTER TABLE " + GROUP_TABLE_NAME + " ADD COLUMN " + 
			            COLUMN_GROUP_IS_BLOCKED + " INTEGER ;");
			}
			
			if(oldVersion < 9){
			    db.execSQL("ALTER TABLE " + GROUP_TABLE_NAME + " ADD COLUMN " + 
			            COLUMN_GROUP_MAX_USERS + " INTEGER ;");
		    }
			
			if(oldVersion < 10){				
			    db.execSQL(CREATE_CONVERSATION_LIST_TABLE);
			    initializeConversation(db);
			    // 因为之前本地Contact表中的内容是错误的， 收发过消息的group都作为联系人存在contact表中
			    // 本次升级需要将contact表清空
			    clearContactTable_v10(db);
			    EMDBManager.getInstance().updateFrom9To10 = true;
			}
			
			if(oldVersion < 12){
				db.execSQL(CREATE_CHAT_ROOM_TABLE);
				
                // 因为快牙部分用户0.5%左右，在"ALTER TABLE  CONVERSATION_LIST_TABLE_NAME"这句, 出现table找不到的问题
				// 无法锁定问题的真正原因，所以增加一句保护，如果该table不存在，判断并创建table
                if (EMDBManager.getInstance().config.getAppKey().equals("dewmobile#kuaiya")) {
	                try {
		                db.execSQL(CREATE_CONVERSATION_LIST_TABLE);
	                } catch (Exception e) {
		                e.printStackTrace();
	                }
                }

				if(!isColumnExist(db,CONVERSATION_LIST_TABLE_NAME,COLUMN_CONVERSATION_LIST_TYPE)){
					db.execSQL("ALTER TABLE " + CONVERSATION_LIST_TABLE_NAME  + " ADD COLUMN " + COLUMN_CONVERSATION_LIST_TYPE + " INTEGER ;");
				}

				db.execSQL("ALTER TABLE " + CHAT_TABLE_NAME + " ADD COLUMN " + COLUMN_MSG_TYPE + " INTEGER ;");
				migrateFrom10To12(db);
			}
			
			if (oldVersion < 13) {
				migrateToOneSDK(db);
			}
		}
		
		private void dropDB(SQLiteDatabase db) {
			try {
				db.execSQL("drop table " + CHAT_TABLE_NAME);
				db.execSQL("drop table " + GROUP_TABLE_NAME);
				db.execSQL("drop table " + UNREAD_TABLE_NAME);
				db.execSQL("drop table " + TOKEN_TABLE_NAME);
				db.execSQL("drop table " + CONTACT_TABLE_NAME);
				db.execSQL("drop table " + BLACK_LIST_TABLE_NAME);
				db.execSQL("drop table " + CONVERSATION_LIST_TABLE_NAME);
			} catch (Exception e) {
			}
		}
		/*
		private String _migrateGetAttachmentPath(EMMessage msg) {
			EMAMessageBody body = msg.bodies().get(0);
			if (body == null) {
				return "";
			}
			if (body instanceof EMAImageMessageBody) {
				return ((EMAImageMessageBody)body).getLocalUrl();
			} else if (body instanceof EMAVoiceMessageBody) {
				return ((EMAVoiceMessageBody)body).getLocalUrl();
			} else if (body instanceof EMAVideoMessageBody) {
				return ((EMAVideoMessageBody)body).getLocalUrl();
			} else if (body instanceof EMAFileMessageBody) {
				return ((EMAFileMessageBody)body).getLocalUrl();
			}
			return "";
		}
		
		private void _migrateUpdateAttachmentPath(EMMessage msg) {
			EMAMessageBody body = msg.bodies() == null ? null : msg.bodies().get(0);
			if (body == null) {
				return;
			}
			
			String path = _migrateGetAttachmentPath(msg);
			int pos = path.lastIndexOf("/");
			if (pos == -1) {
				return;
			}
			String fileName = path.substring(pos + 1, path.length());
			if (body instanceof EMAImageMessageBody) {
				String newPath = PathUtil.getInstance().getImagePath().getAbsolutePath() + "/" + fileName;
				((EMAImageMessageBody)body).setLocalPath(newPath);
			} else if (body instanceof EMAVoiceMessageBody) {
				String newPath = PathUtil.getInstance().getVoicePath().getAbsolutePath() + "/" + fileName;
				((EMAVoiceMessageBody)body).setLocalPath(newPath);
			} else if (body instanceof EMAVideoMessageBody) {
				String newPath = PathUtil.getInstance().getVideoPath().getAbsolutePath() + "/" + fileName;
				((EMAVideoMessageBody)body).setLocalPath(newPath);
			} else if (body instanceof EMAFileMessageBody) {
				String newPath = PathUtil.getInstance().getFilePath().getAbsolutePath() + "/" + fileName;
				((EMAFileMessageBody)body).setLocalPath(newPath);
			}
		}
	    */
		
		/**
		 * import messages to onesdk
		 * @param db
		 * @return success
		 */
		private boolean importMessages(SQLiteDatabase db) {
		    EMLog.d(TAG, "importMessages");
		    Cursor cursor = null;
            try{
                cursor = db.rawQuery( "SELECT * FROM " + CHAT_TABLE_NAME, null);
                
                if(!cursor.moveToFirst()){
                    cursor.close();
                    return true;
                }
                
                final int IMPORT_MSGS_BATCH_LIMIT = 1000;
                List<EMMessage> msgs = new ArrayList<EMMessage>();
                do{
                    EMMessage msg = loadMsgFromCursor(cursor);
                    msgs.add(msg);
                    
//                  // update attachment storage path, prepare for [5. action move ] 
//                  if (msg.direct == Direct.RECEIVE && 
//                          (msg.getType() == Type.IMAGE || msg.getType() == Type.VIDEO ||
//                           msg.getType() == Type.VOICE || msg.getType() == Type.FILE)) {
//                      String oldPath = _migrateGetAttachmentPath(msg);
//                      _migrateUpdateAttachmentPath(msg);
//                      String newPath = _migrateGetAttachmentPath(msg);
//                      if (!oldPath.equals("") && !newPath.equals("")) {
//                          toMoveItems.add(new Pair<String, String>(oldPath, newPath));
//                      }
//                  }
                    
                    if (msgs.size() >= IMPORT_MSGS_BATCH_LIMIT) {
                        EMClient.getInstance().chatManager().importMessages(msgs);
                        msgs.clear();
                    }
                }while(cursor.moveToNext());

                if (msgs.size() > 0) {
                    EMClient.getInstance().chatManager().importMessages(msgs);
                    msgs.clear();
                }
                cursor.close();
                return true;
            }catch (Exception e){
                 EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
                 return false;
            }finally{
                if(null != cursor && !cursor.isClosed()){
                    cursor.close() ;
                }
            }
		}
		
		/**
		 * import conversation to onesdk
		 * @param db
		 * @return success
		 */
		private boolean importConversations(SQLiteDatabase db) {
	        EMLog.d(TAG, "importConversations");

		    Cursor cursor = null;
		    try{
                List<EMConversation> conversations = new ArrayList<EMConversation>();
                
                cursor = db.rawQuery("select * from " + CONVERSATION_LIST_TABLE_NAME, null);
                if (!cursor.moveToFirst()) {
                    cursor.close();
                    return true;
                }
                
                List<Pair<String, EMConversationType>> convInfos = new ArrayList<Pair<String,EMConversationType>>();
                do {
                    boolean isGroup = !cursor.isNull(cursor.getColumnIndex(COLUMN_CONVERSATION_LIST_GROUP));
                    String username = !isGroup ? cursor.getString(cursor.getColumnIndex(COLUMN_CONVERSATION_LIST_USERNAME)) :
                        cursor.getString(cursor.getColumnIndex(COLUMN_CONVERSATION_LIST_GROUP));
                    EMConversationType type = !isGroup ? EMConversationType.Chat : EMConversationType.GroupChat;
                    importConversation(username, type.ordinal(), cursor.getString(cursor.getColumnIndex(COLUMN_CONVERSATION_LIST_EXT)));
                } while(cursor.moveToNext());
                
                return true;
            } catch (Exception e) {
                 EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
                 return false;
            } finally {
                if(null != cursor && !cursor.isClosed()){
                    cursor.close() ;
                }
            }   
		}
		
		/**
		 * import blacklist to onesdk
		 * @param db
		 * @return
		 */
		private boolean importBlackList(SQLiteDatabase db)
		{
		    EMLog.d(TAG, "importBlackList");

		    Cursor cursor = null;
		    try {
		        cursor = db.rawQuery("select * from " + BLACK_LIST_TABLE_NAME, null);
                if (!cursor.moveToFirst()) {
                    cursor.close();
                    return true;
                }
                
                List<String> blacklist = new ArrayList<String>();
                do {
                    String username = cursor.getString(0);
                    blacklist.add(username);
                } while(cursor.moveToNext());
                
                EMDBManager.getInstance().config.importBlackList(blacklist);
                return true;
		    } catch (Exception e) {
                EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
                return false;
           } finally {
               if(null != cursor && !cursor.isClosed()){
                   cursor.close() ;
               }
           }
		}
		
		
		/**
         * import conversation unread count to onesdk
         * @param db
         * @return
         */
        private boolean importUnread(SQLiteDatabase db)
        {
            EMLog.d(TAG, "importUnread");

            Cursor cursor = null;
            try {
                cursor = db.rawQuery("select * from " + UNREAD_TABLE_NAME, null);
                if (!cursor.moveToFirst()) {
                    cursor.close();
                    return true;
                }
                
                do {
                    String conversationId = cursor.getString(0);
                    int count = cursor.getInt(1);
                    EMDBManager.getInstance().config.updateConversationUnreadCount(conversationId, count);
                } while(cursor.moveToNext());
                
                return true;
            } catch (Exception e) {
                EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
                return false;
           } finally {
               if(null != cursor && !cursor.isClosed()){
                   cursor.close() ;
               }
           }
        }
        
        /**
         * import contacts to onesdk
         * @param db
         */
        private boolean importContacts(SQLiteDatabase db) {
            EMLog.d(TAG, "importContacts");
            Cursor cursor = null;
            try {
                cursor = db.rawQuery("select * from " + CONTACT_TABLE_NAME, null);
                if (!cursor.moveToFirst()) {
                    cursor.close();
                    return true;
                }
                
                List<String> contacts = new ArrayList<String>();
                
                do {
                    String username = cursor.getString(1);
                    contacts.add(username);
                } while(cursor.moveToNext());
                EMDBManager.getInstance().config.importContacts(contacts);
                return true;
            } catch (Exception e) {
                EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
                return false;
           } finally {
               if(null != cursor && !cursor.isClosed()){
                   cursor.close() ;
               }
           } 
        }
        
		static void createMigrateTable(SQLiteDatabase db) {
		    try {
		        // 1.create MIGRATE_ONESDK_TABLE_NAME table
		        db.execSQL(CREATE_MIGRATE_ONESDK_TABLE);
		    } catch (Exception e) {
		    }
		}
		
		static boolean migrationFinished(SQLiteDatabase db) {
	        try {
	            boolean done = false;
	            Cursor cursor = db.rawQuery("select * from " + MIGRATE_ONESDK_TABLE_NAME, new String[] { });
	            if(cursor != null){
	                if(cursor.moveToFirst()){
	                    int doneVal = cursor.getInt(cursor.getColumnIndex(COLUMN_MIGRATE_ONESDK_DONE));
	                    done = doneVal == 1;
	                    cursor.close();
	                }else{
	                    cursor.close();
	                }
	            }
	            return done;
	        } catch (Exception e) {
	        }
	        return false;
		}
		
		static boolean setMigrationDone(SQLiteDatabase db) {
            try {
                    ContentValues values = new ContentValues();
                    values.put(COLUMN_MIGRATE_ONESDK_DONE, 1);
                    return db.replace(MIGRATE_ONESDK_TABLE_NAME, null, values) != -1;
            } catch (Exception e) {
                e.printStackTrace();
            }
            return false;
		}
		
		public void migrateToOneSDK(SQLiteDatabase db) {
			/*
			 * 1. create MIGRATE_ONESDK_TABLE_NAME table, return when porting got finished
			 * 2. porting message
			 * 3. porting conversation
			 * 4. porting group
			 * 5. porting blacklist
			 * 6. porting contacts
			 * 7. porting conversation unread count
             * 8. update MIGRATE_ONESDK_TABLE_NAME table. recognize it as porting finish
			 * 9. clear previous data ?
             * note:
             *    no need conversation unread, onesdk don't have unread table
             *    no need porting chatrooms
             *    Not move image & voice & video & file to new places
			 */
            boolean fail = false;
            boolean result = false;
            Cursor cursor = null;
            
            // 1. create table, detect porting done
            EMLog.d(TAG, "---->start migrate");
            createMigrateTable(db);
            if (migrationFinished(db)) {
                return;
            }
			
			// 2. porting message
			result = importMessages(db);
			fail = fail | !result;
			
		    // 3. porting conversations
		    result = importConversations(db);
		    fail = fail | !result;

		    // 4. porting groups information
		    result = importGroups(db);
		    fail = fail | !result;
		    
		    // 5. porting blacklist
		    result = importBlackList(db);
		    fail = fail | !result;
		    
            // 6. porting contacts
            result = importContacts(db);
            fail = fail | !result;

            // 7. porting conversation unread count
		    result = importUnread(db);
		    fail = fail | !result;
            EMClient.getInstance().chatManager().loadAllConversations();
            
            
		    // 8. update MIGRATE_ONESDK_TABLE_NAME table. recognize it as porting finish
            if (!fail) {
                EMLog.d(TAG, "---->finished migrate");
                result = setMigrationDone(db);
                fail = fail | !result;
            }
            
		    // 9. clear previous data to save disk ?
            if (!fail) {
            	dropDB(db);
            }
            EMLog.d(TAG, "migration success!");
            reloadAll();
		}
		
		private boolean importGroups(SQLiteDatabase db) {
            EMLog.d(TAG,"importGroups");
			Cursor cursor = null;
	        try {
	            cursor = db.rawQuery("select * from " + GROUP_TABLE_NAME, new String[] {});
	            if (!cursor.moveToFirst()) {
	                cursor.close();
	                return true;
	            }
	            do {
	                importGroupFromCursor(cursor);
	            } while (cursor.moveToNext());
	            cursor.close();
	            return true;
	        } catch (Exception e) {
	            EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
	            e.printStackTrace();
	            return false;
	        } finally {
		        if(null != cursor && !cursor.isClosed()){
		            cursor.close() ;
		        }
		    }
	    }
		
		private boolean importChatRooms(SQLiteDatabase db) {
            EMLog.d(TAG,"importChatRooms");
			Cursor cursor = null;
	        try {
	            cursor = db.rawQuery("select * from " + CHAT_ROOM_TABLE_NAME, new String[] {});
	            if (!cursor.moveToFirst()) {
	                cursor.close();
	                return true;
	            }
	            do {
	                importChatRoomFromCursor(cursor);
	            } while (cursor.moveToNext());
	            cursor.close();
	            return true;
	        } catch (Exception e) {
	            EMLog.e(TAG,"migrateToOneSDK..." + e.getMessage()) ;
	            e.printStackTrace();
	            return false;
	        } finally {
		        if(null != cursor && !cursor.isClosed()){
		            cursor.close() ;
		        }
		    }
		}

		
		private boolean isColumnExist(SQLiteDatabase db, String tableName, String columnName) {
		    boolean result = false ;
		    Cursor cursor = null ;
		    try{
		        //查询一行
		        cursor = db.rawQuery( "SELECT * FROM " + tableName + " LIMIT 0"
		            , null );
		        result = cursor != null && cursor.getColumnIndex(columnName) != -1 ;
		    }catch (Exception e){
		         EMLog.e(TAG,"checkColumnExists..." + e.getMessage()) ;
		    }finally{
		        if(null != cursor && !cursor.isClosed()){
		            cursor.close() ;
		        }
		    }

		    return result ;
		}
		
		/**
		 * in order to keep compatibility with kuaiya's branch, we skip db version 11
		 * and make a jump from 10 to 12
		 * @param db
		 */
		private void migrateFrom10To12(SQLiteDatabase db){
			Cursor cursor = db.rawQuery("select * from " + CONVERSATION_LIST_TABLE_NAME,null);
			if(!cursor.moveToFirst()){
				cursor.close();
				return;
			}
			
			// we try to add a new conversation value into the new column - type
			do{
				// check if that's the group
				boolean isGroup = !cursor.isNull(cursor.getColumnIndex(COLUMN_CONVERSATION_LIST_GROUP));
				
				int type = EMConversation.EMConversationType.Chat.ordinal();
				String column = COLUMN_CONVERSATION_LIST_USERNAME;
				
				if(isGroup){
					type = EMConversation.EMConversationType.GroupChat.ordinal();
					column = COLUMN_CONVERSATION_LIST_GROUP;
				}
				
				ContentValues values = new ContentValues();
				
				values.put(COLUMN_CONVERSATION_LIST_TYPE, type);
				
				db.update(CONVERSATION_LIST_TABLE_NAME, values, column + " = ?", new String[]{cursor.getString(cursor.getColumnIndex(column))});
			}while(cursor.moveToNext());
			
			cursor.close();
		}
		
		private void clearContactTable_v10(SQLiteDatabase db) {
			try {
				EMLog.d(TAG, "add converstion with:" + username);
				db.execSQL("delete from " + CONTACT_TABLE_NAME, new String[] {});
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

		private void initializeConversation(SQLiteDatabase db) {
			EMLog.d(TAG, "initializeConversation");
			List<String> participants = new ArrayList<String>();
			List<String> groups = new ArrayList<String>();
			try {
				Cursor cursor = db.rawQuery("select distinct " + COLUMN_PARTICIPANT + " from " + CHAT_TABLE_NAME + " where groupname is null",
						null);
				if (cursor.moveToFirst()) {
					do {
						participants.add(cursor.getString(0));
					} while (cursor.moveToNext());
				}
				cursor.close();
				
				cursor = db.rawQuery("select distinct " + COLUMN_MSG_GROUP + " from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_GROUP
						+ " is not null", null);
				if (cursor.moveToFirst()) {
					do {
						groups.add(cursor.getString(0));
					} while (cursor.moveToNext());			
				}
				cursor.close();				
				EMLog.d(TAG, "load participants size:" + participants.size());
			} catch (Exception e) {
				e.printStackTrace();
			}
				
			for (String username : participants) {
				addConversation(db, username, false);
			}
			for (String group : groups) {
				addConversation(db, group, true);
			}
		}
		
		private void addConversation(SQLiteDatabase db, String username, boolean isGroup) {
			try {
				EMLog.d(TAG, "add converstion with:" + username);
				String columnName = !isGroup ? COLUMN_CONVERSATION_LIST_USERNAME : COLUMN_CONVERSATION_LIST_GROUP; 
				db.execSQL("insert into " + CONVERSATION_LIST_TABLE_NAME + " ("
						+ columnName +")" +
						" select ? where not exists (select null from " + CONVERSATION_LIST_TABLE_NAME + " where " + columnName + " = ?)", 
						new String[] {username, username});
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
}
