package com.hyphenate.chat.adapter;

import com.hyphenate.chat.EMClient;
import com.hyphenate.cloud.CustomMultiPartEntity;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.cloud.HttpClientConfig;
import com.hyphenate.util.EMLog;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.util.ByteArrayBuffer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.Map;
import java.util.Map.Entry;

import internal.org.apache.http.entity.mime.content.FileBody;
import internal.org.apache.http.entity.mime.content.StringBody;

public class EMARHttpAPI {
	
	public static final String TAG = "EMARHttpAPI";
	
    public static String Method_GET = "GET";
	public static String Method_POST = "POST";
	public static String Method_PUT = "PUT";
	public static String Method_DELETE = "DELETE";
	public static int REQUEST_FAILED_CODE = 408;
	public static int REQUEST_AUTHENTICATION_FAILED = 400;
    public static int HIGH_SPEED_DOWNLOAD_BUF_SIZE = 1024*30;

    final boolean tokenRetrieved = false;

    public static int upload(final String localFilePath, String remoteFilePath,  final Map<String, String> headers,
            StringBuilder responseBuf, final EMARHttpCallback callback) {

        String appId = EMClient.getInstance().getOptions().getAppKey();
        String jid = EMClient.getInstance().getCurrentUser();
        File sourceFile = new File(localFilePath);
        int code = REQUEST_FAILED_CODE;
        EMLog.d(TAG, "upload");

        CustomMultiPartEntity multipartEntity = new CustomMultiPartEntity(null);
        try {
            if (appId != null) {
                multipartEntity.addPart("app", new StringBody(appId));
            }
            if (jid != null) {
                multipartEntity.addPart("id", new StringBody(jid));
            }
        } catch (Exception e) {
            EMLog.e(TAG, e.getMessage());
            return code;
        }

        String mimeType = getMimeType(sourceFile);
        String remoteFileName = remoteFilePath;
        if (localFilePath.indexOf("/") > 0) {
            remoteFileName = remoteFileName.substring(localFilePath.lastIndexOf("/"));
        }
        multipartEntity.addPart("file", new FileBody(sourceFile, remoteFileName, mimeType, "UTF-8"));

        final long totalSize = multipartEntity.getContentLength();

        multipartEntity.setListener(new CustomMultiPartEntity.ProgressListener() {
            @Override
            public void transferred(long num) {
                int progressCode = (int) ((num / (float) totalSize* 100));
                if(progressCode != 100){
                    if (callback != null) {
                        callback.onProgress(totalSize, num);
                    }
                }
            }
        });
        
        String remoteUrl = HttpClientConfig.getFileRemoteUrl(remoteFilePath);
        
        HttpPost request = new HttpPost(remoteUrl);

        request.setEntity(multipartEntity);

        if (headers != null) {
            for (Entry<String, String> item : headers.entrySet()) {
                request.addHeader(item.getKey(), item.getValue());
            }
        }

        DefaultHttpClient httpclient = HttpClientConfig.getDefaultHttpClient();

        try {
            HttpResponse response = httpclient.execute(request);

            code = response.getStatusLine().getStatusCode();
            
            switch (code) {
            case HttpStatus.SC_OK:
                try {
                    ByteArrayBuffer buffer = getResponseContent(/*EMChat.getInstance().getAppContext(), */response);
                    EMLog.d(TAG, "getBackupInfo result:" + new String(buffer.buffer(), 0, buffer.length()));
                    responseBuf.append(new String(buffer.buffer(), 0, buffer.length()));
                } catch (Exception e) {
                    EMLog.e("sendImageMessage", "json parse exception remotefilepath:" + remoteFilePath);
                }
    
            case HttpStatus.SC_UNAUTHORIZED:
            default:
                break;
            }
        } catch(ConnectTimeoutException e) {
            EMLog.e(TAG, "ConnectTimeoutException");
        } catch (Exception e) {
            EMLog.e(TAG, e.getMessage());
        }
        return code;
    }

	public static int download(String remoteUrl, final String localFilePath, final Map<String, String> headers, EMARHttpCallback callback) {
	    int responseCode = REQUEST_FAILED_CODE;

	    if(remoteUrl == null || remoteUrl.length() <= 0){
            EMLog.e(TAG, "invalid remoteUrl");
            return responseCode;
        }

        remoteUrl = HttpClientConfig.getFileRemoteUrl(remoteUrl);
        remoteUrl = processUrl(remoteUrl);
        EMLog.d(TAG, "download file: remote url : " + remoteUrl + " , local file : " + localFilePath);
        
        File file = new File(localFilePath);
        EMLog.d(TAG, "local exists:" + file.exists());
        
        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
        }

        int timeout = HttpClientConfig.getTimeout(headers);

        DefaultHttpClient client = HttpClientConfig.getDefaultHttpClient(timeout);

        try {
            HttpGet request = new HttpGet(remoteUrl);

            processHeaders(request,headers);

            HttpResponse response = client.execute(request);
            responseCode = response.getStatusLine().getStatusCode();

            switch(responseCode){
                case HttpStatus.SC_OK:
                {
                    long size = onDownloadCompleted(response,callback,localFilePath);

                    if(size <= 0){
                        return REQUEST_FAILED_CODE;
                    }

                    if (callback != null) {             
                        EMLog.e(TAG, "download successfully");
                    }

                    return responseCode;
                }

                case HttpStatus.SC_UNAUTHORIZED:
                {
                    // in case SC_UNAUTHORIZED, we think this would be caused by invalid token
                    // so firstly, we need to retrieve token again and check, if token is retrieved but still failed
                    // we will report a failure
                    long tokenSaveTime = EMHttpClient.getInstance().chatConfig().getTokenSaveTime();
                    
                    // in case we have retrieved 10 minutes before, we just report an error
                    if((System.currentTimeMillis()-tokenSaveTime) <= 10*60*1000){
                        if (callback != null){
                            EMLog.e(TAG, "unauthorized file");  
                        }
                        return responseCode;
                    }
                    break;
                }

                default:
                {
                    EMLog.e(TAG, "error response code is :" + responseCode);
                    return responseCode;
                }
            }
        } catch (Exception e) {
            String errorMsg = e.getMessage();
            if(errorMsg == null){
                errorMsg = e.toString();

                if(errorMsg == null){
                    errorMsg = "failed to download file";
                }
            }

            EMLog.e(TAG, errorMsg);
            return responseCode;
        }
        return responseCode;
	}

    public static int httpExecute(String reqURL, Map<String, String> headers, String body, String method, StringBuilder strResponse) {
    	ByteArrayBuffer buffer;
    	int code = REQUEST_FAILED_CODE;

    	EMLog.d(TAG, "httpExecute");
		try {
	    	HttpResponse response = _httpExecute(reqURL, headers, body, method);
			buffer = getResponseContent(/*EMChat.getInstance().getAppContext(),*/ response);
			strResponse.delete(0, strResponse.length());
			strResponse.append(new String(buffer.buffer(), 0, buffer.length()));
			code = response.getStatusLine().getStatusCode();
			EMLog.d(TAG, "httpExecute code: " + code);
			return code;
		} catch(ConnectTimeoutException e) {
			EMLog.e(TAG, "ConnectTimeoutException");
		} catch (Exception e) {
			EMLog.e(TAG, e.getMessage());
			e.printStackTrace();
		}
		EMLog.e(TAG, "can't catch exceptions");
		return code;
    }

    private static HttpResponse _httpExecute(String reqURL, Map<String, String> headers, String body, String method) throws ClientProtocolException, IOException, KeyStoreException, KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, CertificateException, ConnectTimeoutException{
        HttpResponse response = null;
        
        int timeout = HttpClientConfig.getTimeout(headers);
        
        HttpClient httpClient = HttpClientConfig.getDefaultHttpClient(timeout);
        
        HttpRequestBase request = null;
        
        if (method.equals(Method_POST)) {
            HttpPost httpPost = new HttpPost(reqURL);
            httpPost.setEntity(new StringEntity(body, "UTF-8"));
            request = httpPost;
        } else if (method.equals(Method_PUT)) {
            HttpPut httpPut = new HttpPut(reqURL);
            httpPut.setEntity(new StringEntity(body, "UTF-8"));
            request = httpPut;
        } else if (method.equals(Method_GET)) {
            request = new HttpGet(reqURL);
        } else if (method.equals(Method_DELETE)) {
            request = new HttpDelete(reqURL);
        }
        
        if(request != null){
            if (headers != null) {
                for (Entry<String, String> item : headers.entrySet()) {
                    request.setHeader(item.getKey(), item.getValue());
                }
            }
            
            response = httpClient.execute(request);
        }
        
        return response;
    }

    private static ByteArrayBuffer getResponseContent(/*Context context, */HttpResponse response) throws Exception {
        InputStream input = null;

        final int bufSize = HIGH_SPEED_DOWNLOAD_BUF_SIZE;
        byte[] temp = new byte[bufSize];
        ByteArrayBuffer buffer = new ByteArrayBuffer(bufSize);

        long count = 0;
        try {
            input = response.getEntity().getContent();
            
            while ((count = input.read(temp)) != -1) {
                buffer.append(temp, 0, (int) count);
            }
        } catch (IllegalStateException e) {
            e.printStackTrace();
            throw e;
        } catch (IOException e) {
            e.printStackTrace();
            throw e;
        } finally {
            input.close();
        }
        return buffer;
    }
    
	private static String getMimeType(File sourceFile){
		String sourceName=sourceFile.getName();
		if (sourceName.endsWith(".3gp") || sourceName.endsWith(".amr")) {
			return "audio/3gp";
		}if(sourceName.endsWith(".jpe")||sourceName.endsWith(".jpeg")||sourceName.endsWith(".jpg")){
			return "image/jpeg";
		}if(sourceName.endsWith(".amr")){
			return "audio/amr";
		}if(sourceName.endsWith(".mp4")){
			return "video/mp4";
		}else {
			return "image/png";
		}
	}

    private static long onDownloadCompleted(HttpResponse response, final EMARHttpCallback callback, final String localFilePath) throws IOException, IllegalStateException{
        HttpEntity entity = response.getEntity();
        
        if(entity == null){
            return 0;
        }
        
        InputStream input = null;
        OutputStream output = null;
        
        int count = 0;
        int current_progress = 0;

        // get file length
        long fileLength = entity.getContentLength();

        try {
            input = entity.getContent();
        } catch (IllegalStateException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            throw e;
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            throw e;
        }
        
        File outputFile = new File(localFilePath);
        
        try {
            output = new FileOutputStream(outputFile);
        } catch (FileNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            input.close();
            
            throw e;
        }
        
        int bufSize = HIGH_SPEED_DOWNLOAD_BUF_SIZE;
        byte[] buffer = new byte[bufSize];
        long total = 0;
        
        try {
            while ((count = input.read(buffer)) != -1) {
                total += count;
                int progress = (int) ((total * 100) / fileLength);
                EMLog.d(TAG, progress + "");
                if (progress == 100 || progress > current_progress + 5) {
                    current_progress = progress;
                    if (callback != null)
                        callback.onProgress(fileLength, total);
                }
                output.write(buffer, 0, count);
            }
            
            return outputFile.length();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            throw e;
        } finally{
            output.close();
            input.close();
        }
    }
    
    private static String processUrl(String remoteUrl){
        if (remoteUrl.contains("+")) {
            remoteUrl = remoteUrl.replaceAll("\\+", "%2B");
        }
        
        if (remoteUrl.contains("#")) {
            remoteUrl = remoteUrl.replaceAll("#", "%23");
        }
        
        return remoteUrl;
    }
    
    private static void processHeaders(HttpGet request, Map<String, String> headers){
        request.addHeader("Authorization", "Bearer " + EMClient.getInstance().getOptions().getAccessToken());
        request.addHeader("Accept", "application/octet-stream");
        
        
        if (headers != null) {
            for (Entry<String, String> item : headers.entrySet()) {
                if(item.getKey().equals("Authorization")||item.getKey().equals("Accept")){
                    continue;
                }
                request.addHeader(item.getKey(), item.getValue());
            }
        }
    }
}
