package com.hyphenate.chat.adapter;

import java.util.List;

import com.hyphenate.chat.EMCursorResult;
import com.hyphenate.chat.EMGroupInfo;

public class EMAGroupManager extends EMABase {

    /**
      * \brief Add a listener to group manager.
      *
      * @param  A group manager listener.
      * @return NA
      */
    public void addListener(EMAGroupManagerListener listener) {
    	nativeAddListener(listener);
    }
    native void nativeAddListener(EMAGroupManagerListener listener);
    
    /**
      * \brief Remove a listener.
      *
      * @param  A group manager listener.
      * @return NA
      */
    public void removeListener(EMAGroupManagerListener listener) {
    	nativeRemoveListener(listener);
	}
    native void nativeRemoveListener(EMAGroupManagerListener listener);
    
    /**
      * \brief Remove all the listeners.
      *
      * @param  NA
      * @return NA
      */
    public void clearListeners() {
        nativeClearListeners();
    }
    native void nativeClearListeners();
    
    /**
      * \brief Get groups for login user from memory.
      *
      * @param  EMAError used for output.
      * @return All user joined groups in memory.
      */
    public List<EMAGroup> allMyGroups(EMAError error) {
    	return nativeAllMyGroups(error);
    }
    native List<EMAGroup> nativeAllMyGroups(EMAError error);
    
    /**
      * \brief Fetch all groups for login user from server.
      *
      * Note: Groups in memory will be updated.
      * @param  EMAError used for output.
      * @return All user joined groups.
      */
    public List<EMAGroup> fetchAllMyGroups(EMAError error) {
    	return nativeFetchAllMyGroups(error);
    }
    native List<EMAGroup> nativeFetchAllMyGroups(EMAError error);
    
    /**
      * \brief Fetch app's public groups.
      *
      * Note: User can input empty string as cursor at the first time.
      * @param  Page's cursor.
      * @param  Page's size.
      * @param  EMAError used for output.
      * @return Wrapper of next page's cursor and current page result.
      */
    public EMCursorResult<EMGroupInfo> fetchPublicGroupsWithCursor(
        String cursor,
        int pageSize,
        EMAError error) {
    	return nativeFetchPublicGroupsWithCursor(
            cursor,
            pageSize,
            error);
	}
    native EMCursorResult<EMGroupInfo> nativeFetchPublicGroupsWithCursor(
            String cursor,
            int pageSize,
            EMAError error);
   
    /**
      * \brief Create a new group.
      *
      * Note: Login user will be the owner of created .
      * @param  Group's subject.
      * @param  Group's description.
      * @param  Welcome message that will be sent to invited user.
      * @param  Group's setting.
      * @param  Group's members.
      * @param  EMAError used for output.
      * @return The created group.
      */
    public EMAGroup createGroup(
        String subject,
        String description,
        String welcomeMessage,
        EMAGroupSetting setting,
        List<String> members,
        EMAError error) {
    return nativeCreateGroup(
            subject,
            description,
            welcomeMessage,
            setting,
            members,
            error);
    }
    native EMAGroup nativeCreateGroup(
            String subject,
            String description,
            String welcomeMessage,
             EMAGroupSetting setting,
             List<String> members,
            EMAError error);
    
    /**
      * \brief Join a public group.
      *
      * Note: The group's style must be PUBLIC_JOIN_OPEN, or will return error.
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The joined group.
      */
    public EMAGroup joinPublicGroup(
        String groupId,
        EMAError error
        ) {
    	return nativeJoinPublicGroup(groupId,error);
    }
    native EMAGroup nativeJoinPublicGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Apply to join a public group, need owner's approval.
      *
      * Note: The group's style must be PUBLIC_JOIN_APPROVAL, or will return error.
      * @param  Group's ID.
      * @param  Nick name in the group.
      * @param  Apply message, that will be sent to group owner.
      * @param  EMAError used for output.
      * @return The group that try to join.
      */
    public EMAGroup applyJoinPublicGroup(
        String groupId,
        String nickName,
        String message,
        EMAError error) {
    	return nativeApplyJoinPublicGroup(
            groupId,
            nickName,
            message,
            error
            );
    }
    native EMAGroup nativeApplyJoinPublicGroup(
            String groupId,
            String nickName,
            String message,
            EMAError error
            );
    
    /**
      * \brief Leave a group.
      *
      * Note: Group's owner can't leave the group.
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The leaved group.
      */
    public EMAGroup leaveGroup(
        String groupId,
        EMAError error
        ) {
    	return nativeLeaveGroup(
            groupId,
            error
            );
    }
    native EMAGroup nativeLeaveGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Destroy a group.
      *
      * Note: Only group's owner can destroy the group.
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The destroyed group.
      */
    public EMAGroup destroyGroup(
        String groupId,
        EMAError error
        ) {
    	return nativeDestroyGroup(
             groupId,
             error
            );
    }
    native EMAGroup nativeDestroyGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Add members to a group.
      *
      * Note: Maybe user don't have the priviledge, it depends on group's setting.
      * @param  Group's ID.
      * @param  Invited users.
      * @param  Welcome message that will be sent to invited user.
      * @param  EMAError used for output.
      * @return The group that added members.
      */
    public EMAGroup addGroupMembers(
        String groupId,
        List<String> members,
        String welcomeMessage,
        EMAError error
        ) {
    	return nativeAddGroupMembers(
            groupId,
            members,
            welcomeMessage,
            error
            );
    }
    native EMAGroup nativeAddGroupMembers(
            String groupId,
            List<String> members,
            String welcomeMessage,
            EMAError error
            );
    
    /**
      * \brief Remove members from a group.
      *
      * Note: Only group's owner can remove members.
      * @param  Group's ID.
      * @param  Removed users.
      * @param  EMAError used for output.
      * @return The group that removed members.
      */
    public EMAGroup removeGroupMembers(
        String groupId,
        List<String> members,
        EMAError error
        ) {
    	return nativeRemoveGroupMembers(
            groupId,
            members,
            error
            );
    }
    native EMAGroup nativeRemoveGroupMembers(
            String groupId,
            List<String> members,
            EMAError error
            );
    
    /**
      * \brief Block group's members, the blocked user can't send message in the group.
      *
      * Note: Only group's owner can block other members.
      * @param  Group's ID.
      * @param  Blocked users.
      * @param  EMAError used for output.
      * @param  The reason that why block the members.
      * @return The group that blocked members.
      */
    public EMAGroup blockGroupMembers(
        String groupId,
        List<String> members,
        EMAError error,
        String reason
        ) {
    	return nativeBlockGroupMembers(
            groupId,
            members,
            error,
            reason
            );
    }
    native EMAGroup nativeBlockGroupMembers(
            String groupId,
            List<String> members,
            EMAError error,
            String reason
            );
    
    /**
      * \brief Unblock group's members.
      *
      * Note: Only group's owner can unblock other members.
      * @param  Group's ID.
      * @param  Unblocked users.
      * @param  EMAError used for output.
      * @return The group that unblocked members.
      */
    public EMAGroup unblockGroupMembers(
        String groupId,
        List<String> members,
        EMAError error
        ) {
    return nativeUnblockGroupMembers(
            groupId,
            members,
            error
            );
    }
    native EMAGroup nativeUnblockGroupMembers(
            String groupId,
            List<String> members,
            EMAError error
            );
    
    /**
      * \brief Change group's subject.
      *
      * Note: Only group's owner can change group's subject.
      * @param  Group's ID.
      * @param  The new group subject.
      * @param  EMAError used for output.
      * @return The group that with new subject.
      */
    public EMAGroup changeGroupSubject(
        String groupId,
        String newSubject,
        EMAError error
        ) {
    	return nativeChangeGroupSubject(
            groupId,
            newSubject,
            error
            );
    }
    native EMAGroup nativeChangeGroupSubject(
            String groupId,
            String newSubject,
            EMAError error
            );
    
    /**
      * \brief Change group's description.
      *
      * Note: Only group's owner can change group's description.
      * @param  Group's ID.
      * @param  The new group description.
      * @param  EMAError used for output.
      * @return The group that with new description.
      */
    public EMAGroup changeGroupDescription(
        String groupId,
        String newDescription,
        EMAError error
        ) {
    	return nativeChangeGroupDescription(
            groupId,
            newDescription,
            error
            );
    }
    native EMAGroup nativeChangeGroupDescription(
            String groupId,
            String newDescription,
            EMAError error
            );
    
    /**
      * \brief Get group's specification.
      *
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @param  Whether get group's members.
      * @return The group that update it's specification.
      */
    public EMAGroup fetchGroupSpecification(
        String groupId,
        EMAError error,
        boolean fetchMembers
        ) {
    	return nativeFetchGroupSpecification(
            groupId,
            error,
            fetchMembers
            );
    }
    native EMAGroup nativeFetchGroupSpecification(
            String groupId,
            EMAError error,
            boolean fetchMembers
            );
    
    /**
      * \brief Get group's black list.
      *
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The group's black list.
      */
    public List<String> fetchGroupBans(
        String groupId,
        EMAError error
        ) {
    	return nativeFetchGroupBans(
            groupId,
            error
            );
    }
    native List<String> nativeFetchGroupBans(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Search for a public group.
      *
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The search result.
      */
    public EMAGroup searchPublicGroup(
        String groupId,
        EMAError error
        ) {
    	return nativeSearchPublicGroup(
            groupId,
            error
            );
    }
    native EMAGroup nativeSearchPublicGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Block group message.
      *
      * Note: Owner can't block the group message.
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The group that blocked message.
      */
    public EMAGroup blockGroupMessage(
        String groupId,
        EMAError error
        ) {
    	return nativeBlockGroupMessage(
            groupId,
            error
            );
    }
    native EMAGroup nativeBlockGroupMessage(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Unblock group message.
      *
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The group that unclocked message.
      */
    public EMAGroup unblockGroupMessage(
        String groupId,
        EMAError error
        ) {
    	return nativeUnblockGroupMessage(
            groupId,
            error
            );
    }
    native EMAGroup nativeUnblockGroupMessage(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Accept user's join application.
      *
      * Note: Only group's owner can approval someone's application.
      * @param  Group's ID.
      * @param  The user that send the application.
      * @param  EMAError used for output.
      * @return NA.
      */
    public void acceptJoinGroupApplication(
        String groupId,
        String user,
        EMAError error) {
    	nativeAcceptJoinGroupApplication(
            groupId,
            user,
            error);
    }
    native void nativeAcceptJoinGroupApplication(
            String groupId,
            String user,
            EMAError error);
    
    /**
      * \brief Reject user's join application.
      *
      * Note: Only group's owner can reject someone's application.
      * @param  Group's ID.
      * @param  The user that send the application.
      * @param  The reject reason.
      * @param  EMAError used for output.
      * @return NA.
      */
    public void declineJoinGroupApplication(
        String groupId,
        String user,
        String reason,
        EMAError error) {
    nativeDeclineJoinGroupApplication(
            groupId,
            user,
            reason,
            error);
    }
    native void nativeDeclineJoinGroupApplication(
            String groupId,
            String user,
            String reason,
            EMAError error);
    
    public void loadAllMyGroupsFromDB() {
        nativeLoadAllMyGroupsFromDB();
    }
    native void nativeLoadAllMyGroupsFromDB();
    
    

    /**
      * \brief accept group's invitation.
      *
      * @param  Group's ID.
      * @param  Inviter.
      * @param  EMError used for output.
      * @return The group user has accepted.
      */
    public EMAGroup acceptInvitationFromGroup(
        String groupId,
        String inviter,
        EMAError error) {
        return nativeacceptInvitationFromGroup(groupId, inviter, error);
    }
    native EMAGroup nativeacceptInvitationFromGroup(
            String groupId,
            String inviter,
            EMAError error);
    
    /**
      * \brief decline group's invitation.
      *
      * @param  Group's ID.
      * @param  Inviter.
      * @param  The decline reason.
      * @param  EMError used for output.
      * @return NA.
      */
    public void declineInvitationFromGroup(
        String groupId,
        String inviter,
        String reason,
        EMAError error) {
        nativedeclineInvitationFromGroup(groupId, inviter, reason, error);
    }
    native void nativedeclineInvitationFromGroup(
            String groupId,
            String inviter,
            String reason,
            EMAError error);
    
    
    public EMAGroupManager() {
    }
    
    public EMAGroupManager(EMAGroupManager old) {
    	nativeInit(old);
    }
    native void nativeInit(EMAGroupManager groupManager);
}
