package com.hyphenate.chat;

import android.os.Build;
import android.text.TextUtils;
import android.util.Pair;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.gcm.GoogleCloudMessaging;
import com.huawei.android.pushagent.api.PushManager;
import com.hyphenate.EMError;
import com.hyphenate.chat.core.EMChatConfigPrivate.EMMipushConfig;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;
import com.xiaomi.mipush.sdk.MiPushClient;

import org.apache.http.HttpStatus;
import org.json.JSONObject;

import java.lang.reflect.Method;
import java.util.Random;

class EMPushHelper {

	public static final String TAG = EMPushHelper.class.getSimpleName();
	private static EMPushHelper instance;
	private Thread pushThread = null;
	private Object sendTokenLock = new Object();
	private String notifyDeviceToken;
	private boolean isLogout = false;


	public static EMPushHelper getInstance(){
		if(instance == null){
			instance = new EMPushHelper();
		}
		return instance;
	}
    
    private EMPushType pushType = EMPushType.NORMAL;
    
    public void setPushType(EMPushType pushType){
        this.pushType = pushType;
    }
    
    public EMPushType getPushType(){
        return pushType;
    }

    
    // we will check if the push service available
	// the push service could be GCM, XIAOMI, HuaWei
	boolean checkAvailablePushService() {
		boolean gcmEnabled = EMClient.getInstance().getChatConfigPrivate().isGcmEnabled();
		
		EMLog.d(TAG, "GCM is enabled : " + gcmEnabled);
		
		boolean pushAvailable = false;
        try {
		    if(gcmEnabled && Class.forName("com.google.android.gms.common.GooglePlayServicesUtil") != null){
				int resultCode = GooglePlayServicesUtil.isGooglePlayServicesAvailable(EMClient.getInstance().getContext());
				
				pushAvailable = (resultCode == ConnectionResult.SUCCESS);
				EMLog.d(TAG, "GCM service available : " + pushAvailable);
				if(pushAvailable) {
					setPushType(EMPushType.GCM);
				}
			 }
		} catch (ClassNotFoundException e) {
			EMLog.e(TAG, "" + e.toString());
		} catch (Exception e){}
        
        //优先使用GCM
        if(pushAvailable){
            return pushAvailable;
        }
        
        try {
            if(Class.forName("com.xiaomi.mipush.sdk.MiPushClient") != null){
                pushAvailable = MiPushClient.shouldUseMIUIPush(EMClient.getInstance().getContext());
                EMLog.d(TAG, "mipush available : " + pushAvailable);
                if(pushAvailable)
                    setPushType(EMPushType.MIPUSH);
            }
        } catch (ClassNotFoundException e) {
            EMLog.d(TAG, "no mipush sdk");
            e.printStackTrace();
        } catch (Exception e) {
            EMLog.d(TAG, "exception, regard it as no mipush sdk");
            e.printStackTrace();
        }
        //华为push
        if(!pushAvailable){
            try {
                if(Class.forName("com.huawei.android.pushagent.api.PushManager") != null){
                    Class<?> classType = Class.forName("android.os.SystemProperties");
                    Method getMethod = classType.getDeclaredMethod("get", new Class<?>[] {String.class});
                    String buildVersion = (String)getMethod.invoke(classType, new Object[]{"ro.build.version.emui"});
                    //在某些手机上，invoke方法不报错
                    if(!TextUtils.isEmpty(buildVersion)){
                        pushAvailable = true;
                        EMLog.d(TAG, "huawei push available : " + pushAvailable);
                        setPushType(EMPushType.HUAWEIPUSH);
                    }
                }
            } catch (Exception e) {
                EMLog.d(TAG, "no huawei push sdk or mobile is not a huawei phone");
                e.printStackTrace();
            }

        }
        
        
        return pushAvailable;
	}
	 
    boolean isPushServiceEnabled(){
    	return pushType != EMPushType.NORMAL;
    }
    
	void onInit(){
		 isLogout = false;
	}
	
	void sendDeviceTokenToServer(){
		if(isPushServiceEnabled()){
		    EMLog.d(TAG, "third-party push available");
			if(isLogout){
				return;
			}
			if(pushThread != null && pushThread.isAlive()){
				return;
			}
			
			if(pushThread == null){
				pushThread = new Thread() {
					@Override
					public void run() {

	                    class HandleSendFail {
	                        void onSendFail() {
                                setPushType(EMPushType.NORMAL);
	                            EMClient.getInstance().doStopService();
	                            EMClient.getInstance().doStartService();
	                        }
	                    }
	                    
						try {
				               // get device token from google, token cached in local
	                        String deviceToken = null;
	                        for (int i = 0; i < 3; i++) {
	                            deviceToken = getDeviceToken();
	                            if (deviceToken != null) {
	                                break;
	                            }
	                            try {
	                                sleep(new Random().nextInt(10) * 1000);
	                            } catch (Exception e) {
	                                e.printStackTrace();
	                            }
	                            if(isInterrupted()) {
	                                return;
	                            }
	                        }
	                        if (deviceToken == null) {
	                            new HandleSendFail().onSendFail();
	                            return;
	                        }
	                        
	                        // register device token to easemob server, be executed only once
	                        if (getPushToken() == null) {
	                            boolean response = false;
	                            for (int i = 0; i< 3; i++) {
	                                response = sendDeviceInfo(deviceToken);
	                                if (response == true) {
	                                    break;
	                                }
	                                try {
	                                    sleep((20 + new Random().nextInt(10)) * 1000);
	                                } catch (Exception e) {
	                                    e.printStackTrace();
	                                }
	                                if(isInterrupted()) {
	                                    return;
	                                }
	                            }
	                            if (response == false) {
	                                new HandleSendFail().onSendFail();
	                                return;
	                            }
	                        }
	                        
	                        
	                        // bind device token to easemob server, each time login 
	                        boolean responseToken = false;
	                        int attempts = 0;
	                        EMRandomDelay randomDelay = new EMRandomDelay();
	                        while (responseToken == false) {
	                            responseToken = sendTokenToServer(deviceToken);
	                            if (responseToken == true) {
	                                break;
	                            }
	                            try {
	                                sleep(randomDelay.timeDelay(attempts++) * 1000);
	                            } catch (Exception e) {
	                                e.printStackTrace();
	                            }
	                            if(isInterrupted()) {
	                                return;
	                            }
	                        }
	                        if (responseToken == true) {
								//cancel job scheduler if push is available
								EMClient.getInstance().cancelJob();
	                            EMClient.getInstance().doStopService();
	                        }
						} catch (Exception e) {
							EMLog.e(TAG, e.toString());
						}
						
					}
				};
				pushThread.start();
			}
		}else {
			EMLog.d(TAG, "GCM and mipush not available");
		}
	}
	 
	 
    public void onDestroy(boolean isUnbindDeviceToken) throws HyphenateException{
        EMLog.d(TAG, "push notification helper ondestory");
        //notify send push token lock
        onReceiveToken(null);
        if(pushThread != null){
            pushThread.interrupt();
            pushThread = null;
        }
        isLogout = true;
        
        if(isUnbindDeviceToken && isPushServiceEnabled()){
        	boolean ret = sendTokenToServer("");
        	
        	if(!ret){
        	    EMLog.d(TAG, "unbind device token faild");
        		throw new HyphenateException(EMError.USER_UNBIND_DEVICETOKEN_FAILED,"unbind device token failed");
        	}
        	
        	setPushType(EMPushType.NORMAL);
        }
	}
    
    /**
     * 从第三方推送服务器获取到token，如果失败传null
     * @param notifyDeviceToken
     */
    void onReceiveToken(String notifyDeviceToken){
        this.notifyDeviceToken = notifyDeviceToken;
        synchronized (sendTokenLock) {
            try {
                sendTokenLock.notify();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
	 
	/**
	 * 
	 * @param deviceToken
	 * @return
	 * @throws HyphenateException 
	 */
	 boolean sendTokenToServer(String deviceToken){
		 synchronized (sendTokenLock) {
			 
			 String remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl() + "/users/"
							+ EMClient.getInstance().getCurrentUser();
			try {
				JSONObject json = new JSONObject();
				json.put("device_token", deviceToken);
				String notifier_name = null;
				switch (pushType) {
                case GCM:
                    notifier_name = EMClient.getInstance().getOptions().getGCMNumber();
                    break;
                case MIPUSH:
                    notifier_name = EMClient.getInstance().getOptions().getMipushConfig().appId;
                    break;
                case HUAWEIPUSH:
                    notifier_name = EMClient.getInstance().getOptions().getHuaweiPushAppId();
                    break;

                }
				if(notifier_name == null)
				    notifier_name = "";
				json.put("notifier_name", notifier_name);
				
				EMLog.d(TAG, "send device token to server, token = " + deviceToken + ",url = " + remoteUrl + 
				        ",notifier_name = " + notifier_name);
				Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl, json.toString(), EMHttpClient.PUT);
				int statusCode = response.first;
				String content = response.second;
				

				switch (statusCode) {
				case HttpStatus.SC_OK:
					EMLog.d(TAG, "sendTokenToServer SC_OK:");
					
					return true;
				
				default:
					EMLog.d(TAG, "sendTokenToServer error:" + content);
                    break;
				}
			} catch (Exception e) {
                EMLog.e(TAG, e.toString());
			}
			
			return false;
		}
	 }
	 
	 boolean sendDeviceInfo(String deviceToken){
		String remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl() + "/devices";
		
		JSONObject json = new JSONObject();
		try {
			json.put("model", "android");
			json.put("name", deviceToken);
			json.put("token", deviceToken);
			json.put("sdk_version", EMClient.getInstance().getChatConfigPrivate().getVersion());
			json.put("os_version", Build.VERSION.RELEASE);
            DeviceUuidFactory deviceFactory = new DeviceUuidFactory(EMClient.getInstance().getContext());
            final String uuid = deviceFactory.getDeviceUuid().toString();
            json.put("device_uuid", uuid);

			Pair<Integer, String> response = EMHttpClient.getInstance().sendRequest(remoteUrl, null, json.toString(), EMHttpClient.POST);

			int statusCode = response.first;
			String content = response.second;
			
			switch (statusCode) {
			case HttpStatus.SC_OK:
				setPushToken(deviceToken);
				EMLog.d(TAG, "sendDeviceToServer SC_OK:");
				return true;
			default:
				if(content.contains("duplicate_unique_property_exists")){
				    EMLog.d(TAG, "device token already exists");
				    setPushToken(deviceToken);
					return true;
				}
				
				EMLog.d(TAG, "sendDeviceToServer error : " + content);
				break;
			}
		} catch (Exception e){
			EMLog.e(TAG, e.toString());
		}
		
      
		return false;
	}
	 
	String getDeviceToken() {
		String deviceToken = getPushToken();
		
		if (deviceToken != null) {
			return deviceToken;
		}else {
			try {
			    EMPushType pushType = getPushType();
                if(pushType == EMPushType.GCM){ //GCM
                    if(EMClient.getInstance().getOptions().getGCMNumber() != null){
                        GoogleCloudMessaging gcm = null;
                        if (gcm == null) {
                            gcm = GoogleCloudMessaging.getInstance(EMClient.getInstance()
                                    .getContext());
                        }
                        deviceToken = gcm.register(EMClient.getInstance().getOptions().getGCMNumber());
                    }
                }else if(pushType == EMPushType.MIPUSH){ //xiaomi push
                    EMMipushConfig mipushConfig = EMClient.getInstance().getOptions().getMipushConfig();
                    if(mipushConfig != null){
                        MiPushClient.registerPush(EMClient.getInstance().getContext(), mipushConfig.appId, mipushConfig.appKey);
                        //异步请求，wait到register结果，logout时可能正在wait，需要解锁
                        synchronized (sendTokenLock) {
                            try {
                                sendTokenLock.wait();
                            } catch (InterruptedException e) {
                                e.printStackTrace();
                            }
                        }
                        deviceToken = notifyDeviceToken;
                    }
                }else if(pushType == EMPushType.HUAWEIPUSH){
                    PushManager.requestToken(EMClient.getInstance().getContext());
                    synchronized (sendTokenLock) {
                        //最多等待60s,防止华为push请求token失败不返回结果
                        try {
                            sendTokenLock.wait(60 * 1000);
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }
                    }
                    deviceToken = notifyDeviceToken;
                }
				
			} catch (Exception e) {
				EMLog.e(TAG, "get device token with error: " + e.toString());
			}
			EMLog.d(TAG, "devicetoken = " + deviceToken);
		}

		if(deviceToken == null || "".equals(deviceToken)){
			setPushType(EMPushType.NORMAL);
		}
		
		return deviceToken;

	}
	
	public String getPushToken(){
        return EMPreferenceUtils.getInstance().getPushToken();
    }
    
    public void setPushToken(String token){
        EMPreferenceUtils.getInstance().setPushToken(token);
    }


	public enum EMPushType{
       GCM,
       MIPUSH,
       HUAWEIPUSH,
       NORMAL
	}
}

class EMRandomDelay{
    public int timeDelay(int attempts){
        if(attempts == 0){
            return new Random().nextInt(5) + 1;
        }

        if(attempts == 1){
            return new Random().nextInt(54) + 6;
        }

        return new Random().nextInt(540) + 60;
    }
}