package com.hyphenate.chat;

import com.hyphenate.EMChatRoomChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;
import com.hyphenate.chat.adapter.EMAChatRoomManager;
import com.hyphenate.chat.adapter.EMAChatRoomManagerListener;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class EMChatRoomManager {
    EMAChatRoomManager emaObject;
	private List<EMChatRoomChangeListener> chatRoomListeners = Collections.synchronizedList(new ArrayList<EMChatRoomChangeListener>());
	private ExecutorService threadPool = null;
	private List<EMChatRoom> chatRooms = Collections.synchronizedList(new ArrayList<EMChatRoom>());
	
	EMClient mClient;
	public EMChatRoomManager(EMClient client, EMAChatRoomManager manager) {
	    emaObject = manager;
	    emaObject.addListener(chatRoomListenerImpl);
		mClient = client;
		threadPool = Executors.newCachedThreadPool();
	}

	/**
	 * \~chinese
	* 注册聊天室事件监听
	* @param listener
	* 
	* \~english
	* register chat room change listener
	* @param listener
	*/
	public void addChatRoomChangeListener(EMChatRoomChangeListener listener) {
		chatRoomListeners.add(listener);
	}

    /**
     * \~chinese
     * 移除聊天室事件监听
	 * @param listener
     * @deprecated  使用 {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} 替代
	 *
	 * \~english
	 * remove chat room change listener
	 * @param listener
     * @deprecated  use {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} instead
	 */
    @Deprecated
	public void removeChatRoomChangeListener(EMChatRoomChangeListener listener) {
	    removeChatRoomListener(listener);
    }

	/**
	 * \~chinese
	 * 移除聊天室事件监听
	 * @param listener
	 *
	 * \~english
	 * remove chat room change listener
	 * @param listener
	 */
	public void removeChatRoomListener(EMChatRoomChangeListener listener)
	{
		chatRoomListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 加入聊天室,异步方法
	 * @param roomId 
     * @param callback
	 * 
	 * \~english
     * join the chat room asynchronously
	 * @param roomId chat room id to join
	 * @param callback callback for success or fail 
	 */
	public void joinChatRoom(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMAChatRoom r = EMChatRoomManager.this.emaObject.joinChatRoom(roomId, error);
				EMChatRoom room = new EMChatRoom(r);
				if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == EMAError.CHATROOM_ALREADY_JOINED) {
					callback.onSuccess(room);
				} else {
					callback.onError(error.errCode(), error.errMsg());
				}
			}
		});

	}
	
	/**
	* \~chinese
	* 离开聊天室,异步方法
	* @param roomId 
	* 
	* \~english
	* leave the chat room asynchronously
	* @param roomId 
	*/
	public void leaveChatRoom(final String roomId) {
		EMChatRoom room = getChatRoom(roomId);
		if (room == null) {
		    return;
		}

        EMOptions options = EMClient.getInstance().getOptions();

        boolean allowed = options.isChatroomOwnerLeaveAllowed();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
            return;
        }

        // temp workaround to delete conversation in case of leaving the room
        // but deleteConversation could be a time consuming operation?
        EMClient.getInstance().chatManager().deleteConversation(roomId,true);
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
			}
		});
	}
	
	/**
	    * \~chinese
	    * 分页从服务器获取聊天室
	    * 耗时操作
	    * @param pageNum 当前要拉取的页数，从1开始
	    * @param pageSize 当前要拉取页的数量
	    * @return EMPageResult<EMChatRoom>
	    * @throws HyphenateException
	    *
	    * \~english
	    * fetch the chatroom in page synchronoulsy
	    * this is a time consuming ooperation
	    *
	    * @param pageNum --page number, start from 1
	    * @param pageSize -- size of page
	    * @return EMPageResult<EMChatRoom>List<EMChatRoom>
	    * @throws HyphenateException
	    */
	    public EMPageResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageNum, int pageSize) throws HyphenateException {
	        EMAError error = new EMAError();
	        EMPageResult<EMAChatRoom> result = emaObject.fetchChatroomsWithPage(pageNum, pageSize, error);
	        handleError(error);

	        List<EMAChatRoom> rooms = result.getData();
	        int count = result.getPageCount();

	        EMPageResult<EMChatRoom> ret = new EMPageResult<EMChatRoom>();
	        List<EMChatRoom> data = new ArrayList<EMChatRoom>();
	        for (EMAChatRoom room : rooms) {
	            data.add(new EMChatRoom(room));
	        }
	        ret.setPageCount(count);
	        ret.setData(data);
	        
	        chatRooms.clear();
	        chatRooms.addAll(data);
	        return ret;
	    }

	/**
	* \~chinese
	* 翻页从服务器获取聊天室
	* 耗时操作
	* @param pageSize 一次取多少条
    * @param cursor 从这个游标位置开始取
    * @return EMResult<EMChatRoom>List<EMChatRoom>
    * @throws HyphenateException
    * @deprecated use fetchPublicChatRoomsFromServer(int pageNum, int pageSize)
	*
	* \~english
	* fetch the chatroom in page synchronoulsy
	* this is a time consuming ooperation
	*
	* @param pageSize how many chat room to get
	* @param cursor cursor position
	* @return EMResult<EMChatRoom>List<EMChatRoom>
	* @throws HyphenateException
	* @deprecated use fetchPublicChatRoomsFromServer(int pageNum, int pageSize)
	*/
	public EMCursorResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAChatRoom> result = emaObject.fetchAllChatrooms(error);
		handleError(error);
		
		EMCursorResult<EMChatRoom> ret = new EMCursorResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : result) {
			data.add(new EMChatRoom(room));
		}
		ret.setCursor(null);
		ret.setData(data);
		
		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}
	
	/**
     * \~chinese
     * 翻页从服务器获取聊天室
     * 耗时操作
     * @param pageSize 一次取多少条
     * @param cursor 从这个游标位置开始取
     * @return EMResult<EMChatRoom>List<EMChatRoom>
     * @throws HyphenateException
     * @deprecated use asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback)
     *
     * \~english
     * fetch the chatroom in page synchronoulsy
     * this is a time consuming ooperation
     *
     * @param pageSize how many chat room to get
     * @param cursor cursor position
     * @return EMResult<EMChatRoom>List<EMChatRoom>
     * @throws HyphenateException
     * @deprecated use asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback)
     */
    public void asyncFetchPublicChatRoomsFromServer(final int pageSize, final String cursor, final EMValueCallBack<EMCursorResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMCursorResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
    
    /**
     * \~chinese
     * 翻页从服务器获取聊天室
     * 耗时操作
     * @param pageNum 当前要拉取的页数，从1开始
     * @param pageSize 当前要拉取页的数量
     * @return EMResult<EMChatRoom>List<EMChatRoom>
     * @throws HyphenateException
     *
     * \~english
     * fetch the chatroom in page asynchronously
     * this is a time consuming operation
     *
     * @param pageNum --page number, start from 1
     * @param pageSize -- size of page
     * @return EMResult<EMChatRoom>List<EMChatRoom>
     * @throws HyphenateException
     */
    public void asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMPageResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageNum, pageSize);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	/**
	* \~chinese
	* 从服务器获取聊天室详情
	* 耗时操作
	* @param roomId
    * @return EMChatRoom
    * @throws HyphenateException
    * 
    * \~english
	* fetch the chat room by room id
	*
	* @param roomId
	* @return EMChatRoom
	* @throws HyphenateException
	*/
	public EMChatRoom fetchChatRoomFromServer(String roomId) throws HyphenateException {
		return fetchChatRoomFromServer(roomId, false);
	}
	
	/**
    * \~chinese
    * 从服务器获取聊天室详情
    * 耗时操作
    * @param roomId 聊天室id
    * @param fetchMembers 是否需要获取聊天室成员
    * @return EMChatRoom 
    * @throws HyphenateException
    * 
    * \~english
    * fetch the chat room by room id
    *
    * @param roomId chat room id
    * @param fetchMembers Whether you need to get chat room members
    * @return EMChatRoom
    * @throws HyphenateException
    */
 	public EMChatRoom fetchChatRoomFromServer(String roomId, boolean fetchMembers)throws HyphenateException {
 	   EMAError error = new EMAError();
       EMAChatRoom room = emaObject.fetchChatroomSpecification(roomId, error, fetchMembers);
       handleError(error);
       return new EMChatRoom(room);
 	}
	
	/**
    * \~chinese
    * 从服务器获取聊天室详情
    * 耗时操作
    * @param roomId
    * @return EMChatRoom
    * @throws HyphenateException
    * 
    * \~english
    * fetch the chat room by room id
    *
    * @param roomId
    * @return EMChatRoom
    * @throws HyphenateException
    */
    public void asyncFetchChatRoomFromServer(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMChatRoom chatRoom = fetchChatRoomFromServer(roomId);
                    callback.onSuccess(chatRoom);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });   
    }
	
	/**
	 * \~chinese
	 * 从内存中获取聊天室
	 * 非耗时操作
	 * @param roomId
     * @return
	 *
	 * \~english
	 * get the chat room in the cache
	 * @param roomId
	 * @return
	 */
	public EMChatRoom getChatRoom(String roomId) {
		/*for (EMChatRoom room : chatRooms) {
			if (room != null && room.getId().equals(roomId)) {
				return room;
			}
		}*/
		
		
		EMAChatRoom room = emaObject.getChatroom(roomId);
		
		if(room == null){
			return null;
		}
		
		return new EMChatRoom(room);
	}
	
	/**
	 * \~chinese
	 * 获取当前内存的聊天室
	 * @return
	 * 
	 * \~english
	 * get all chat rooms in the cache
	 * @return
	 */
	public List<EMChatRoom> getAllChatRooms() {
		return Collections.unmodifiableList(chatRooms);
	}
	
	void importChatRoom(String username, String ownerName, String subject, String description, List<String> members, int maxUsers) {
		
	}
	
	//===========================================================================================================
	EMAChatRoomManagerListener chatRoomListenerImpl = new EMAChatRoomManagerListener() {

		@Override
		public void onLeaveChatRoom(EMAChatRoom chatroom, int leaveReason) {
            EMClient.getInstance().chatManager().caches.remove(chatroom.getId());
		    synchronized (chatRoomListeners) {
		        for (EMChatRoomChangeListener listener : chatRoomListeners) {
		            if (leaveReason == EMAChatRoomManagerListener.BE_KICKED) {
						listener.onRemovedFromChatRoom(chatroom.getId(), chatroom.getName(), "");
		            } else {
		                listener.onChatRoomDestroyed(chatroom.getId(), chatroom.getName());
		            }
		        }
		    }
		}

		@Override
		public void onMemberJoinedChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
		        for (EMChatRoomChangeListener listener : chatRoomListeners) {
		            listener.onMemberJoined(chatroom.getId(), member);
		        }
		    }
		}

		@Override
		public void onMemberLeftChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
		        for (EMChatRoomChangeListener listener : chatRoomListeners) {
		            listener.onMemberExited(chatroom.getId(), chatroom.getName(), member);
		        }
		    }
		}
	};
//	
//	EMChatRoomManager(EMAChatRoomManager manager) {
//		super(manager);
//	}
	
	private void handleError(EMAError error)  throws HyphenateException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new HyphenateException(error);
	    }
	}
	
	void onLogout() {
	    chatRoomListeners.clear();
	}
}
