package com.hyphenate.chat;

import android.content.Intent;
import android.hardware.Camera;
import android.hardware.Camera.CameraInfo;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.support.v4.util.Pair;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.chat.EMCallStateChangeListener.CallError;
import com.hyphenate.chat.EMCallStateChangeListener.CallState;
import com.hyphenate.chat.adapter.EMACallConference;
import com.hyphenate.chat.adapter.EMACallManager;
import com.hyphenate.chat.adapter.EMACallManagerListener;
import com.hyphenate.chat.adapter.EMACallRtcImpl;
import com.hyphenate.chat.adapter.EMACallRtcListenerDelegate;
import com.hyphenate.chat.adapter.EMACallSession;
import com.hyphenate.chat.adapter.EMACallStream;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.EMNoActiveCallException;
import com.hyphenate.exceptions.EMServiceNotReadyException;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.media.EMLocalSurfaceView;
import com.hyphenate.media.EMOppositeSurfaceView;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.superrtc.sdk.RtcConnection;
import com.superrtc.sdk.RtcConnection.RtcStatistics;
import com.superrtc.sdk.VideoView;
import com.superrtc.sdk.VideoViewRenderer;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;


public class EMCallManager {
    /**
     * Be careful about synchronized usage,
     * there are lots of callback, may lead to concurrent issue
     */
    static final String TAG = "EMCallManager";
    EMACallManager emaObject;
    EMClient mClient;
    List<EMCallStateChangeListener> callListeners = Collections.synchronizedList(new ArrayList<EMCallStateChangeListener>());
    EMACallListenerDelegate delegate = new EMACallListenerDelegate();
    EMCallSession currentSession;
    EMCallConference currentConference;
    EMVideoCallHelper callHelper = new EMVideoCallHelper();
    EMCameraDataProcessor processor;
    EMCameraDataProcessorDelegate mProcessorDelegate = new EMCameraDataProcessorDelegate();
    CallStateUnion callState = new CallStateUnion();
    EMCallOptions callOptions;
    boolean isConnectedFromRinging = false;


    static {
        RtcConnection.registerLogListener(new RtcConnection.LogListener() {
            @Override
            public void onLog(int i, String s) {
                EMLog.d(TAG + "$RTC", s);
            }
        });
    }

    /**
     * Need a more sophiscated to record more state info
     * @author linan
     *
     */
    static class CallStateUnion {
        /**
         * 0-3 bit:
         *      CALLState.ordinal
         * 4 bit: voice pause | resume
         *      0 resume
         *      1 pause
         * 5 bit: video pause | resume
         *      0 resume
         *      1 pause
         * 6-7: network state
         *     00 NETWORK_NORMAL
         *     01 NETWORK_UNSTABLE
         *     02 NETWORK_DISCONNECTED
         */
        short BIT0 = 0x01;
        short BIT1 = 0x02;
        short BIT2 = 0x04;
        short BIT3 = 0x08;
        short BIT4 = 0x10;
        short BIT5 = 0x20;
        short BIT6 = 0x40;
        short BIT7 = 0x80;

        BitSet callState = new BitSet();

        /**
         * patch: for incoming call, state change flow -> ring -> connected
         * for this case, connected should also be viewed as ringing, for reject action
         */
        boolean ringingToConnected = false;

        void reset() {
            callState.clear();
            ringingToConnected = false;
        }
        void changeState(CallState state) {
            switch (state) {
                case IDLE:
                case RINGING:
                case ANSWERING:
                case CONNECTING:
                case CONNECTED:
                case ACCEPTED:
                case DISCONNECTED:
                    if (isRinging() && state == CallState.CONNECTED) {
                        ringingToConnected = true;
                    }
                    callState.clear(0, 3);
                    int val = state.ordinal();
                    if ((val & BIT0) > 0) {
                        callState.set(0);
                    } else {
                        callState.clear(0);
                    }
                    if ((val & BIT1) > 0) {
                        callState.set(1);
                    } else {
                        callState.clear(1);
                    }
                    if ((val & BIT2) > 0) {
                        callState.set(2);
                    } else {
                        callState.clear(2);
                    }
                    if ((val & BIT3) > 0) {
                        callState.set(3);
                    } else {
                        callState.clear(3);
                    }
                    if (state == CallState.DISCONNECTED) {
                        reset();
                    }
                    break;
                case VOICE_PAUSE:
                    callState.set(4);
                    break;
                case VOICE_RESUME:
                    callState.clear(4);
                    break;
                case VIDEO_PAUSE:
                    callState.set(5);
                    break;
                case VIDEO_RESUME:
                    callState.clear(5);
                    break;
                case NETWORK_NORMAL:
                    callState.clear(6, 7);
                    break;
                case NETWORK_UNSTABLE:
                    callState.set(6);
                    callState.clear(7);
                    break;
                case NETWORK_DISCONNECTED:
                    callState.clear(6);
                    callState.set(7);
                    break;
                default:
                    break;
            }
        }

        CallState getState() {
            int val = 0;
            if (callState.get(0)) {
                val |= BIT0;
            }
            if (callState.get(1)) {
                val |= BIT1;
            }
            if (callState.get(2)) {
                val |= BIT2;
            }
            if (callState.get(3)) {
                val |= BIT3;
            }
            return CallState.values()[val];
        }

        boolean isIdle() {
            return (callState.get(0) == false &&
                    callState.get(1) == false &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isRinging_() {
            return (callState.get(0) == true &&
                    callState.get(1) == false &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isRinging() {
            return isRinging_() || (isConnected() && ringingToConnected);
        }

        boolean isAnswering() {
            return (callState.get(0) == false &&
                    callState.get(1) == true &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isConnecting() {
            return (callState.get(0) == true &&
                    callState.get(1) == true &&
                    callState.get(2) == false &&
                    callState.get(3) == false);
        }

        boolean isConnected() {
            return (callState.get(0) == false &&
                    callState.get(1) == false &&
                    callState.get(2) == true &&
                    callState.get(3) == false);
        }

        boolean isAccepted() {
            return (callState.get(0) == true &&
                    callState.get(1) == false &&
                    callState.get(2) == true &&
                    callState.get(3) == false);
        }

        boolean isDisconnected() {
            return (callState.get(0) == false &&
                    callState.get(1) == true &&
                    callState.get(2) == true &&
                    callState.get(3) == false);
        }

        boolean isVoicePause() {
            return callState.get(4);
        }

        boolean isVideoPause() {
            return callState.get(5);
        }
    }

    RtcConnection mRtcConnection;
    RtcConnection.Listener mRtcListener;

    boolean isVideoCall = true;

    @Deprecated
    public void clearRtcConnection() {
        mRtcConnection = null;
    }

//    Hashtable<String, RtcConnection> mRtcMap = new Hashtable<String, RtcConnection>();

    interface EMConferenceListener
    {
        void onConferenceMemberEntered(final String callId, final String enteredName);

        void onConferenceMemberExited(final String callId, final String exitedName);

        void onConferenceMemberPublished(final String callId, final String pubedName);

        void onConferenceMembersUpdated(final String callId);

        void onConferenceClosed(final String callId);
    }
    List<EMConferenceListener> mConferenceListeners = Collections.synchronizedList(new ArrayList<EMConferenceListener>());

    //===================== public interface =====================
    /**
     * Interface for user pre-process camera data
     *
     */
    public interface EMCameraDataProcessor {
        void onProcessData(byte[] data, final Camera camera, final int width, final int height, final int rotateAngle);
    }

    class EMCameraDataProcessorDelegate implements RtcConnection.RtcCameraDataProcessor {

        @Override
        public void onProcessData(byte[] bytes, Camera camera, int width, int height, int rotate) {
            EMCameraDataProcessor processor = EMCallManager.this.processor;
            if (processor != null) {
                processor.onProcessData(bytes, camera, width, height, rotate);
            }
        }

        @Override
        public void setResolution(int i, int i1) {
        }
    }

    public interface EMCallPushProvider {
        /**
         * Function is called when remote peer is offline, we want to let remote peer known we are calling.
         * For IOS that's go through APNS, for Android that's will go through GCM, MiPush, HuaweiPush.
         * Condition: Only works when EMOptions's configuration setPushCall to be true.
         */
        void onRemoteOffline(final String to);
    }

    EMCallPushProvider mPushProvider;

    final EMCallPushProvider defaultProvider = new EMCallManager.EMCallPushProvider() {

        void updateMessageText(final EMMessage oldMsg, final String to) {
            // update local message text
            EMConversation conv = EMClient.getInstance().chatManager().getConversation(oldMsg.getTo());
            if (conv != null) {
                conv.removeMessage(oldMsg.getMsgId());
            }
        }

        @Override
        public void onRemoteOffline(final String to) {

            //this function should exposed & move to Demo
            EMLog.d(TAG, "onRemoteOffline, to:" + to);

            final EMMessage message = EMMessage.createTxtSendMessage("You have an incoming call", to);
            // set the user-defined extension field
            JSONObject jTitle = new JSONObject();
            try {
                jTitle.put("em_push_title", "You have an incoming call");
            } catch (Exception e) {
                e.printStackTrace();
            }
            message.setAttribute("em_apns_ext", jTitle);

            message.setAttribute("is_voice_call", !isVideoCall);

            message.setMessageStatusCallback(new EMCallBack(){

                @Override
                public void onSuccess() {
                    EMLog.d(TAG, "onRemoteOffline success");
                    updateMessageText(message, to);
                }

                @Override
                public void onError(int code, String error) {
                    EMLog.d(TAG, "onRemoteOffline Error");
                    updateMessageText(message, to);
                }

                @Override
                public void onProgress(int progress, String status) {
                }
            });
            // send messages
            EMClient.getInstance().chatManager().sendMessage(message);
        }
    };

    public static final String IncomingCallAction = "com.hyphenate.action.incomingcall";
    public String getIncomingCallBroadcastAction() {
        return IncomingCallAction;
    }

    public EMCallManager(EMClient client, EMACallManager manager) {
        mClient = client;
        emaObject = manager;
        emaObject.addListener(delegate);
        try {
            RtcConnection.initGlobal(EMClient.getInstance().getContext());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Deprecated
    public void setCameraDataProcessor(EMCameraDataProcessor processor) {
        this.processor = processor;
    }

    public void setPushProvider(EMCallPushProvider provider) {
        this.mPushProvider = provider;
    }

    public CallState getCallState() {
        return callState.getState();
    }

    public EMVideoCallHelper getVideoCallHelper() {
        return callHelper;
    }

    /**
     * \~chinese
     * 发起呼叫，进行视频通话请求，进行视频呼叫前，请先在Activity.onCreate中先执行setSurfaceView。
     * @param username
     *             被呼叫的用户id
     * @throws EMServiceNotReadyException
     *             呼叫过程中遇到异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate firstly.
     * @param username
     *             callee's user id.
     * @throws HyphenateException
     */
    public void makeVideoCall(String username) throws EMServiceNotReadyException {
        makeVideoCall(username, "");
    }

    /**
     * \~chinese
     * 发起呼叫，进行视频通话请求，进行视频呼叫前，请先在Activity.onCreate中先执行setSurfaceView。
     * @param username
     *             被呼叫的用户id
     * @param ext  扩展字段，用户可以用来传输自定义的内容
     * @throws EMServiceNotReadyException
     *             呼叫过程中遇到异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate firstly.
     * @param username
     *             callee's user id.
     * @param ext  extension string, user can transfer self defined content
     * @throws HyphenateException
     */
    public void makeVideoCall(String username, String ext) throws EMServiceNotReadyException {
        EMLog.d(TAG, "makeVideoCall");
        isVideoCall = true;
        if (!EMClient.getInstance().isConnected()) {
            EMLog.d(TAG, "exception isConnected:" + false);
            throw new EMServiceNotReadyException("exception isConnected:" + false);
        }
        if (!NetUtils.hasDataConnection(EMClient.getInstance().getContext())) {
            EMLog.d(TAG, "Has no network connection");
            throw new EMServiceNotReadyException(EMError.NETWORK_ERROR, "Has no network connection");
        }
        if (!callState.isIdle() && !callState.isDisconnected()) {
            EMLog.d(TAG, "exception callState:" + callState);
            throw new EMServiceNotReadyException("exception callState:" + callState.getState().toString());
        }

        EMAError error = new EMAError();
        EMACallSession _session = emaObject.makeCall(username, EMACallSession.Type.VIDEO, ext, error);
        synchronized (this) {
            if (error.errCode() != EMAError.EM_NO_ERROR) {
                EMLog.d(TAG, "errorCode:" + error.errCode());
                currentSession = null;
                changeState(CallState.DISCONNECTED, getCallError(error));
                throw new EMServiceNotReadyException(error.errCode(), error.errMsg());
            }
            currentSession = new EMCallSession(_session);
            changeState(CallState.CONNECTING);
        }
    }

    /**
     * \~chinese
     * 发起呼叫，进行语音通话请求
     * @param username
     *             被呼叫的用户id
     * @throws EMServiceNotReadyException
     *             呼叫过程中遇到异常
     *             如果IM没有连接，或者之前通话的连接没有断开，会抛出次异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate firstly.
     * @param username
     *             callee's user id.
     * @throws EMServiceNotReadyException
     *             if IM is not connected, or previous call doesn't disconnected, will throw EMServiceNotReadyException
     */
    public void makeVoiceCall(String username) throws EMServiceNotReadyException {
        makeVoiceCall(username, "");
    }

    /**
     * \~chinese
     * 发起呼叫，进行语音通话请求
     * @param username
     *             被呼叫的用户id
     * @param ext  扩展字段，用户可以用来传输自定义的内容
     * @throws EMServiceNotReadyException
     *             呼叫过程中遇到异常
     *             如果IM没有连接，或者之前通话的连接没有断开，会抛出次异常
     *
     * \~english
     * make video call, before make video call, call setSurfaceView in Activity.onCreate firstly.
     * @param username
     *             callee's user id.
     * @param ext  extension string, user can transfer self defined content
     * @throws EMServiceNotReadyException
     *             if IM is not connected, or previous call doesn't disconnected, will throw EMServiceNotReadyException
     */
    public void makeVoiceCall(String username, String ext) throws EMServiceNotReadyException {
        EMLog.d(TAG, "makeVoiceCall");
        isVideoCall = false;
        if (!EMClient.getInstance().isConnected()) {
            EMLog.d(TAG, "exception isConnected:" + false);
            throw new EMServiceNotReadyException("exception isConnected:" + false);
        }
        if (!NetUtils.hasDataConnection(EMClient.getInstance().getContext())) {
            EMLog.d(TAG, "Has no network connection");
            throw new EMServiceNotReadyException(EMError.NETWORK_ERROR, "Has no network connection");
        }
        if (callState.isIdle() == false && callState.isDisconnected() == false) {
            EMLog.d(TAG, "exception callState:" + callState);
            throw new EMServiceNotReadyException("exception callState:" + callState.getState().toString());
        }
        EMAError error = new EMAError();
        EMACallSession _session =  emaObject.makeCall(username, EMACallSession.Type.VOICE, ext, error);
        synchronized (this) {
            if (error.errCode() != EMAError.EM_NO_ERROR) {
                EMLog.d(TAG, "errorCode:" + error.errCode());
                currentSession = null;
                changeState(CallState.DISCONNECTED, getCallError(error));
                throw new EMServiceNotReadyException(error.errCode(), error.errMsg());
            }
            currentSession = new EMCallSession(_session);
            changeState(CallState.CONNECTING);
        }
    }

    private CallError getCallError(EMAError error) {
        CallError callError = CallError.ERROR_TRANSPORT;
        switch (error.errCode()) {
            case EMError.CALL_INVALID_ID:
                callError = CallError.ERROR_NONE;
                break;
            case EMError.CALL_BUSY:
                callError = CallError.ERROR_BUSY;
                break;
            case EMError.CALL_REMOTE_OFFLINE:
                callError = CallError.ERROR_UNAVAILABLE;
                break;
            case EMError.CALL_CONNECTION_ERROR:
                callError = CallError.ERROR_TRANSPORT;
                break;
            default:
                break;
        }
        return callError;
    }

    /**
     * 设置通话状态监听
     * @param listener
     */
    public void addCallStateChangeListener(EMCallStateChangeListener listener) {
        synchronized (callListeners) {
            if (!callListeners.contains(listener)) {
                callListeners.add(listener);
            }
        }
    }

    /**
     * 移除通话监听
     * @param listener
     */
    public void removeCallStateChangeListener(EMCallStateChangeListener listener) {
        synchronized (callListeners) {
            if (callListeners.contains(listener)) {
                callListeners.remove(listener);
            }
        }
    }

    static RtcConnection createRtcConnection(String name) {
        RtcConnection rtc = new RtcConnection(name);
        EMCallOptions options = EMClient.getInstance().callManager().getCallOptions();
        if (options.isUserSetAutoResizing) {
            rtc.enableFixedVideoResolution(options.userSetAutoResizing);
        }
        if (options.isUserSetMaxFrameRate) {
            rtc.setMaxVideoFrameRate(options.userSetMaxFrameRate);
        }
        if (options.isChangeVideoResolution &&
                options.changeVideoResolutionWidth  != -1 &&
                options.changeVideoResolutionHeight != -1) {
            rtc.changeVideoResolution(options.changeVideoResolutionWidth, options.changeVideoResolutionHeight);
        }
        return rtc;
    }

    /**
     * Must be called in Activity.onCreate, otherwise can not get surface size accurately.
     * @param localSurface
     * @param oppositeSurface
     */
    public synchronized void setSurfaceView(EMLocalSurfaceView localSurface, EMOppositeSurfaceView oppositeSurface) {
        if (mRtcConnection == null) {
            mRtcConnection = createRtcConnection("rtc");
        }
        mRtcConnection.setViews(localSurface.getRenderer(), oppositeSurface.getRenderer());
        mRtcConnection.setRtcCameraDataProcessor(mProcessorDelegate);
    }

    /**
     * 接听通话
     * @throws EMNoActiveCallException
     */
    public void answerCall() throws EMNoActiveCallException {
        synchronized (this) {
            if (currentSession == null) {
                throw new EMNoActiveCallException("no incoming active call");
            }
        }
        if (callState.isRinging() == false) {
            throw new EMNoActiveCallException("Current callstate is not ringing callState:" + callState.getState());
        }
        EMAError error = new EMAError();
        emaObject.answerCall(currentSession.getCallId(), error);
        synchronized (this) {
            changeState(CallState.ANSWERING);
            if (error.errCode() != EMAError.EM_NO_ERROR) {
                EMLog.d(TAG, "errorCode:" + error.errCode());
//                changeState(CallState.DISCONNECTED, getCallError(error));
//                currentSession = null;
                endCall();
            }
        }
    }

    /**
     * 拒绝接听
     * @throws EMNoActiveCallException
     */
    public void rejectCall() throws EMNoActiveCallException {
        final EMCallSession session = currentSession;
        if (session == null) {
            EMLog.e(TAG, "no incoming active call");
            throw new EMNoActiveCallException("no incoming active call");
        }
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                emaObject.endCall(session.getCallId(), callState.isRinging() ?
                        EMACallSession.EndReason.REJECT : EMACallSession.EndReason.HANGUP);

            }
        });
        synchronized (this) {
            // onRecvCallEnded will also trigger changeState(CallState.DISCONNECT)
            //changeState(CallState.DISCONNECTED);
            mRtcConnection = null;
            mRtcListener = null;
        }
    }

    /**
     * 挂断通话
     * @throws EMNoActiveCallException
     */
    public void endCall() throws EMNoActiveCallException {
        String _callId = "";
        synchronized (this) {
            if (currentSession == null) {
                EMLog.e(TAG, "no incoming active call");
                throw new EMNoActiveCallException("no incoming active call");
            }
            _callId = currentSession.getCallId();
        }
        final String callId = _callId;
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                emaObject.endCall(callId, EMACallSession.EndReason.HANGUP);
            }
        });
        synchronized (this) {
            // onRecvCallEnded will also trigger changeState(CallState.DISCONNECT)
            // changeState(CallState.DISCONNECTED);
            mRtcConnection = null;
            mRtcListener = null;
        }
    }

    /**
     * 返回当前通话时是否为P2P直连
     *
     * check if that's a P2P call
     * @return
     */
    public boolean isDirectCall() {
        RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
        if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
            EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
            RtcStatistics statistics = lis.getStatistics();
            if (statistics != null && statistics.connectionType != null) {
                return statistics.connectionType.equals("direct");
            }
        }
        return true;
    }

    /**
     * 实时通话时暂停语音数据传输
     */
    public void pauseVoiceTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.setMute(true);
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.PAUSE_VOICE, error);
            handleError(error);
        }
    }

    /**
     * 实时通话时恢复语音数据传输
     */
    public void resumeVoiceTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.setMute(false);
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.RESUME_VOICE, error);
            handleError(error);
        }
    }

    /**
     * 实时通话时停止视频数据传输
     */
    public void pauseVideoTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.stopCapture();
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.PAUSE_VIDEO, error);
            handleError(error);
        }
    }

    /**
     * 实时通话时恢复视频数据传输
     */
    public void resumeVideoTransfer() throws HyphenateException {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.startCapture();
        }
        EMCallSession session = currentSession;
        if (session != null) {
            EMAError error = new EMAError();
            emaObject.updateCall(session.getCallId(), EMACallSession.StreamControlType.RESUME_VOICE, error);
            handleError(error);
        }
    }

    /*!
    * \~chinese
    * 开启相机拍摄
    * @param facing: 参数可以是CameraInfo.CAMERA_FACING_BACK, 或者CameraInfo.CAMERA_FACING_FRONT
    * @HyphenateException: 如果cameraIndex不是CameraInfo.CAMERA_FACING_BACK，也不是CameraInfo.CAMERA_FACING_FRONT，会抛出异常.
    *
    * \~english
    * start camera capture
    * @param facing: can be CameraInfo.CAMERA_FACING_BACK, CameraInfo.CAMERA_FACING_FRONT
    * @HyphenateException: if cameraIndex not in CameraInfo.CAMERA_FACING_BACK, 或者CameraInfo.CAMERA_FACING_FRONT, got the exception.
    */
    public void setCameraFacing(int facing) throws HyphenateException {
        if (facing != CameraInfo.CAMERA_FACING_BACK && facing != CameraInfo.CAMERA_FACING_FRONT) {
            throw new HyphenateException(EMError.CALL_INVALID_CAMERA_INDEX, "Invalid camera index");
        }
        RtcConnection.setCameraFacing(facing);
    }

    public synchronized void switchCamera() {
        RtcConnection rtc = mRtcConnection;
        if (rtc != null) {
            rtc.switchCamera();
        }
    }

    //===================== conference interface =====================

    // Boolean flag show whether this renderer already has bind to RtcConnection
    // the first item is local render localView
    static class ViewRendererBinder {
        public ViewRendererBinder(String rtcName) {
//        public ViewRendererBinder(VideoViewRenderer r, VideoView localView) {
            this.rtc = EMCallManager.createRtcConnection(rtcName);
//            this.oppositeRenderer = r;
//            this.localView = localView;
        }
        boolean bindRtc = false;
        VideoViewRenderer localRenderer = null;
        VideoViewRenderer oppositeRenderer = null;
        VideoView localView = null;
        VideoView oppositeView = null;
        RtcConnection rtc = null;
        EMCallStream callStream;
    }

    List<ViewRendererBinder> mViewBinders = Collections.synchronizedList(new ArrayList<ViewRendererBinder>());
    boolean mFirstOppositeRenderer = true;

    @Deprecated
    public void clearView() {
    }

    /**
     * 必须在UI线程中执行Activity.onCreate
     * must be executed in Activity.onCreate(Context context)
     * @param localSurface
     */
    void setLocalView(EMLocalSurfaceView localSurface) {
        ViewRendererBinder localBinder = new ViewRendererBinder("local");
        localBinder.localRenderer = localSurface.getRenderer();
        localBinder.localView = localSurface;
        localBinder.rtc.setViews(localSurface.getRenderer(), null);

        mViewBinders.add(localBinder);
    }

    /**
     * 必须在UI线程中执行Activity.onCreate
     * must be executed in Activity.onCreate(Context context)
     * @param viewList
     */
    void setOppositeViews(List<EMOppositeSurfaceView> viewList) {
        //================ test begin
        //mViewBinders.get(0).rtc.setViews(mLocalRenderer, viewList.get(0).getRenderer());
        //================ test end

        /*
        if (viewList == null || viewList.size() == 0) {
            return;
        }
        mFirstOppositeRenderer = true;
        mViewBinders.clear();

        // set local
        ViewRendererBinder localBinder = new ViewRendererBinder(mLocalRenderer, null);
        localBinder.rtc.setViews(mLocalRenderer, null);
        mViewBinders.add(localBinder);

        // set opposite views
        int  i = 0;
        for (EMOppositeSurfaceView localView : viewList) {
            ViewRendererBinder binder = new ViewRendererBinder(new VideoViewRenderer(localView, "oppositeView:" + i), localView);
//            binder.rtc.setViews(null, mViewBinders.get(i).renderer);
            mViewBinders.add(binder);
            i++;
        }
        */
    }

    public void clearRenderView() {
        mViewBinders.clear();
    }

    ViewRendererBinder getNextAvailableRenderer(String to) {
        synchronized (mViewBinders) {
//            //if (to.equals(EMClient.getInstance().getCurrentUser())) {
//            if (to.equals("easemob.com")) {
//                if (mViewBinders.size() > 0) {
//                    return mViewBinders.get(0);
//                } else {
//                    return null;
//                }
//            }
            for (int i = 0; i < mViewBinders.size(); i++) {
                ViewRendererBinder p = mViewBinders.get(i);
                if (p.bindRtc == false) {
                    return p;
                }
            }
        }
        return null;
    }

    ViewRendererBinder getRenderer(EMACallStream callStream) {
        if (callStream == null) {
            return null;
        }

        /*
        if (mFirstOppositeRenderer) {
            if (mViewBinders.size() > 0) {
                mFirstOppositeRenderer = false;
                return mViewBinders.get(0);
            }
        }
        */
        EMLog.d(TAG, "callStream.username:" + callStream.getUserName());
        synchronized (mViewBinders) {
            if (callStream.getUserName().equals(EMClient.getInstance().getCurrentUser()) && mViewBinders.size() > 0) {
                return mViewBinders.get(0);
            }
            for (ViewRendererBinder renderer : mViewBinders) {
                if (renderer != null && renderer.callStream != null && renderer.callStream.emaObject.equals(callStream)) {
                    return renderer;
                }
            }
        }
        return null;
    }

    void addConferenceListener(EMConferenceListener listener) {
        synchronized (mConferenceListeners) {
            if (!mConferenceListeners.contains(listener)) {
                mConferenceListeners.add(listener);
            }
        }
    }

    void removeConferenceListener(EMConferenceListener listener) {
        synchronized (mConferenceListeners) {
            if (mConferenceListeners.contains(listener)) {
                mConferenceListeners.remove(listener);
            }
        }
    }

    private void handleError(EMAError error)  throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            EMLog.e(TAG, "error code:" + error.errCode() + " errorMsg:" + error.errMsg());
            throw new HyphenateException(error);
        }
    }

    public enum EMCallType {
        VOICE,
        VIDEO,
    }

    EMCallConference createConference(EMCallType type, String password) throws HyphenateException {
        mFirstOppositeRenderer = true;
        EMAError error = new EMAError();
        EMCallConference conference = new EMCallConference(emaObject.createConference(type.ordinal(), password, error));
        synchronized (this) {
            currentConference = conference;
        }
        handleError(error);
        return conference;
    }

    EMCallConference createAndJoinConference(EMCallType type, String password) throws HyphenateException {
        mFirstOppositeRenderer = true;
        EMAError error = new EMAError();
        EMCallConference conference = new EMCallConference(emaObject.createAndJoinConference(type.ordinal(), password, error));
        synchronized (this) {
            currentConference = conference;
        }
        handleError(error);
        return conference;
    }

    void deleteConference(String callId) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.deleteConference(callId, error);
        handleError(error);

        // release all resource
        for (ViewRendererBinder binder : mViewBinders) {
            if (binder.localView instanceof EMOppositeSurfaceView) {
                ((EMOppositeSurfaceView)binder.localView).release();
            }
        }
        mFirstOppositeRenderer = true;
        mViewBinders.clear();
    }

    EMCallConference joinConference(String confId, String token) throws HyphenateException {
        EMAError error = new EMAError();
        EMCallConference conference = new EMCallConference(emaObject.joinConference(confId, token, error));
        synchronized (this) {
            currentConference = conference;
        }
        handleError(error);
        return currentConference;
    }

    void leaveConference(String callId) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.leaveConference(callId, error);
        handleError(error);
    }

    void asyncPublishConferenceStream(String callId) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.asyncPublishConferenceStream(callId, error);
        handleError(error);
    }

    void asyncSubscribeConferenceStream(String callId, EMCallStream callStream, EMOppositeSurfaceView oppositeView) throws HyphenateException {
        if (callId == null || callStream == null || oppositeView == null) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "null pointer exception");
        }

        EMLog.d(TAG, "asyncSubscribeConferenceStream:" + oppositeView);

        for (ViewRendererBinder render : mViewBinders) {
            if (render.oppositeView == oppositeView) {
                EMLog.d(TAG, "asyncSubscribeConferenceStream oppositeView already in used");
                throw new HyphenateException(EMError.GENERAL_ERROR, "asyncSubscribeConferenceStream oppositeView already in used");
            }
        }
        ViewRendererBinder binder;
        if (mFirstOppositeRenderer && mViewBinders.size() > 0) {
            binder = mViewBinders.get(0);

            // same
            binder.oppositeView = oppositeView;
            binder.oppositeRenderer = oppositeView.getRenderer();
            binder.callStream = callStream;

            EMLog.d(TAG, "asyncSubscribeConferenceStream setViews 1");
            binder.rtc.setViews(binder.localRenderer, oppositeView.getRenderer());
            mFirstOppositeRenderer = false;
        } else {
            binder = new ViewRendererBinder("remote:" + callStream.getUserName());

            // same
            binder.oppositeView = oppositeView;
            binder.oppositeRenderer = oppositeView.getRenderer();
            binder.callStream = callStream;

            EMLog.d(TAG, "asyncSubscribeConferenceStream setViews 2");
            binder.rtc.setViews(null, oppositeView.getRenderer());
            mViewBinders.add(binder);
        }

        EMAError error = new EMAError();
        emaObject.asyncSubscribeConferenceStream(callId, callStream.emaObject, error);
        handleError(error);
    }

    List<String> getConferenceMembersFromServer(String callId) throws HyphenateException {
        EMAError error = new EMAError();
        List<String> members = emaObject.getConferenceMembersFromServer(callId, error);
        handleError(error);
        return members;
    }

    void deleteConferenceStream(String callId, String remoteName) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.deleteConference(callId, remoteName, error);
        handleError(error);
    }

    /**
     * \~chinease
     * 获取当前正在使用的摄像头
     * @return 值可以是
     */
    public int getCameraFacing() {
        return RtcConnection.getCameraFacing();
    }

    //=====================end of public interface=====================
    HandlerThread stateChangeHandlerThread = new HandlerThread("CallStateHandlerThread");
    { stateChangeHandlerThread.start(); }
    Handler stateChangeHandler = new Handler(stateChangeHandlerThread.getLooper()) {

        @Override
        public void handleMessage(Message msg) {
            @SuppressWarnings("unchecked")
            Pair<CallState, CallError> pair = (Pair<CallState, CallError>)msg.obj;
            CallState state = pair.first;
            CallError error = pair.second;
            EMLog.d(TAG, "stateChangeHandler handleMessage BEGIN ---- state:" + state);
            notifyCallStateChanged(state, error);
            EMLog.d(TAG, "stateChangeHandler handleMessage  END  ----");
        }
    };

    private void notifyCallStateChanged(CallState callState, CallError callError){
        synchronized (callListeners) {
            for (EMCallStateChangeListener listener : callListeners) {
                listener.onCallStateChanged(callState, callError);
            }
        }
    }

    void changeState(CallState state) {
        changeState(state, CallError.ERROR_NONE);
    }

    protected void changeState(final CallState state, final CallError callError){
        EMLog.d(TAG, "changeState:" + state);
//        if(currentSession == null || currentSession.getCallId() == null) {
//            EMLog.d(TAG, "currentSession: null");
//            return;
//        }
        this.callState.changeState(state);
        stateChangeHandler.sendMessage(stateChangeHandler.obtainMessage(0, new Pair<CallState, CallError>(state, callError)));
    }

    void clearStateMessages() {
        stateChangeHandler.removeMessages(0);
    }

    class EMACallListenerDelegate extends EMACallManagerListener {

        @Override
        public void onSendPushMessage(String from, String to) {
            EMCallPushProvider pushProvider = EMCallManager.this.mPushProvider;
            if (pushProvider == null) {
                defaultProvider.onRemoteOffline(to);
            } else {
                pushProvider.onRemoteOffline(to);
            }
        }

        @Override
        public void onRecvCallFeatureUnsupported(EMACallSession session, EMAError error) {
            EMLog.d(TAG, "onRecvCallFeatureUnsupported, callId:" + session.getCallId());
        }

        @Override
        public void onRecvCallIncoming(EMACallSession session) {
            EMLog.d(TAG, "onRecvSessionRemoteInitiate");
            synchronized (EMCallManager.this) {
                currentSession = new EMCallSession(session);
                changeState(CallState.RINGING);
            }

            Intent intent = new Intent(getIncomingCallBroadcastAction());
            intent.putExtra("type", currentSession.getType() == EMCallSession.Type.VIDEO ? "video" : "voice");
            intent.putExtra("from", currentSession.getRemoteName());
            intent.putExtra("to", EMClient.getInstance().getCurrentUser());
            EMClient.getInstance().getContext().sendBroadcast(intent);
        }

        @Override
        public void onRecvCallConnected(EMACallSession session) {
            EMLog.d(TAG, "onRecvSessionConnected");
            synchronized (EMCallManager.this) {
                if (currentSession == null) {
                    currentSession = new EMCallSession(session);
                }
                changeState(CallState.CONNECTED);
            }
        }

        @Override
        public void onRecvCallAccepted(EMACallSession session) {
            EMLog.d(TAG, "onReceiveCallAccepted");
            synchronized (EMCallManager.this) {
                if (currentSession == null) {
                    currentSession = new EMCallSession(session);
                }
                EMLog.d(TAG, "onReceiveCallAccepted");
                changeState(CallState.ACCEPTED);
            }
        }


        CallError endReasonToCallError(EMCallSession.EndReason reason, EMAError emaError){
            CallError error = CallError.ERROR_NONE;
            switch (reason) {
                case HANGUP:
                    error = CallError.ERROR_NONE;
                    break;
                case NORESPONSE:
                    error = CallError.ERROR_NORESPONSE;
                    break;
                case REJECT:
                    error = CallError.REJECTED;
                    break;
                case BUSY:
                    error = CallError.ERROR_BUSY;
                    break;
                case FAIL:
                    error = CallError.ERROR_TRANSPORT;
                    if (emaError.errCode() != EMAError.EM_NO_ERROR) {
                        if (emaError.errCode() == EMAError.CALL_REMOTE_OFFLINE) {
                            error = CallError.ERROR_UNAVAILABLE;
                        } else if (emaError.errCode() == EMAError.CALL_BUSY) {
                            error = CallError.ERROR_UNAVAILABLE;
                        } else if (emaError.errCode() == EMAError.CALL_CONNECTION_FAILED) {
                            error = CallError.ERROR_TRANSPORT;
                        }
                    }
                    break;
                case OFFLINE:
                    error = CallError.ERROR_UNAVAILABLE;
                    break;
                default:
                    break;
            }
            return error;
        }

        @Override
        public void onRecvCallEnded(EMACallSession callSession, int reasonOrdinal, EMAError error) {
            EMLog.d(TAG, "onReceiveCallTerminated, reasonOrdinal: " + reasonOrdinal);
            synchronized (EMCallManager.this) {
                if (currentSession != null) {
                    currentSession = null;
                }
//              mRtcMap.clear();
                changeState(CallState.DISCONNECTED, endReasonToCallError(EMCallSession.getEndReason(reasonOrdinal), error));
                mRtcConnection = null;
                mRtcListener = null;
            }
        }

        @Override
        public void onRecvCallNetworkStatusChanged(EMACallSession callSession, int toStatus) {
            EMLog.d(TAG, "onRecvCallNetworkStatusChanged, callId: " + callSession.getCallId() + " toStatus:" + toStatus);
            CallState callState = CallState.DISCONNECTED;

            if (toStatus == EMACallSession.NetworkStatus.CONNECTED.ordinal()) {
                callState = CallState.NETWORK_NORMAL;
            } else if (toStatus == EMACallSession.NetworkStatus.UNSTABLE.ordinal()) {
                // ONESDK can not distinguish between unstable and disconnected
                callState = CallState.NETWORK_UNSTABLE;
            } else if (toStatus == EMACallSession.NetworkStatus.DISCONNECTED.ordinal()) {
                callState = CallState.NETWORK_DISCONNECTED;
            } else {
                try {
                    throw new HyphenateException("onRecvCallNetworkStatusChanged invalid toStatus:" + toStatus);
                } catch (HyphenateException e) {
                    e.printStackTrace();
                }
            }

            if (EMCallManager.this.callState.equals(callState)) {
                EMLog.d(TAG, "onRecvCallNetworkStatusChanged toStatus equals to current callState");
                return;
            }
            changeState(callState);
        }

        @Override
        public void onRecvCallStateChanged(EMACallSession callSession, int streamControlType) {
            EMLog.d(TAG, "onRecvCallStateChanged, callId: " + callSession.getCallId() + " StreamControlType:" + streamControlType);

            CallState callState = CallState.DISCONNECTED;

            if (streamControlType == EMACallSession.StreamControlType.PAUSE_VIDEO.ordinal()) {
                callState = CallState.VIDEO_PAUSE;
            } else if (streamControlType == EMACallSession.StreamControlType.PAUSE_VOICE.ordinal()) {
                callState = CallState.VOICE_PAUSE;
            } else if (streamControlType == EMACallSession.StreamControlType.RESUME_VIDEO.ordinal()) {
                callState = CallState.VIDEO_RESUME;
            } else if (streamControlType == EMACallSession.StreamControlType.RESUME_VOICE.ordinal()) {
                callState = CallState.VOICE_RESUME;
            } else {
                try {
                    throw new HyphenateException("onRecvCallStateChanged invalid streamControlType:" + streamControlType);
                } catch (HyphenateException e) {
                    e.printStackTrace();
                }
            }
            if (EMCallManager.this.callState.equals(callState)) {
                EMLog.d(TAG, "onRecvCallStateChanged toStatus equals to current callState");
                return;
            }
            changeState(callState);
        }


        @Override
        public void onNewRtcConnection(String callId, int mode, EMACallStream callStream, String to,
                                       RtcConnection.Listener listener, EMACallRtcImpl rtcImpl ) {

            EMLog.d(TAG, "onNewRtcConnection, mode:" + mode + " remoteName: " + to);

            if (rtcImpl == null || listener == null) {
                return;
            }

            if (mode == EMACallSession.Mode.CONFERENCE.ordinal()) {
                ViewRendererBinder binder = getRenderer(callStream);
//                ViewRendererBinder binder = mViewBinders.get(0);
                if (binder != null) {
                    EMLog.d(TAG, "get new View rtc binder");
                    binder.bindRtc = true;
                    binder.rtc.setListener(listener);
//                    binder.rtc.setViews(null, binder.renderer);
                    rtcImpl.setRtcConnection(EMCallManager.this, binder.rtc);
                } else {
                    EMLog.e(TAG, "can not find ViewRendererBinder");
                }
            } else {
                synchronized (EMCallManager.this) {
                    if (mRtcConnection == null) {
                        mRtcConnection = EMCallManager.createRtcConnection("rtc:" + to);
                        mRtcConnection.setRtcCameraDataProcessor(mProcessorDelegate);
                    }
                    mRtcListener = listener;
                    mRtcConnection.setListener(listener);
                    rtcImpl.setRtcConnection(EMCallManager.this, mRtcConnection);
                }
            }
        }

        @Override
        public void onConferenceMemberJoined(EMACallConference callConference, String enteredName) {
            EMLog.d(TAG, "onConferenceMemberEntered sessionId:" + callConference.getCallId() + " enteredName:" + enteredName);
            synchronized (mConferenceListeners) {
                for (EMConferenceListener listener : mConferenceListeners) {
                    listener.onConferenceMemberEntered(callConference.getCallId(), enteredName);
                }
            }
        }

        @Override
        public void onConferenceMemberLeaved(EMACallConference callConference, String exitedName) {
            EMLog.d(TAG, "onConferenceMemberExited sessionId:" + callConference.getCallId() + " leavedName:" + exitedName);
            synchronized (mConferenceListeners) {
                for (EMConferenceListener listener : mConferenceListeners) {
                    listener.onConferenceMemberExited(callConference.getCallId(), exitedName);
                }
            }
        }

        @Override
        public void onConferenceMemberPublished(EMACallConference callConference, String pubedName) {
            EMLog.d(TAG, "onConferenceMemberPublished sessionId:" + callConference.getCallId() + " publishedName:" + pubedName);
            synchronized (mConferenceListeners) {
                for (EMConferenceListener listener : mConferenceListeners) {
                    listener.onConferenceMemberPublished(callConference.getCallId(), pubedName);
                }
            }
        }

        @Override
        public void onConferenceMembersUpdated(EMACallConference callConference) {
            EMLog.d(TAG, "onConferenceMembersUpdated sessionId:" + callConference.getCallId());
            synchronized (mConferenceListeners) {
                for (EMConferenceListener listener : mConferenceListeners) {
                    listener.onConferenceMembersUpdated(callConference.getCallId());
                }
            }
        }

        @Override
        public void onConferenceClosed(EMACallConference callConference) {
            EMLog.d(TAG, "onConferenceClosed sessionId:" + callConference.getCallId());
            synchronized (mConferenceListeners) {
                for (EMConferenceListener listener : mConferenceListeners) {
                    listener.onConferenceClosed(callConference.getCallId());
                }
            }
            for (ViewRendererBinder binder : mViewBinders) {
                if (binder.localView instanceof EMOppositeSurfaceView) {
                    ((EMOppositeSurfaceView)binder.localView).release();
                }
            }
            mViewBinders.clear();
            mFirstOppositeRenderer = false;
        }

    }

    public static class EMVideoCallHelper {
        private EMVideoCallHelper() {}

        public enum CallType {
            /**
             * 音频通话
             */
            audio,
            /**
             * 视频通话
             */
            video;
        }

        /**
         * Capture video image to local storage
         * @param filename
         * @return true for success
         */
        public boolean takePicture(String filename) {
            return EMClient.getInstance().callManager().emaObject.capturePicture(filename);
        }

        /**
         * 开始视频录制
         * @param dirPath 录制的文件存储目录
         * @return 录制文件的路径，用户可以自己实时检测该文件大小，如果占用太多空间提醒用户
         */
        public void startVideoRecord(String dirPath) {
            EMClient.getInstance().callManager().emaObject.startRecordVideo(dirPath);
        }

        /**
         * 停止视频录制
         * @return 返回录制的视频文件的路径，录制失败则返回null
         */
        public String stopVideoRecord() {
            return EMClient.getInstance().callManager().emaObject.stopRecordVideo();
        }

        /**
         * This function is only meaningful when your app need recording feature
         * If not, remove it.
         * This function need be called before the video stream started, so we set it in onCreate function.
         * This method will set the preferred video record encoding codec.
         * Using default encoding format, recorded file can not be played by mobile player.
         */
        public void setPreferMovFormatEnable(boolean enabled){
            if(enabled)
            {
                RtcConnection.setGlobalVideoCodec(RtcConnection.RtcConstStringH264);
            }
            else
            {
                RtcConnection.setGlobalVideoCodec(null);
            }
        }

        /**
         * //时延，单位是ms
         * @return
         */
        @Deprecated
        public int getVideoLatency(){
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.localVideoRtt;
                }
            }
            return 0;
        }

        /**
         * 帧率
         * @return
         */
        @Deprecated
        public int getVideoFrameRate() {
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.remoteFps;
                }
            }
            return 0;
        }

        /**
         * 每一百个包中丢包个数
         * @return
         */
        @Deprecated
        public int getVideoLostRate(){
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.remoteVideoPacketsLostrate;
                }
            }
            return 0;
        }

        /**
         * 对方图像宽度
         * @return 获取失败，返回0
         */
        public int getVideoWidth(){
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.remoteWidth;
                }
            }
            return 0;
        }

        /**
         * 对方图像高度
         * @return 获取失败，返回-1
         */
        public int getVideoHeight(){
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.remoteHeight;
                }
            }
            return 0;
        }

        /**
         * 接收视频比特率，单位为kbps
         * @return
         */
        @Deprecated
        public int getRemoteBitrate(){
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.remoteVideoBps;
                }
            }
            return 0;
        }

        //dep, instead
        /**
         * 发送视频比特率，单位为kbps
         * @return
         */
        @Deprecated
        public int getLocalBitrate(){
            RtcConnection.Listener listener = EMClient.getInstance().callManager().mRtcListener;
            if (listener != null && listener instanceof EMACallRtcListenerDelegate) {
                EMACallRtcListenerDelegate lis = (EMACallRtcListenerDelegate)listener;
                RtcStatistics statistics = lis.getStatistics();
                if (statistics != null) {
                    return statistics.localVideoActualBps;
                }
            }
            return 0;
        }

    }

    void onLogout() {
        stateChangeHandler.removeMessages(0);
        // TODO:
    }

    void printStackTrace() {
        try {
            throw new Exception();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public EMCallOptions getCallOptions() {
        if (callOptions == null) {
            callOptions = new EMCallOptions(emaObject);
        }
        return callOptions;
    }

    public EMCallSession getCurrentCallSession() {
        return currentSession;
    }
}
