package com.hippo.utils;

import android.os.Handler;
import android.text.Editable;
import android.text.TextWatcher;
import android.widget.EditText;

/**
 * Helper class for performing efficient network suggestions with SearchView
 *
 * @author Gurmail
 */
public class SearchViewNetworkQueryHelper implements TextWatcher {

    @Override
    public void beforeTextChanged(CharSequence s, int start, int count, int after) {

    }

    @Override
    public void onTextChanged(CharSequence s, int start, int before, int count) {

    }

    @Override
    public void afterTextChanged(Editable query) {
        if (mSearchCallbacks != null && query != null && query.length() > 0) {
            mSearchCallbacks.showLoader();
        }
        removeAnyCallbacks();
        if (query != null && query.length() >= mSuggestCountThreshold) {
            mPerformSearchRunnable = searchClickRunnable(query.toString());
            mHandler.postDelayed(mPerformSearchRunnable, mSuggestWaitThreshold);
        } else if (mSearchCallbacks != null && query != null && query.length() <= mSuggestCountThreshold) {
            mSearchCallbacks.showAllData();
        }
    }

    public static interface SearchCallbacks {
        public void showLoader();

        public void performQuery(String query);

        public void showAllData();
    }

    /**
     * SearchView to which this helper is set
     */
    private EditText mEditTextView;

    /**
     * {@link SearchCallbacks} implementation to perform the actual network
     * query
     */
    private SearchCallbacks mSearchCallbacks;

    private Handler mHandler;

    private Runnable mPerformSearchRunnable;

    /**
     * Minimum number of characters that need to be entered before the network
     * search is triggered
     */
    private int mSuggestCountThreshold;

    /**
     * Minimum duration we need to wait after the user has stopped typing to
     * actually perform the search
     */
    private int mSuggestWaitThreshold;


    public SearchViewNetworkQueryHelper(EditText searchView, SearchCallbacks searchCallbacks) {
        mEditTextView = searchView;
        mSearchCallbacks = searchCallbacks;
        mHandler = new Handler();

        mEditTextView.addTextChangedListener(this);
    }

    public void removeTextChanged() {
        mEditTextView.removeTextChangedListener(this);
    }

    public void addTextChanged() {
        mEditTextView.addTextChangedListener(this);
    }

    public int getSuggestCountThreshold() {
        return mSuggestCountThreshold;
    }

    public void setSuggestCountThreshold(final int suggestCountThreshold) {
        mSuggestCountThreshold = suggestCountThreshold;
    }

    public int getSuggestWaitThreshold() {
        return mSuggestWaitThreshold;
    }

    public void setSuggestWaitThreshold(final int suggestWaitThreshold) {
        mSuggestWaitThreshold = suggestWaitThreshold;
    }

    /**
     * Removes any pending callbacks(if any) from the handler
     */
    private void removeAnyCallbacks() {
        if (mPerformSearchRunnable != null) {
            mHandler.removeCallbacks(mPerformSearchRunnable);
        }
    }


    /**
     * Creates a runnable for perfoming a search query
     *
     * @param query The query to search for
     * @return a {@link Runnable} for performing a search request
     */
    private Runnable searchClickRunnable(final String query) {
        return new Runnable() {

            @Override
            public void run() {
                if (mSearchCallbacks != null) {
                    mSearchCallbacks.performQuery(query);
                }
            }
        };
    }


}
