package com.hippo.utils.showmoretextview;

import android.animation.LayoutTransition;
import android.content.Context;
import android.graphics.Color;
import android.os.Build;
import android.os.Handler;
import android.text.SpannableString;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.TextPaint;
import android.text.method.LinkMovementMethod;
import android.text.style.ClickableSpan;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

/**
 * Created by ${Deven} on 6/1/18.
 */
public class ReadMoreOption {
    public static final int TYPE_LINE = 1;
    public static final int TYPE_CHARACTER = 2;
    private static final String TAG = ReadMoreOption.class.getSimpleName();
    // required
    private Context context;
    // optional
    private int textLength;
    private int textLengthType;
    private String moreLabel;
    private String lessLabel;
    private int moreLabelColor;
    private int lessLabelColor;
    private boolean labelUnderLine;
    private boolean expandAnimation;
    private OptionClick optionClick;
    private int position;
    private int status;

    private ReadMoreOption(Builder builder) {
        this.context = builder.context;
        this.textLength = builder.textLength;
        this.textLengthType = builder.textLengthType;
        this.moreLabel = builder.moreLabel;
        this.lessLabel = builder.lessLabel;
        this.moreLabelColor = builder.moreLabelColor;
        this.lessLabelColor = builder.lessLabelColor;
        this.labelUnderLine = builder.labelUnderLine;
        this.expandAnimation = builder.expandAnimation;
        this.optionClick = builder.optionClick;
        this.position = builder.position;
        this.status = builder.status;
    }

    public void addReadLessTo(final TextView textView, final CharSequence text) {
        textView.post(new Runnable() {
            @Override
            public void run() {
                textView.setMaxLines(Integer.MAX_VALUE);
                SpannableStringBuilder spannableStringBuilder = new SpannableStringBuilder(text)
                        .append("...")
                        .append(lessLabel);
                SpannableString ss = SpannableString.valueOf(spannableStringBuilder);
                ClickableSpan clickableSpan = new ClickableSpan() {
                    @Override
                    public void onClick(View view) {
                        new Handler().post(new Runnable() {
                            @Override
                            public void run() {
                                addReadMoreTo(textView, text);
                                if(optionClick != null)
                                    optionClick.onReadLessClickListener(position);
                            }
                        });
                    }

                    @Override
                    public void updateDrawState(TextPaint ds) {
                        super.updateDrawState(ds);
                        ds.setUnderlineText(labelUnderLine);
                        ds.setColor(lessLabelColor);
                    }
                };
                ss.setSpan(clickableSpan, ss.length() - lessLabel.length(), ss.length(), Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                textView.setText(ss);
                textView.setMovementMethod(LinkMovementMethod.getInstance());
            }
        });
    }
    public void addReadMoreTo(final TextView textView, final CharSequence text) {
        if (textLengthType == TYPE_CHARACTER) {
            if (text.length() <= textLength) {
                textView.setText(text);
                return;
            }
        } else {
            // If TYPE_LINE
            textView.setLines(textLength);
            textView.setText(text);
        }
        textView.post(new Runnable() {
            @Override
            public void run() {
                int textLengthNew = textLength;
                SpannableStringBuilder spannableStringBuilder;
                if (textLengthType == TYPE_LINE) {
                    if (textView.getLayout().getLineCount() <= textLength) {
                        textView.setText(text);
                        return;
                    }
                    ViewGroup.MarginLayoutParams lp = (ViewGroup.MarginLayoutParams) textView.getLayoutParams();
                    //get start index for last line
                    final int startIndex = textView.getLayout().getLineStart(textLength - 1);
                    //get end index for last line
                    final int endIndex = textView.getLayout().getLineEnd(textLength - 1);
                    //check char count for last line
                    final int charCount = endIndex - startIndex;
                    //more label length
                    int moreLabelLength = (moreLabel.length() + 4 + lp.rightMargin / 6);
                    //check if char count is greater than or equals more label length.
                    if (charCount >= moreLabelLength) {
                        //get substring for no of max lines and append more label
                        CharSequence subString = text.subSequence(textView.getLayout().getLineStart(0),
                                textView.getLayout().getLineEnd(textLength - 1));
                        textLengthNew = subString.length() - (moreLabel.length() + 4 + lp.rightMargin / 6);
                    } else {
                        //get substring for ( no of max lines - 1)
                        CharSequence subString = text.subSequence(textView.getLayout().getLineStart(0),
                                textView.getLayout().getLineStart(textLength - 1));
                        textLengthNew = subString.length();
                    }
                }


                //get subsequence for given length and append label
                spannableStringBuilder = new SpannableStringBuilder(text.subSequence(0, textLengthNew))
                        .append("...")
                        .append(moreLabel);
                SpannableString ss = SpannableString.valueOf(spannableStringBuilder);
                ClickableSpan clickableSpan = new ClickableSpan() {
                    @Override
                    public void onClick(View view) {
                        addReadLess(textView, text);
                        if(optionClick != null)
                            optionClick.onReadMoreClickListener(position);
                    }

                    @Override
                    public void updateDrawState(TextPaint ds) {
                        super.updateDrawState(ds);
                        ds.setUnderlineText(labelUnderLine);
                        ds.setColor(moreLabelColor);
                    }
                };
                ss.setSpan(clickableSpan, ss.length() - moreLabel.length(), ss.length(), Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN && expandAnimation) {
                    LayoutTransition layoutTransition = new LayoutTransition();
                    layoutTransition.enableTransitionType(LayoutTransition.CHANGING);
                    ((ViewGroup) textView.getParent()).setLayoutTransition(layoutTransition);
                }
                textView.setText(ss);
                textView.setMovementMethod(LinkMovementMethod.getInstance());
            }
        });
    }

    private void addReadLess(final TextView textView, final CharSequence text) {
        textView.setMaxLines(Integer.MAX_VALUE);
        SpannableStringBuilder spannableStringBuilder = new SpannableStringBuilder(text)
                .append("...")
                .append(lessLabel);
        SpannableString ss = SpannableString.valueOf(spannableStringBuilder);
        ClickableSpan clickableSpan = new ClickableSpan() {
            @Override
            public void onClick(View view) {
                new Handler().post(new Runnable() {
                    @Override
                    public void run() {
                        addReadMoreTo(textView, text);
                        if(optionClick != null)
                            optionClick.onReadLessClickListener(position);
                    }
                });
            }

            @Override
            public void updateDrawState(TextPaint ds) {
                super.updateDrawState(ds);
                ds.setUnderlineText(labelUnderLine);
                ds.setColor(lessLabelColor);
            }
        };
        ss.setSpan(clickableSpan, ss.length() - lessLabel.length(), ss.length(), Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
        textView.setText(ss);
        textView.setMovementMethod(LinkMovementMethod.getInstance());
    }

    public static class Builder {
        // required
        private Context context;
        // optional
        private int textLength = 100;
        private int textLengthType = ReadMoreOption.TYPE_CHARACTER;
        private String moreLabel = "read more";
        private String lessLabel = "read less";
        private int moreLabelColor = Color.parseColor("#ff00ff");
        private int lessLabelColor = Color.parseColor("#ff00ff");
        private boolean labelUnderLine = false;
        private boolean expandAnimation = false;
        private int status;
        private OptionClick optionClick;
        private int position;

        public Builder(Context context) {
            this.context = context;
        }

        /**
         * @param length         can be no. of line OR no. of characters - default is 100 character
         * @param textLengthType ReadMoreOption.TYPE_LINE for no. of line OR
         *                       ReadMoreOption.TYPE_CHARACTER for no. of character
         *                       - default is ReadMoreOption.TYPE_CHARACTER
         * @return Builder obj
         */
        public Builder textLength(int length, int textLengthType) {
            this.textLength = length;
            this.textLengthType = textLengthType;
            return this;
        }

        public Builder moreLabel(String moreLabel) {
            this.moreLabel = moreLabel;
            return this;
        }

        public Builder lessLabel(String lessLabel) {
            this.lessLabel = lessLabel;
            return this;
        }

        public Builder moreLabelColor(int moreLabelColor) {
            this.moreLabelColor = moreLabelColor;
            return this;
        }

        public Builder lessLabelColor(int lessLabelColor) {
            this.lessLabelColor = lessLabelColor;
            return this;
        }

        public Builder labelUnderLine(boolean labelUnderLine) {
            this.labelUnderLine = labelUnderLine;
            return this;
        }

        /**
         * @param expandAnimation either true to enable animation on expand or false to disable animation
         *                        - default is false
         * @return Builder obj
         */
        public Builder expandAnimation(boolean expandAnimation) {
            this.expandAnimation = expandAnimation;
            return this;
        }

        public Builder optionClick(OptionClick optionClick) {
            this.optionClick = optionClick;
            return this;
        }

        public Builder position(int position) {
            this.position = position;
            return this;
        }

        public Builder status(int status) {
            this.status = status;
            return this;
        }

        public ReadMoreOption build() {
            return new ReadMoreOption(this);
        }
    }

    public interface OptionClick {
        void onReadMoreClickListener(int position);
        void onReadLessClickListener(int position);
    }
}