package faye

import android.os.Handler
import org.json.JSONException
import org.json.JSONObject
import java.util.HashSet
import android.os.HandlerThread
import android.view.WindowManager
import com.hippo.HippoConfig
import com.hippo.callback.OnCloseListener
import com.hippo.callback.OnInitializedListener
import com.hippo.constant.FuguAppConstant
import com.hippo.database.CommonData
import com.hippo.eventbus.BusProvider
import com.hippo.helper.BusEvents
import com.hippo.helper.FayeMessage
import com.hippo.helper.ParseMessage
import com.hippo.utils.HippoLog


/**
 * Created by gurmail on 2020-04-18.
 * @author gurmail
 */
object ConnectionManager: OnCloseListener, OnInitializedListener {

    override fun onClose() {
        stopFayeClient()
        //closeConnection()
    }

    override fun onInitialized() {
        initFayeConnection()
    }

    private var fayeClient: FayeClient? = null
    private var meta = MetaMessage()
    private val mChannels: HashSet<String> = HashSet()
    private val mPendingChannels: HashSet<String> = HashSet()
    private var connecting = false
    private var isDelaySubscribe = false

    //    private var handler: Handler? = null
//    private val RECONNECTION_TIME = 2000
    private var retryCount = 0

    private val NOT_CONNECTED = 0
    private val CONNECTED_TO_INTERNET = 1
    private val CONNECTED_TO_INTERNET_VIA_WIFI = 2
    private var NETWORK_STATUS = 1

    private var pongCount = 0
    private const val MAX_COUNT = 5
    private const val MAX_RETRY_ATTEMPTS = 15

    fun initFayeConnection() {
        fayeClient = getFayeClient()
        if(fayeClient != null && !isConnected() && !connecting) {
            fayeClient!!.connectServer()
            connecting = true
            println("########################## ConnectionManager initFayeConnection ##########################")
        }
        setListener()
//        if(handler == null)
//            handler = Handler(Looper.getMainLooper())
    }

    private fun getFayeClient(): FayeClient? {
        if(fayeClient == null) {
            println("########################## ConnectionManager getFayeClient ##########################")
            val jsonExt = JSONObject()
            try {
                if (HippoConfig.getInstance().userData != null) {
                    jsonExt.put("user_id", HippoConfig.getInstance().userData.userId)
                    jsonExt.put("device_type", 1)
                    jsonExt.put("source", 1)
                } else {
                    connecting = false
                    return null
                }
            } catch (e: JSONException) {
                e.printStackTrace()
            }

            meta.setAllExt(jsonExt.toString())
            fayeClient = FayeClient(getFayeUrl(), meta)
        }
        return fayeClient
    }

    private fun reconnectConnection() {
        initFayeConnection()
    }

    private fun getFayeUrl(): String {
        var url = CommonData.getServerUrl() + ":3002/faye"
        if (CommonData.getServerUrl() == FuguAppConstant.LIVE_SERVER) {
            url = CommonData.getServerUrl() + ":3002/faye"
        } else if (CommonData.getServerUrl() == FuguAppConstant.BETA_LIVE_SERVER) {
            url = FuguAppConstant.BETA_LIVE_SERVER + ":3001/faye"
        } else if (CommonData.getServerUrl() == FuguAppConstant.TEST_SERVER) {
            url = "https://hippo-api-dev.fuguchat.com:3012/faye"
        } else if (CommonData.getServerUrl() == FuguAppConstant.BETA_LIVE_SERVER) {
            url = FuguAppConstant.BETA_LIVE_SERVER + ":3001/faye"
        } else if (CommonData.getServerUrl() == FuguAppConstant.DEV_SERVER) {
            url = "https://hippo-api-dev.fuguchat.com:3002/faye"
        } else if (CommonData.getServerUrl() == FuguAppConstant.DEV_SERVER_3004) {
            url = "https://hippo-api-dev.fuguchat.com:3004/faye"
        } else {
            url = FuguAppConstant.LIVE_SERVER + ":3002/faye"
        }
        HippoLog.e("faye url", ">>>>>>>>>>>>>>>>>> $url")
        return url
    }

    private fun setListener() {
        if(fayeClient != null) {
            fayeClient!!.setmConnectionListener(object: FayeClientListener {

                override fun onConnectedServer(fc: FayeClient?) {
                    connecting = true
                    retryCount = 0
                    if(!isDelaySubscribe)
                        subscribePendingChannels()
                    else
                        subscribeDelayPendingChannels()
                    BusProvider.getInstance().post(FayeMessage(BusEvents.CONNECTED_SERVER.toString(), "", ""))
                    println(">>>>>>>>>>>>>>>>> ConnectionManager onConnectedServer <<<<<<<<<<<<<<<<<<<<<<")
                    stopAutoAttemptConnection()
                }

                override fun onDisconnectedServer(fc: FayeClient?) {
                    connecting = false
                    if(!forceStop) {
                        saveSubsribedChannels()
                        BusProvider.getInstance().post(FayeMessage(BusEvents.DISCONNECTED_SERVER.toString(), "", ""))
                        println(">>>>>>>>>>>>>>>>> ConnectionManager onDisconnectedServer <<<<<<<<<<<<<<<<<<<<<<")
                        attemptAutoConnection()
                    }
                }

                override fun onReceivedMessage(fc: FayeClient?, msg: String?, channel: String?) {
                    connecting = true
                    retryCount = 0
                    ParseMessage.receivedMessage(msg, channel)
                    println(">>>>>>>>>>>>>>>>> ConnectionManager onReceivedMessage <<<<<<<<<<<<<<<<<<<<<<")
                }

                override fun onPongReceived() {
                    connecting = true
                    retryCount = 0
                    subscribePendingChannels()
                    BusProvider.getInstance().post(FayeMessage(BusEvents.PONG_RECEIVED.toString(), "", ""))
                    pongCount()
                    println(">>>>>>>>>>>>>>>>> ConnectionManager onPongReceived <<<<<<<<<<<<<<<<<<<<<<")
                }

                override fun onWebSocketError() {
                    connecting = false
                    pongCount = 0
                    BusProvider.getInstance().post(FayeMessage(BusEvents.WEBSOCKET_ERROR.toString(), "", ""))
                    println(">>>>>>>>>>>>>>>>> ConnectionManager onWebSocketError <<<<<<<<<<<<<<<<<<<<<<")
                }

                override fun onErrorReceived(fc: FayeClient?, msg: String?, channel: String?) {
                    connecting = true
                    BusProvider.getInstance().post(FayeMessage(BusEvents.ERROR_RECEIVED.toString(), channel, msg))
                    println(">>>>>>>>>>>>>>>>> ConnectionManager onErrorReceived <<<<<<<<<<<<<<<<<<<<<<")
                }

                override fun onNotConnected() {
                    connecting = false
                    pongCount = 0
                    BusProvider.getInstance().post(FayeMessage(BusEvents.NOT_CONNECTED.toString(), "", ""))
                    println(">>>>>>>>>>>>>>>>> ConnectionManager onConnectedServer <<<<<<<<<<<<<<<<<<<<<<")
                }

                override fun onSubscriptionError() {
                    connecting = true
                    retryCount = 0
                    println(">>>>>>>>>>>>>>>>> ConnectionManager subscriptionError <<<<<<<<<<<<<<<<<<<<<<")
                    reSubscribeChannels()
                }
            })
        }
    }

    public fun isConnected(): Boolean {
        return fayeClient != null && fayeClient!!.isConnectedServer && fayeClient!!.isOpened
    }

    public fun subScribeChannel(channel: String) {
        if(isConnected()) {
            if(!mChannels.contains(channel)) {
                mChannels.add(channel)
                fayeClient!!.subscribeChannel(channel)
            } else {
                // already subscribed channel
            }
        } else {
            // pending channels for subscriptions
            mPendingChannels.add(channel)
            reconnectConnection()
        }
    }

    public fun subscribeOnDelay(channel: String) {
        if(isConnected()) {
            if(!mChannels.contains(channel)) {
                mChannels.add(channel)
                fayeClient!!.subscribeChannel(channel)
            } else {
                mChannels.add(channel)
                fayeClient!!.subscribeChannel(channel)
                // already subscribed channel
            }
        } else {
            isDelaySubscribe = true
            mPendingChannels.add(channel)
            reconnectConnection()
        }
    }

    public fun unsubScribeChannel(channel: String) {
        mChannels.remove(channel)
        fayeClient!!.unsubscribeChannel(channel)
    }

    public fun publish(channel: String, jsonObject: JSONObject) {
        publish(channel, jsonObject, false)
    }
    public fun publish(channel: String, jsonObject: JSONObject, canStoreLocally: Boolean) {
        if(isConnected()) {
            fayeClient!!.publish(channel, jsonObject)
        } else {
            mPendingChannels.add(channel)
            if(canStoreLocally) {

            }
            //CoroutineScope
            reconnectConnection()
        }
    }

    private fun pongCount() {
        pongCount += 1
        if(pongCount > MAX_COUNT) {
            if(HippoConfig.getInstance().context == null || !ConnectionUtils.isAppRunning(HippoConfig.getInstance().context)) {
                if (HippoConfig.getInstance().fayeCallDate == null || !HippoConfig.getInstance().fayeCallDate.isCallServiceRunning()) {
                    println("Closing connection")
                    stopFayeClient()
                } else {
                    pongCount = 0
                    println("Inside the running activities")
                }
            } else {
                pongCount = 0
                println("Outside the running activities")
            }
        }
    }

    var forceStop: Boolean = false
    private fun stopFayeClient() {
        if(fayeClient != null &&
            (HippoConfig.getInstance().fayeCallDate == null || !HippoConfig.getInstance().fayeCallDate.isCallServiceRunning())) {
            val thread = HandlerThread("TerminateThread")
            thread.start()
            Handler(thread.looper).post(Runnable {
                if(fayeClient != null) {
                    mPendingChannels.clear()
                    mChannels.clear()
                    if(isConnected()) {
                        forceStop = true
                        fayeClient!!.disconnectServer()
                    }
                    pongCount = 0
                    fayeClient = null
                    connecting = false
                }
            })
        }
    }

    private fun reSubscribeChannels() {
        synchronized(this) {
            val tempChannel = mChannels
            for(channel in tempChannel) {
                unsubScribeChannel(channel)
            }
            Handler().postDelayed({
                for(channel in tempChannel) {
                    subScribeChannel(channel)
                }
            }, 500)
        }
    }

    private fun subscribePendingChannels() {
        synchronized(this) {
            if(mPendingChannels.size>0) {
                val tempChannel = mPendingChannels
                for (channel in tempChannel) {
                    fayeClient!!.subscribeChannel(channel)
                    mChannels.add(channel)
                }
                mPendingChannels.removeAll(tempChannel)
            }
        }
    }

    private fun subscribeDelayPendingChannels() {
        Handler().postDelayed({
            isDelaySubscribe = false
            if(mPendingChannels.size>0) {
                val tempChannel = mPendingChannels
                for (channel in tempChannel) {
                    fayeClient!!.subscribeChannel(channel)
                }
                mPendingChannels.removeAll(tempChannel)
            }
        }, 2000)
    }

    public fun changeStatus(status: Int) {
        NETWORK_STATUS = status
        when (status) {
            NOT_CONNECTED -> {
                saveSubsribedChannels()
            }
            CONNECTED_TO_INTERNET, CONNECTED_TO_INTERNET_VIA_WIFI -> {
                initFayeConnection()
            }
        }
    }

    @Synchronized
    private fun saveSubsribedChannels() {
        connecting = false
        if(mChannels.size>0) {
            mPendingChannels.addAll(mChannels)
            mChannels.clear()
        }
    }

    private fun attemptAutoConnection() {
        if(NETWORK_STATUS != NOT_CONNECTED) {
            if(retryCount < MAX_RETRY_ATTEMPTS) {
                retryCount += 1
                initFayeConnection()

//                retryCount += 1
//                handler?.postDelayed(runnable, RECONNECTION_TIME.toLong())
            } else {
                //stopAutoAttemptConnection()
            }
        }
    }

    private fun stopAutoAttemptConnection() {
        //handler?.removeCallbacks(runnable)
    }

    internal var runnable: Runnable = Runnable {
        try {
            initFayeConnection()
        } catch (e: Exception) {

        }
    }

}