/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.gcp;

import com.hazelcast.gcp.GcpAddress;
import com.hazelcast.gcp.GcpAuthenticator;
import com.hazelcast.gcp.GcpComputeApi;
import com.hazelcast.gcp.GcpConfig;
import com.hazelcast.gcp.GcpDiscoveryStrategy;
import com.hazelcast.gcp.GcpMetadataApi;
import com.hazelcast.gcp.Label;
import com.hazelcast.gcp.RestClientException;
import com.hazelcast.gcp.RetryUtils;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.Logger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;

class GcpClient {
    private static final ILogger LOGGER = Logger.getLogger(GcpDiscoveryStrategy.class);
    private static final int HTTP_UNAUTHORIZED = 401;
    private static final int HTTP_FORBIDDEN = 403;
    private static final int HTTP_NOT_FOUND = 404;
    private static final int RETRIES = 3;
    private static final List<String> NON_RETRYABLE_KEYWORDS = Arrays.asList("Private key json file not found", "Request had insufficient authentication scopes", "Required 'compute.instances.list' permission", "Service account not enabled on this instance");
    private boolean isKnownExceptionAlreadyLogged;
    private final GcpMetadataApi gcpMetadataApi;
    private final GcpComputeApi gcpComputeApi;
    private final GcpAuthenticator gcpAuthenticator;
    private final String privateKeyPath;
    private final List<String> projects;
    private final List<String> zones;
    private final Label label;

    GcpClient(GcpMetadataApi gcpMetadataApi, GcpComputeApi gcpComputeApi, GcpAuthenticator gcpAuthenticator, GcpConfig gcpConfig) {
        this.gcpMetadataApi = gcpMetadataApi;
        this.gcpComputeApi = gcpComputeApi;
        this.gcpAuthenticator = gcpAuthenticator;
        this.privateKeyPath = gcpConfig.getPrivateKeyPath();
        this.projects = this.projectFromConfigOrMetadataApi(gcpConfig);
        this.zones = this.zonesFromConfigOrComputeApi(gcpConfig);
        this.label = gcpConfig.getLabel();
    }

    private List<String> projectFromConfigOrMetadataApi(GcpConfig gcpConfig) {
        if (!gcpConfig.getProjects().isEmpty()) {
            return gcpConfig.getProjects();
        }
        LOGGER.finest("Property 'projects' not configured, fetching the current GCP project");
        return Collections.singletonList(RetryUtils.retry(new Callable<String>(){

            @Override
            public String call() {
                return GcpClient.this.gcpMetadataApi.currentProject();
            }
        }, 3, NON_RETRYABLE_KEYWORDS));
    }

    private List<String> zonesFromConfigOrComputeApi(final GcpConfig gcpConfig) {
        try {
            if (gcpConfig.getRegion() != null) {
                LOGGER.finest("Property 'region' configured, fetching GCP zones of the specified GCP region");
                return RetryUtils.retry(new Callable<List<String>>(){

                    @Override
                    public List<String> call() {
                        return GcpClient.this.fetchZones(gcpConfig.getRegion());
                    }
                }, 3, NON_RETRYABLE_KEYWORDS);
            }
            if (!gcpConfig.getZones().isEmpty()) {
                return gcpConfig.getZones();
            }
            LOGGER.finest("Property 'zones' not configured, fetching GCP zones of the current GCP region");
            return RetryUtils.retry(new Callable<List<String>>(){

                @Override
                public List<String> call() {
                    String region = GcpClient.this.gcpMetadataApi.currentRegion();
                    return GcpClient.this.fetchZones(region);
                }
            }, 3, NON_RETRYABLE_KEYWORDS);
        }
        catch (RestClientException e) {
            this.handleKnownException(e);
            return Collections.emptyList();
        }
    }

    List<GcpAddress> getAddresses() {
        try {
            return RetryUtils.retry(new Callable<List<GcpAddress>>(){

                @Override
                public List<GcpAddress> call() {
                    return GcpClient.this.fetchGcpAddresses();
                }
            }, 3, NON_RETRYABLE_KEYWORDS);
        }
        catch (RestClientException e) {
            this.handleKnownException(e);
            return Collections.emptyList();
        }
    }

    private List<String> fetchZones(String region) {
        ArrayList<String> zones = new ArrayList<String>();
        String accessToken = this.fetchAccessToken();
        for (String project : this.projects) {
            zones.addAll(this.gcpComputeApi.zones(project, region, accessToken));
        }
        return zones;
    }

    private List<GcpAddress> fetchGcpAddresses() {
        LOGGER.finest("Fetching OAuth Access Token");
        String accessToken = this.fetchAccessToken();
        ArrayList<GcpAddress> result = new ArrayList<GcpAddress>();
        for (String project : this.projects) {
            for (String zone : this.zones) {
                LOGGER.finest(String.format("Fetching instances for project '%s' and zone '%s'", project, zone));
                List<GcpAddress> addresses = this.gcpComputeApi.instances(project, zone, this.label, accessToken);
                LOGGER.finest(String.format("Found the following instances for project '%s' and zone '%s': %s", project, zone, addresses));
                result.addAll(addresses);
            }
        }
        return result;
    }

    private String fetchAccessToken() {
        if (this.privateKeyPath != null) {
            return this.gcpAuthenticator.refreshAccessToken(this.privateKeyPath);
        }
        return this.gcpMetadataApi.accessToken();
    }

    String getAvailabilityZone() {
        return this.gcpMetadataApi.currentZone();
    }

    private void handleKnownException(RestClientException e) {
        if (e.getHttpErrorCode() == 401) {
            if (!this.isKnownExceptionAlreadyLogged) {
                LOGGER.warning("Google Cloud API Authorization failed! Check your credentials. Starting standalone.");
                this.isKnownExceptionAlreadyLogged = true;
            }
        } else if (e.getHttpErrorCode() == 403) {
            if (!this.isKnownExceptionAlreadyLogged) {
                LOGGER.warning("Google Cloud API access is forbidden! Starting standalone. To use Hazelcast GCP discovery, make sure that your service account has at minimum \"Read Only\" Access Scope to Compute Engine API.");
                this.isKnownExceptionAlreadyLogged = true;
            }
        } else if (e.getHttpErrorCode() == 404) {
            if (!this.isKnownExceptionAlreadyLogged) {
                LOGGER.warning("Google Cloud API Not Found! Starting standalone. Please check that you have a service account assigned to your VM instance or `private-key-path` property defined.");
                this.isKnownExceptionAlreadyLogged = true;
            }
        } else {
            throw e;
        }
        LOGGER.finest(e);
    }
}

