/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.jet.server;

import com.hazelcast.client.config.ClientConfig;
import com.hazelcast.client.config.XmlClientConfigBuilder;
import com.hazelcast.core.Cluster;
import com.hazelcast.instance.BuildInfoProvider;
import com.hazelcast.instance.JetBuildInfo;
import com.hazelcast.jet.Jet;
import com.hazelcast.jet.JetException;
import com.hazelcast.jet.JetInstance;
import com.hazelcast.jet.Job;
import com.hazelcast.jet.JobStateSnapshot;
import com.hazelcast.jet.Util;
import com.hazelcast.jet.core.JobNotFoundException;
import com.hazelcast.jet.core.JobStatus;
import com.hazelcast.jet.impl.ClusterMetadata;
import com.hazelcast.jet.impl.JetClientInstanceImpl;
import com.hazelcast.jet.impl.JobSummary;
import com.hazelcast.jet.impl.config.ConfigProvider;
import com.hazelcast.jet.server.JetBootstrap;
import com.hazelcast.jet.server.StartServer;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.LogManager;
import picocli.CommandLine;

@CommandLine.Command(name="jet", description={"Utility for interacting with a Hazelcast Jet cluster.%n%nThe command line tool uses the Jet client to connect and perform operations on the cluster. By default, the client config XML inside the config path will be used for the connection.%n%nGlobal options are:%n"}, versionProvider=JetVersionProvider.class, mixinStandardHelpOptions=true, sortOptions=false, subcommands={CommandLine.HelpCommand.class})
public class JetCommandLine
implements Runnable {
    private static final int MAX_STR_LENGTH = 24;
    private static final int WAIT_INTERVAL_MILLIS = 100;
    private final Function<ClientConfig, JetInstance> jetClientFn;
    private final PrintStream out;
    private final PrintStream err;
    @CommandLine.Option(names={"-f", "--config"}, description={"Optional path to a client config XML file. By default config/hazelcast-client.xml is used.If this option is specified then the addresses and group name options are ignored."}, order=0)
    private File configXml;
    @CommandLine.Option(names={"-a", "--addresses"}, split=",", arity="1..*", paramLabel="<hostname>:<port>", description={"Optional comma-separated list of Jet node addresses in the format <hostname>:<port> to connect to another cluster than the one configured in config/hazelcast-client.xml"}, order=1)
    private List<String> addresses;
    @CommandLine.Option(names={"-g", "--group"}, description={"The group name to use when connecting to the cluster specified by the <addresses> parameter. "}, defaultValue="jet", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, order=2)
    private String groupName;
    @CommandLine.Mixin(name="verbosity")
    private Verbosity verbosity;

    public JetCommandLine(Function<ClientConfig, JetInstance> jetClientFn, PrintStream out, PrintStream err) {
        this.jetClientFn = jetClientFn;
        this.out = out;
        this.err = err;
    }

    public static void main(String[] args) {
        JetCommandLine.runCommandLine(Jet::newJetClient, System.out, System.err, true, args);
    }

    static void runCommandLine(Function<ClientConfig, JetInstance> jetClientFn, PrintStream out, PrintStream err, boolean shouldExit, String[] args) {
        CommandLine cmd = new CommandLine(new JetCommandLine(jetClientFn, out, err));
        cmd.getSubcommands().get("submit").setStopAtPositional(true);
        String jetVersion = BuildInfoProvider.getBuildInfo().getJetBuildInfo().getVersion();
        cmd.getCommandSpec().usageMessage().header("Hazelcast Jet " + jetVersion);
        if (args.length == 0) {
            cmd.usage(out);
        } else {
            List parsed;
            CommandLine.DefaultExceptionHandler excHandler = (CommandLine.DefaultExceptionHandler)((CommandLine.DefaultExceptionHandler)new ExceptionHandler().useErr(err)).useAnsi(CommandLine.Help.Ansi.AUTO);
            if (shouldExit) {
                excHandler.andExit(1);
            }
            if ((parsed = (List)cmd.parseWithHandlers((CommandLine.IParseResultHandler2)((CommandLine.AbstractParseResultHandler)new CommandLine.RunAll().useOut(out)).useAnsi(CommandLine.Help.Ansi.AUTO), excHandler, args)) != null && parsed.size() == 1) {
                cmd.usage(out);
            }
        }
    }

    @Override
    public void run() {
    }

    @CommandLine.Command(description={"Submits a job to the cluster"}, mixinStandardHelpOptions=true)
    public void submit(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Option(names={"-s", "--snapshot"}, paramLabel="<snapshot name>", description={"Name of initial snapshot to start the job from"}) String snapshotName, @CommandLine.Option(names={"-n", "--name"}, paramLabel="<name>", description={"Name of the job"}) String name, @CommandLine.Parameters(index="0", paramLabel="<jar file>", description={"The jar file to submit"}) File file, @CommandLine.Parameters(index="1..*", paramLabel="<arguments>", description={"arguments to pass to the supplied jar file"}) List<String> params) throws Exception {
        if (params == null) {
            params = Collections.emptyList();
        }
        this.verbosity.merge(verbosity);
        this.configureLogging();
        if (!file.exists()) {
            throw new Exception("File " + file + " could not be found.");
        }
        this.printf("Submitting JAR '%s' with arguments %s%n", file, params);
        if (name != null) {
            this.printf("Using job name '%s'%n", name);
        }
        if (snapshotName != null) {
            this.printf("Job will be restored from snapshot with name '%s'%n", snapshotName);
        }
        JetBootstrap.executeJar(this::getJetClient, file.getAbsolutePath(), snapshotName, name, params);
    }

    @CommandLine.Command(description={"Suspends a running job"}, mixinStandardHelpOptions=true)
    public void suspend(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Parameters(index="0", paramLabel="<job name or id>", description={"Name of the job to suspend"}) String name) throws IOException {
        this.runWithJet(verbosity, jet -> {
            Job job = JetCommandLine.getJob(jet, name);
            JetCommandLine.assertJobRunning(name, job);
            this.printf("Suspending job %s...%n", JetCommandLine.formatJob(job));
            job.suspend();
            JetCommandLine.waitForJobStatus(job, JobStatus.SUSPENDED);
            this.println("Job was successfully suspended.");
        });
    }

    @CommandLine.Command(description={"Cancels a running job"})
    public void cancel(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Parameters(index="0", paramLabel="<job name or id>", description={"Name of the job to terminate"}) String name) throws IOException {
        this.runWithJet(verbosity, jet -> {
            Job job = JetCommandLine.getJob(jet, name);
            JetCommandLine.assertJobActive(name, job);
            this.printf("Cancelling job %s...%n", JetCommandLine.formatJob(job));
            job.cancel();
            JetCommandLine.waitForJobStatus(job, JobStatus.FAILED);
            this.println("Job was successfully terminated.");
        });
    }

    @CommandLine.Command(name="save-snapshot", description={"Exports a named snapshot from a job and optionally cancels it"})
    public void saveSnapshot(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Parameters(index="0", paramLabel="<job name or id>", description={"Name of the job to terminate"}) String jobName, @CommandLine.Parameters(index="1", paramLabel="<snapshot name>", description={"Name of the snapshot"}) String snapshotName, @CommandLine.Option(names={"-C", "--cancel"}, description={"Cancel the job after taking the snapshot"}) boolean isTerminal) throws IOException {
        this.runWithJet(verbosity, jet -> {
            Job job = JetCommandLine.getJob(jet, jobName);
            JetCommandLine.assertJobActive(jobName, job);
            if (isTerminal) {
                this.printf("Saving snapshot with name '%s' from job '%s' and terminating the job...%n", snapshotName, JetCommandLine.formatJob(job));
                job.cancelAndExportSnapshot(snapshotName);
                JetCommandLine.waitForJobStatus(job, JobStatus.FAILED);
            } else {
                this.printf("Saving snapshot with name '%s' from job '%s'...%n", snapshotName, JetCommandLine.formatJob(job));
                job.exportSnapshot(snapshotName);
            }
            this.printf("Snapshot '%s' was successfully exported.%n", snapshotName);
        });
    }

    @CommandLine.Command(name="delete-snapshot", description={"Deletes a named snapshot"})
    public void deleteSnapshot(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Parameters(index="0", paramLabel="<snapshot name>", description={"Name of the snapshot"}) String snapshotName) throws IOException {
        this.runWithJet(verbosity, jet -> {
            JobStateSnapshot jobStateSnapshot = jet.getJobStateSnapshot(snapshotName);
            if (jobStateSnapshot == null) {
                throw new JetException(String.format("No snapshot with name '%s' was found", snapshotName));
            }
            jobStateSnapshot.destroy();
            this.printf("Snapshot '%s' was successfully deleted.%n", snapshotName);
        });
    }

    @CommandLine.Command(description={"Restarts a running job"})
    public void restart(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Parameters(index="0", paramLabel="<job name or id>", description={"Name of the job to restart"}) String name) throws IOException {
        this.runWithJet(verbosity, jet -> {
            Job job = JetCommandLine.getJob(jet, name);
            JetCommandLine.assertJobRunning(name, job);
            this.println("Restarting job " + JetCommandLine.formatJob(job) + "...");
            job.restart();
            JetCommandLine.waitForJobStatus(job, JobStatus.RUNNING);
            this.println("Job was successfully restarted.");
        });
    }

    @CommandLine.Command(description={"Resumes a suspended job"})
    public void resume(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Parameters(index="0", paramLabel="<job name or id>", description={"Name of the job to resume"}) String name) throws IOException {
        this.runWithJet(verbosity, jet -> {
            Job job = JetCommandLine.getJob(jet, name);
            if (job.getStatus() != JobStatus.SUSPENDED) {
                throw new RuntimeException("Job '" + name + "' is not suspended. Current state: " + (Object)((Object)job.getStatus()));
            }
            this.println("Resuming job " + JetCommandLine.formatJob(job) + "...");
            job.resume();
            JetCommandLine.waitForJobStatus(job, JobStatus.RUNNING);
            this.println("Job was successfully resumed.");
        });
    }

    @CommandLine.Command(name="list-jobs", description={"Lists running jobs on the cluster"})
    public void listJobs(@CommandLine.Mixin(name="verbosity") Verbosity verbosity, @CommandLine.Option(names={"-a", "--all"}, description={"Lists all jobs including completed and failed ones"}) boolean listAll) throws IOException {
        this.runWithJet(verbosity, jet -> {
            JetClientInstanceImpl client = (JetClientInstanceImpl)jet;
            List<JobSummary> summaries = client.getJobSummaryList();
            String format = "%-24s %-19s %-18s %-23s%n";
            this.printf(format, "NAME", "ID", "STATUS", "SUBMISSION TIME");
            summaries.stream().filter(job -> listAll || JetCommandLine.isActive(job.getStatus())).forEach(job -> {
                String idString = Util.idToString(job.getJobId());
                String name = job.getName().equals(idString) ? "N/A" : JetCommandLine.shorten(job.getName(), 24);
                this.printf(format, new Object[]{name, idString, job.getStatus(), com.hazelcast.jet.impl.util.Util.toLocalDateTime(job.getSubmissionTime())});
            });
        });
    }

    @CommandLine.Command(name="list-snapshots", description={"Lists exported snapshots on the cluster"})
    public void listSnapshots(@CommandLine.Mixin(name="verbosity") Verbosity verbosity) throws IOException {
        this.runWithJet(verbosity, jet -> {
            Collection<JobStateSnapshot> snapshots = jet.getJobStateSnapshots();
            this.printf("%-24s %-15s %-23s %-24s%n", "NAME", "SIZE (bytes)", "TIME", "JOB NAME");
            snapshots.forEach(ss -> {
                String jobName = ss.jobName() == null ? Util.idToString(ss.jobId()) : ss.jobName();
                jobName = JetCommandLine.shorten(jobName, 24);
                String ssName = JetCommandLine.shorten(ss.name(), 24);
                LocalDateTime creationTime = com.hazelcast.jet.impl.util.Util.toLocalDateTime(ss.creationTime());
                this.printf("%-24s %-,15d %-23s %-24s%n", ssName, ss.payloadSize(), creationTime, jobName);
            });
        });
    }

    @CommandLine.Command(description={"Shows current cluster state and information about members"})
    public void cluster(@CommandLine.Mixin(name="verbosity") Verbosity verbosity) throws IOException {
        this.runWithJet(verbosity, jet -> {
            JetClientInstanceImpl client = (JetClientInstanceImpl)jet;
            ClusterMetadata clusterMetadata = ((JetClientInstanceImpl)jet).getClusterMetadata();
            Cluster cluster = client.getCluster();
            this.println("State: " + (Object)((Object)clusterMetadata.getState()));
            this.println("Version: " + clusterMetadata.getVersion());
            this.println("Size: " + cluster.getMembers().size());
            this.println("");
            String format = "%-24s %-19s%n";
            this.printf(format, "ADDRESS", "UUID");
            cluster.getMembers().forEach(member -> this.printf(format, member.getAddress(), member.getUuid()));
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runWithJet(Verbosity verbosity, Consumer<JetInstance> consumer) throws IOException {
        this.verbosity.merge(verbosity);
        this.configureLogging();
        JetInstance jet = this.getJetClient();
        try {
            consumer.accept(jet);
        }
        finally {
            jet.shutdown();
        }
    }

    private JetInstance getJetClient() {
        return com.hazelcast.jet.impl.util.Util.uncheckCall(() -> this.jetClientFn.apply(this.getClientConfig()));
    }

    private ClientConfig getClientConfig() throws IOException {
        if (this.configXml != null) {
            return new XmlClientConfigBuilder(this.configXml).build();
        }
        if (this.addresses != null) {
            ClientConfig config = new ClientConfig();
            config.getNetworkConfig().addAddress(this.addresses.toArray(new String[0]));
            config.getGroupConfig().setName(this.groupName);
            return config;
        }
        return ConfigProvider.locateAndGetClientConfig();
    }

    private void configureLogging() throws IOException {
        StartServer.configureLogging();
        Level logLevel = Level.WARNING;
        if (this.verbosity.isVerbose) {
            this.println("Verbose mode is on, setting logging level to INFO");
            logLevel = Level.INFO;
        }
        LogManager.getLogManager().getLogger("").setLevel(logLevel);
    }

    private static Job getJob(JetInstance jet, String nameOrId) {
        Job job = jet.getJob(nameOrId);
        if (job == null && (job = jet.getJob(Util.idFromString(nameOrId))) == null) {
            throw new JobNotFoundException("No job with name or id '" + nameOrId + "' was found");
        }
        return job;
    }

    private void printf(String format, Object ... objects) {
        this.out.printf(format, objects);
    }

    private void println(String msg) {
        this.out.println(msg);
    }

    private static String shorten(String name, int length) {
        return name.substring(0, Math.min(name.length(), length));
    }

    private static String formatJob(Job job) {
        return "id=" + Util.idToString(job.getId()) + ", name=" + job.getName() + ", submissionTime=" + com.hazelcast.jet.impl.util.Util.toLocalDateTime(job.getSubmissionTime());
    }

    private static void assertJobActive(String name, Job job) {
        if (!JetCommandLine.isActive(job.getStatus())) {
            throw new RuntimeException("Job '" + name + "' is not active. Current state: " + (Object)((Object)job.getStatus()));
        }
    }

    private static void assertJobRunning(String name, Job job) {
        if (job.getStatus() != JobStatus.RUNNING) {
            throw new RuntimeException("Job '" + name + "' is not running. Current state: " + (Object)((Object)job.getStatus()));
        }
    }

    private static void waitForJobStatus(Job job, JobStatus status) {
        while (job.getStatus() != status) {
            LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(100L));
        }
    }

    private static boolean isActive(JobStatus status) {
        return status != JobStatus.FAILED && status != JobStatus.COMPLETED;
    }

    static class ExceptionHandler<R>
    extends CommandLine.DefaultExceptionHandler<R> {
        ExceptionHandler() {
        }

        @Override
        public R handleExecutionException(CommandLine.ExecutionException ex, CommandLine.ParseResult parseResult) {
            CommandLine cmdLine = ex.getCommandLine();
            while (cmdLine.getParent() != null) {
                cmdLine = cmdLine.getParent();
            }
            JetCommandLine jetCmd = (JetCommandLine)cmdLine.getCommand();
            if (jetCmd.verbosity.isVerbose) {
                ex.printStackTrace(this.err());
            } else {
                this.err().println("ERROR: " + ex.getCause().getMessage());
                this.err().println();
                this.err().println("To see the full stack trace, re-run with the -v/--verbosity option");
            }
            if (this.hasExitCode()) {
                this.exit(this.exitCode());
            }
            throw ex;
        }
    }

    public static class Verbosity {
        @CommandLine.Option(names={"-v", "--verbosity"}, description={"Show logs from Jet client and full stack trace of errors"}, order=1)
        private boolean isVerbose;

        void merge(Verbosity other) {
            this.isVerbose |= other.isVerbose;
        }
    }

    public static class JetVersionProvider
    implements CommandLine.IVersionProvider {
        @Override
        public String[] getVersion() {
            JetBuildInfo jetBuildInfo = BuildInfoProvider.getBuildInfo().getJetBuildInfo();
            return new String[]{"Hazelcast Jet " + jetBuildInfo.getVersion(), "Revision " + jetBuildInfo.getRevision(), "Build " + jetBuildInfo.getBuild()};
        }
    }
}

