/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.map.impl.eviction;

import com.hazelcast.cluster.memberselector.MemberSelectors;
import com.hazelcast.config.MapConfig;
import com.hazelcast.config.MaxSizeConfig;
import com.hazelcast.internal.nearcache.NearCache;
import com.hazelcast.logging.ILogger;
import com.hazelcast.map.impl.MapContainer;
import com.hazelcast.map.impl.MapServiceContext;
import com.hazelcast.map.impl.PartitionContainer;
import com.hazelcast.map.impl.nearcache.MapNearCacheManager;
import com.hazelcast.map.impl.recordstore.RecordStore;
import com.hazelcast.memory.MemorySize;
import com.hazelcast.memory.MemoryUnit;
import com.hazelcast.monitor.NearCacheStats;
import com.hazelcast.nio.Address;
import com.hazelcast.spi.NodeEngine;
import com.hazelcast.spi.partition.IPartition;
import com.hazelcast.spi.partition.IPartitionService;
import com.hazelcast.util.MemoryInfoAccessor;
import com.hazelcast.util.Preconditions;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

public class EvictionChecker {
    protected static final double ONE_HUNDRED_PERCENT = 100.0;
    protected final ILogger logger;
    protected final MapServiceContext mapServiceContext;
    protected final MemoryInfoAccessor memoryInfoAccessor;
    protected final AtomicBoolean misconfiguredPerNodeMaxSizeWarningLogged;

    public EvictionChecker(MemoryInfoAccessor givenMemoryInfoAccessor, MapServiceContext mapServiceContext) {
        Preconditions.checkNotNull(givenMemoryInfoAccessor, "givenMemoryInfoAccessor cannot be null");
        Preconditions.checkNotNull(mapServiceContext, "mapServiceContext cannot be null");
        this.logger = mapServiceContext.getNodeEngine().getLogger(this.getClass());
        this.mapServiceContext = mapServiceContext;
        this.memoryInfoAccessor = givenMemoryInfoAccessor;
        if (this.logger.isFinestEnabled()) {
            this.logger.finest("Used memoryInfoAccessor=" + this.memoryInfoAccessor.getClass().getCanonicalName());
        }
        this.misconfiguredPerNodeMaxSizeWarningLogged = new AtomicBoolean();
    }

    public boolean checkEvictable(RecordStore recordStore) {
        if (recordStore.size() == 0) {
            return false;
        }
        String mapName = recordStore.getName();
        MapContainer mapContainer = recordStore.getMapContainer();
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = maxSizeConfig.getMaxSizePolicy();
        switch (maxSizePolicy) {
            case PER_NODE: {
                return this.checkPerNodeEviction(recordStore);
            }
            case PER_PARTITION: {
                int partitionId = recordStore.getPartitionId();
                return this.checkPerPartitionEviction(mapName, maxSizeConfig, partitionId);
            }
            case USED_HEAP_PERCENTAGE: {
                return this.checkHeapPercentageEviction(mapName, maxSizeConfig);
            }
            case USED_HEAP_SIZE: {
                return this.checkHeapSizeEviction(mapName, maxSizeConfig);
            }
            case FREE_HEAP_PERCENTAGE: {
                return this.checkFreeHeapPercentageEviction(maxSizeConfig);
            }
            case FREE_HEAP_SIZE: {
                return this.checkFreeHeapSizeEviction(maxSizeConfig);
            }
        }
        throw new IllegalArgumentException("Not an appropriate max size policy [" + (Object)((Object)maxSizePolicy) + ']');
    }

    protected boolean checkPerNodeEviction(RecordStore recordStore) {
        double recordStoreSize = this.translatePerNodeSizeToRecordStoreSize(recordStore);
        return (double)recordStore.size() > recordStoreSize;
    }

    public double translatePerNodeSizeToRecordStoreSize(RecordStore recordStore) {
        int partitionCount;
        int memberCount;
        MapConfig mapConfig = recordStore.getMapContainer().getMapConfig();
        MaxSizeConfig maxSizeConfig = mapConfig.getMaxSizeConfig();
        NodeEngine nodeEngine = this.mapServiceContext.getNodeEngine();
        int configuredMaxSize = maxSizeConfig.getSize();
        double perNodeMaxRecordStoreSize = 1.0 * (double)configuredMaxSize * (double)(memberCount = nodeEngine.getClusterService().getSize(MemberSelectors.DATA_MEMBER_SELECTOR)) / (double)(partitionCount = nodeEngine.getPartitionService().getPartitionCount());
        if (perNodeMaxRecordStoreSize < 1.0 && this.misconfiguredPerNodeMaxSizeWarningLogged.compareAndSet(false, true)) {
            int minMaxSize = (int)Math.ceil(1.0 * (double)partitionCount / (double)memberCount);
            String msg = "The max size configuration for map '%s' does not allow  any data in the map. Given the current cluster size of %d members with %d partitions, max size should be at least %d.";
            this.logger.warning(String.format(msg, mapConfig.getName(), memberCount, partitionCount, minMaxSize));
        }
        return perNodeMaxRecordStoreSize;
    }

    protected boolean checkPerPartitionEviction(String mapName, MaxSizeConfig maxSizeConfig, int partitionId) {
        double maxSize = maxSizeConfig.getSize();
        PartitionContainer container = this.mapServiceContext.getPartitionContainer(partitionId);
        if (container == null) {
            return false;
        }
        int size = this.getRecordStoreSize(mapName, container);
        return (double)size > maxSize;
    }

    protected boolean checkHeapSizeEviction(String mapName, MaxSizeConfig maxSizeConfig) {
        long usedHeapBytes = this.getUsedHeapInBytes(mapName);
        if (usedHeapBytes == -1L) {
            return false;
        }
        int maxUsableHeapMegaBytes = maxSizeConfig.getSize();
        return MemoryUnit.MEGABYTES.toBytes(maxUsableHeapMegaBytes) < usedHeapBytes;
    }

    protected boolean checkFreeHeapSizeEviction(MaxSizeConfig maxSizeConfig) {
        long currentFreeHeapBytes = this.getAvailableMemory();
        int minFreeHeapMegaBytes = maxSizeConfig.getSize();
        return MemoryUnit.MEGABYTES.toBytes(minFreeHeapMegaBytes) > currentFreeHeapBytes;
    }

    protected boolean checkHeapPercentageEviction(String mapName, MaxSizeConfig maxSizeConfig) {
        long usedHeapBytes = this.getUsedHeapInBytes(mapName);
        if (usedHeapBytes == -1L) {
            return false;
        }
        double maxOccupiedHeapPercentage = maxSizeConfig.getSize();
        long maxMemory = this.getMaxMemory();
        if (maxMemory <= 0L) {
            return true;
        }
        return maxOccupiedHeapPercentage < 100.0 * (double)usedHeapBytes / (double)maxMemory;
    }

    protected boolean checkFreeHeapPercentageEviction(MaxSizeConfig maxSizeConfig) {
        boolean evictable;
        long totalMemory = this.getTotalMemory();
        long freeMemory = this.getFreeMemory();
        long maxMemory = this.getMaxMemory();
        long availableMemory = freeMemory + (maxMemory - totalMemory);
        double actualFreePercentage = 0.0;
        double configuredFreePercentage = 0.0;
        if (totalMemory <= 0L || freeMemory <= 0L || maxMemory <= 0L || availableMemory <= 0L) {
            evictable = true;
        } else {
            actualFreePercentage = 100.0 * (double)availableMemory / (double)maxMemory;
            configuredFreePercentage = maxSizeConfig.getSize();
            boolean bl = evictable = configuredFreePercentage > actualFreePercentage;
        }
        if (evictable && this.logger.isFinestEnabled()) {
            this.logger.finest(String.format("runtime.max=%s, runtime.used=%s, configuredFree%%=%.2f, actualFree%%=%.2f", MemorySize.toPrettyString(maxMemory), MemorySize.toPrettyString(totalMemory - freeMemory), configuredFreePercentage, actualFreePercentage));
        }
        return evictable;
    }

    protected long getTotalMemory() {
        return this.memoryInfoAccessor.getTotalMemory();
    }

    protected long getFreeMemory() {
        return this.memoryInfoAccessor.getFreeMemory();
    }

    protected long getMaxMemory() {
        return this.memoryInfoAccessor.getMaxMemory();
    }

    protected long getAvailableMemory() {
        long totalMemory = this.getTotalMemory();
        long freeMemory = this.getFreeMemory();
        long maxMemory = this.getMaxMemory();
        return freeMemory + (maxMemory - totalMemory);
    }

    protected long getUsedHeapInBytes(String mapName) {
        long heapCost = 0L;
        List<Integer> partitionIds = this.findPartitionIds();
        for (int partitionId : partitionIds) {
            PartitionContainer container = this.mapServiceContext.getPartitionContainer(partitionId);
            if (container == null) continue;
            heapCost += this.getRecordStoreHeapCost(mapName, container);
        }
        MapContainer mapContainer = this.mapServiceContext.getMapContainer(mapName);
        if (!mapContainer.getMapConfig().isNearCacheEnabled()) {
            return heapCost;
        }
        MapNearCacheManager mapNearCacheManager = this.mapServiceContext.getMapNearCacheManager();
        NearCache nearCache = mapNearCacheManager.getNearCache(mapName);
        NearCacheStats nearCacheStats = nearCache.getNearCacheStats();
        return heapCost += nearCacheStats.getOwnedEntryMemoryCost();
    }

    protected int getRecordStoreSize(String mapName, PartitionContainer partitionContainer) {
        RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
        if (existingRecordStore == null) {
            return 0;
        }
        return existingRecordStore.size();
    }

    protected long getRecordStoreHeapCost(String mapName, PartitionContainer partitionContainer) {
        RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
        if (existingRecordStore == null) {
            return 0L;
        }
        return existingRecordStore.getOwnedEntryCost();
    }

    protected List<Integer> findPartitionIds() {
        NodeEngine nodeEngine = this.mapServiceContext.getNodeEngine();
        IPartitionService partitionService = nodeEngine.getPartitionService();
        int partitionCount = partitionService.getPartitionCount();
        ArrayList<Integer> partitionIds = null;
        for (int partitionId = 0; partitionId < partitionCount; ++partitionId) {
            if (!this.isOwnerOrBackup(partitionId)) continue;
            if (partitionIds == null) {
                partitionIds = new ArrayList<Integer>();
            }
            partitionIds.add(partitionId);
        }
        return partitionIds == null ? Collections.emptyList() : partitionIds;
    }

    protected boolean isOwnerOrBackup(int partitionId) {
        NodeEngine nodeEngine = this.mapServiceContext.getNodeEngine();
        IPartitionService partitionService = nodeEngine.getPartitionService();
        IPartition partition = partitionService.getPartition(partitionId, false);
        Address thisAddress = nodeEngine.getThisAddress();
        return partition.isOwnerOrBackup(thisAddress);
    }
}

