/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.kubernetes;

import com.hazelcast.core.HazelcastException;
import com.hazelcast.instance.impl.ClusterTopologyIntentTracker;
import com.hazelcast.internal.json.Json;
import com.hazelcast.internal.json.JsonArray;
import com.hazelcast.internal.json.JsonObject;
import com.hazelcast.internal.json.JsonValue;
import com.hazelcast.internal.util.HostnameUtil;
import com.hazelcast.internal.util.StringUtil;
import com.hazelcast.internal.util.concurrent.BackoffIdleStrategy;
import com.hazelcast.kubernetes.KubernetesApiEndpointProvider;
import com.hazelcast.kubernetes.KubernetesApiEndpointSlicesProvider;
import com.hazelcast.kubernetes.KubernetesApiProvider;
import com.hazelcast.kubernetes.KubernetesClientException;
import com.hazelcast.kubernetes.KubernetesConfig;
import com.hazelcast.kubernetes.KubernetesTokenProvider;
import com.hazelcast.kubernetes.RuntimeContext;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.Logger;
import com.hazelcast.spi.exception.RestClientException;
import com.hazelcast.spi.utils.RestClient;
import com.hazelcast.spi.utils.RetryUtils;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

class KubernetesClient {
    private static final ILogger LOGGER = Logger.getLogger(KubernetesClient.class);
    private static final int HTTP_GONE = 410;
    private static final int HTTP_UNAUTHORIZED = 401;
    private static final int HTTP_FORBIDDEN = 403;
    private static final List<String> NON_RETRYABLE_KEYWORDS = Arrays.asList("\"reason\":\"Forbidden\"", "\"reason\":\"NotFound\"", "Failure in generating SSLSocketFactory");
    private final String stsName;
    private final String namespace;
    private final String kubernetesMaster;
    private final String caCertificate;
    private final int retries;
    private final KubernetesApiProvider apiProvider;
    private final KubernetesConfig.ExposeExternallyMode exposeExternallyMode;
    private final boolean useNodeNameAsExternalAddress;
    private final String servicePerPodLabelName;
    private final String servicePerPodLabelValue;
    @Nullable
    private final Thread stsMonitorThread;
    private final KubernetesTokenProvider tokenProvider;
    @Nullable
    private final ClusterTopologyIntentTracker clusterTopologyIntentTracker;
    private boolean isNoPublicIpAlreadyLogged;
    private boolean isKnownExceptionAlreadyLogged;

    KubernetesClient(String namespace, String kubernetesMaster, KubernetesTokenProvider tokenProvider, String caCertificate, int retries, KubernetesConfig.ExposeExternallyMode exposeExternallyMode, boolean useNodeNameAsExternalAddress, String servicePerPodLabelName, String servicePerPodLabelValue, @Nullable ClusterTopologyIntentTracker clusterTopologyIntentTracker) {
        this.namespace = namespace;
        this.kubernetesMaster = kubernetesMaster;
        this.tokenProvider = tokenProvider;
        this.caCertificate = caCertificate;
        this.retries = retries;
        this.exposeExternallyMode = exposeExternallyMode;
        this.useNodeNameAsExternalAddress = useNodeNameAsExternalAddress;
        this.servicePerPodLabelName = servicePerPodLabelName;
        this.servicePerPodLabelValue = servicePerPodLabelValue;
        this.clusterTopologyIntentTracker = clusterTopologyIntentTracker;
        if (clusterTopologyIntentTracker != null) {
            clusterTopologyIntentTracker.initialize();
        }
        this.apiProvider = this.buildKubernetesApiUrlProvider();
        this.stsName = this.extractStsName();
        this.stsMonitorThread = clusterTopologyIntentTracker != null && clusterTopologyIntentTracker.isEnabled() ? new Thread((Runnable)new StsMonitor(), "hz-k8s-sts-monitor") : null;
    }

    KubernetesClient(String namespace, String kubernetesMaster, KubernetesTokenProvider tokenProvider, String caCertificate, int retries, KubernetesConfig.ExposeExternallyMode exposeExternallyMode, boolean useNodeNameAsExternalAddress, String servicePerPodLabelName, String servicePerPodLabelValue, @Nullable ClusterTopologyIntentTracker clusterTopologyIntentTracker, String stsName) {
        this.namespace = namespace;
        this.kubernetesMaster = kubernetesMaster;
        this.tokenProvider = tokenProvider;
        this.caCertificate = caCertificate;
        this.retries = retries;
        this.exposeExternallyMode = exposeExternallyMode;
        this.useNodeNameAsExternalAddress = useNodeNameAsExternalAddress;
        this.servicePerPodLabelName = servicePerPodLabelName;
        this.servicePerPodLabelValue = servicePerPodLabelValue;
        this.clusterTopologyIntentTracker = clusterTopologyIntentTracker;
        if (clusterTopologyIntentTracker != null) {
            clusterTopologyIntentTracker.initialize();
        }
        this.apiProvider = this.buildKubernetesApiUrlProvider();
        this.stsName = stsName;
        this.stsMonitorThread = clusterTopologyIntentTracker != null && clusterTopologyIntentTracker.isEnabled() ? new Thread((Runnable)new StsMonitor(), "hz-k8s-sts-monitor") : null;
    }

    KubernetesClient(String namespace, String kubernetesMaster, KubernetesTokenProvider tokenProvider, String caCertificate, int retries, KubernetesConfig.ExposeExternallyMode exposeExternallyMode, boolean useNodeNameAsExternalAddress, String servicePerPodLabelName, String servicePerPodLabelValue, KubernetesApiProvider apiProvider) {
        this.namespace = namespace;
        this.kubernetesMaster = kubernetesMaster;
        this.tokenProvider = tokenProvider;
        this.caCertificate = caCertificate;
        this.retries = retries;
        this.exposeExternallyMode = exposeExternallyMode;
        this.useNodeNameAsExternalAddress = useNodeNameAsExternalAddress;
        this.servicePerPodLabelName = servicePerPodLabelName;
        this.servicePerPodLabelValue = servicePerPodLabelValue;
        this.apiProvider = apiProvider;
        this.stsMonitorThread = null;
        this.stsName = this.extractStsName();
        this.clusterTopologyIntentTracker = null;
    }

    public void start() {
        if (this.stsMonitorThread != null) {
            this.stsMonitorThread.start();
        }
    }

    public void destroy() {
        if (this.clusterTopologyIntentTracker != null) {
            this.clusterTopologyIntentTracker.destroy();
        }
        if (this.stsMonitorThread != null) {
            LOGGER.info("Interrupting StatefulSet monitor thread");
            this.stsMonitorThread.interrupt();
        }
    }

    KubernetesApiProvider buildKubernetesApiUrlProvider() {
        try {
            String endpointSlicesUrlString = String.format("%s/apis/discovery.k8s.io/v1/namespaces/%s/endpointslices", this.kubernetesMaster, this.namespace);
            this.callGet(endpointSlicesUrlString);
            LOGGER.finest("Using EndpointSlices API to discover endpoints.");
        }
        catch (Exception e) {
            LOGGER.finest("EndpointSlices are not available, using Endpoints API to discover endpoints.");
            return new KubernetesApiEndpointProvider();
        }
        return new KubernetesApiEndpointSlicesProvider();
    }

    List<Endpoint> endpoints() {
        try {
            String urlString = String.format("%s/api/v1/namespaces/%s/pods", this.kubernetesMaster, this.namespace);
            return this.enrichWithPublicAddresses(KubernetesClient.parsePodsList(this.callGet(urlString)));
        }
        catch (RestClientException e) {
            return this.handleKnownException(e);
        }
    }

    List<Endpoint> endpointsByServiceLabel(String serviceLabels, String serviceLabelValues) {
        try {
            String param = KubernetesClient.getLabelSelectorParameter(serviceLabels, serviceLabelValues);
            String urlString = String.format(this.apiProvider.getEndpointsByServiceLabelUrlString(), this.kubernetesMaster, this.namespace, param);
            return this.enrichWithPublicAddresses(this.apiProvider.parseEndpointsList(this.callGet(urlString)));
        }
        catch (RestClientException e) {
            return this.handleKnownException(e);
        }
    }

    private static String getLabelSelectorParameter(String labelNames, String labelValues) {
        ArrayList<String> labelNameList = new ArrayList<String>(Arrays.asList(labelNames.split(",")));
        ArrayList<String> labelValueList = new ArrayList<String>(Arrays.asList(labelValues.split(",")));
        ArrayList<String> selectorList = new ArrayList<String>(labelNameList.size());
        for (int i = 0; i < labelNameList.size(); ++i) {
            selectorList.add(i, String.format("%s=%s", labelNameList.get(i), labelValueList.get(i)));
        }
        return String.format("labelSelector=%s", String.join((CharSequence)",", selectorList));
    }

    List<Endpoint> endpointsByName(String endpointName) {
        try {
            String urlString = String.format(this.apiProvider.getEndpointsByNameUrlString(), this.kubernetesMaster, this.namespace, endpointName);
            return this.enrichWithPublicAddresses(this.apiProvider.parseEndpoints(this.callGet(urlString)));
        }
        catch (RestClientException e) {
            return this.handleKnownException(e);
        }
    }

    List<Endpoint> endpointsByPodLabel(String podLabels, String podLabelValues) {
        try {
            String param = KubernetesClient.getLabelSelectorParameter(podLabels, podLabelValues);
            String urlString = String.format("%s/api/v1/namespaces/%s/pods?%s", this.kubernetesMaster, this.namespace, param);
            return this.enrichWithPublicAddresses(KubernetesClient.parsePodsList(this.callGet(urlString)));
        }
        catch (RestClientException e) {
            return this.handleKnownException(e);
        }
    }

    String zone(String podName) {
        String nodeUrlString = String.format("%s/api/v1/nodes/%s", this.kubernetesMaster, this.nodeName(podName));
        return KubernetesClient.extractZone(this.callGet(nodeUrlString));
    }

    String nodeName(String podName) {
        String podUrlString = String.format("%s/api/v1/namespaces/%s/pods/%s", this.kubernetesMaster, this.namespace, podName);
        return KubernetesClient.extractNodeName(this.callGet(podUrlString));
    }

    boolean isKnownExceptionAlreadyLogged() {
        return this.isKnownExceptionAlreadyLogged;
    }

    private String extractStsName() {
        String stsName = HostnameUtil.getLocalHostname();
        int dashIndex = stsName.lastIndexOf(45);
        if (dashIndex > 0) {
            stsName = stsName.substring(0, dashIndex);
        }
        return stsName;
    }

    private RuntimeContext extractSts(JsonObject jsonObject) {
        int specReplicas = jsonObject.get("spec").asObject().getInt("replicas", -1);
        int readyReplicas = jsonObject.get("status").asObject().getInt("readyReplicas", -1);
        String resourceVersion = jsonObject.get("metadata").asObject().getString("resourceVersion", null);
        int replicas = jsonObject.get("status").asObject().getInt("currentReplicas", -1);
        return new RuntimeContext(specReplicas, readyReplicas, replicas, resourceVersion);
    }

    private static List<Endpoint> parsePodsList(JsonObject podsListJson) {
        ArrayList<Endpoint> addresses = new ArrayList<Endpoint>();
        for (JsonValue item : KubernetesClient.toJsonArray(podsListJson.get("items"))) {
            String podName = item.asObject().get("metadata").asObject().get("name").asString();
            JsonObject status = item.asObject().get("status").asObject();
            String ip = KubernetesClient.toString(status.get("podIP"));
            if (ip == null) continue;
            Integer port = KubernetesClient.extractContainerPort(item);
            addresses.add(new Endpoint(new EndpointAddress(ip, port, podName), KubernetesClient.isReady(status)));
        }
        return addresses;
    }

    private static Integer extractContainerPort(JsonValue podItemJson) {
        JsonValue port;
        JsonValue containerPort;
        JsonValue container;
        JsonArray ports;
        JsonArray containers = KubernetesClient.toJsonArray(podItemJson.asObject().get("spec").asObject().get("containers"));
        if (containers.size() == 1 && (ports = KubernetesClient.toJsonArray((container = containers.get(0)).asObject().get("ports"))).size() == 1 && (containerPort = (port = ports.get(0)).asObject().get("containerPort")) != null && containerPort.isNumber()) {
            return containerPort.asInt();
        }
        return null;
    }

    private static boolean isReady(JsonObject podItemStatusJson) {
        for (JsonValue containerStatus : KubernetesClient.toJsonArray(podItemStatusJson.get("containerStatuses"))) {
            if (containerStatus.asObject().get("ready").asBoolean()) continue;
            return false;
        }
        return true;
    }

    private static String extractNodeName(JsonObject podJson) {
        return KubernetesClient.toString(podJson.get("spec").asObject().get("nodeName"));
    }

    private static String extractZone(JsonObject nodeJson) {
        JsonObject labels = nodeJson.get("metadata").asObject().get("labels").asObject();
        List<String> zoneLabels = Arrays.asList("topology.kubernetes.io/zone", "failure-domain.kubernetes.io/zone", "failure-domain.beta.kubernetes.io/zone");
        for (String zoneLabel : zoneLabels) {
            JsonValue zone = labels.get(zoneLabel);
            if (zone == null) continue;
            return KubernetesClient.toString(zone);
        }
        return null;
    }

    private List<Endpoint> enrichWithPublicAddresses(List<Endpoint> endpoints) {
        if (this.exposeExternallyMode == KubernetesConfig.ExposeExternallyMode.DISABLED) {
            return endpoints;
        }
        try {
            String endpointsUrl = String.format(this.apiProvider.getEndpointsUrlString(), this.kubernetesMaster, this.namespace);
            if (!StringUtil.isNullOrEmptyAfterTrim(this.servicePerPodLabelName) && !StringUtil.isNullOrEmptyAfterTrim(this.servicePerPodLabelValue)) {
                endpointsUrl = endpointsUrl + String.format("?labelSelector=%s=%s", this.servicePerPodLabelName, this.servicePerPodLabelValue);
            }
            JsonObject endpointsJson = this.callGet(endpointsUrl);
            List<EndpointAddress> privateAddresses = KubernetesClient.privateAddresses(endpoints);
            Map<EndpointAddress, String> services = this.apiProvider.extractServices(endpointsJson, privateAddresses);
            Map<EndpointAddress, String> nodeAddresses = this.apiProvider.extractNodes(endpointsJson, privateAddresses);
            HashMap<EndpointAddress, String> publicIps = new HashMap<EndpointAddress, String>();
            HashMap<EndpointAddress, Integer> publicPorts = new HashMap<EndpointAddress, Integer>();
            HashMap<String, String> cachedNodePublicIps = new HashMap<String, String>();
            for (Map.Entry<EndpointAddress, String> serviceEntry : services.entrySet()) {
                EndpointAddress privateAddress = serviceEntry.getKey();
                String service = serviceEntry.getValue();
                String serviceUrl = String.format("%s/api/v1/namespaces/%s/services/%s", this.kubernetesMaster, this.namespace, service);
                JsonObject serviceJson = this.callGet(serviceUrl);
                try {
                    String loadBalancerAddress = KubernetesClient.extractLoadBalancerAddress(serviceJson);
                    Integer servicePort = KubernetesClient.extractServicePort(serviceJson);
                    publicIps.put(privateAddress, loadBalancerAddress);
                    publicPorts.put(privateAddress, servicePort);
                }
                catch (Exception e) {
                    String nodePublicAddress;
                    Integer nodePort = KubernetesClient.extractNodePort(serviceJson);
                    String node = this.extractNodeName(serviceEntry.getKey(), nodeAddresses);
                    if (cachedNodePublicIps.containsKey(node)) {
                        nodePublicAddress = (String)cachedNodePublicIps.get(node);
                    } else {
                        nodePublicAddress = this.externalAddressForNode(node);
                        cachedNodePublicIps.put(node, nodePublicAddress);
                    }
                    publicIps.put(privateAddress, nodePublicAddress);
                    publicPorts.put(privateAddress, nodePort);
                }
            }
            return KubernetesClient.createEndpoints(endpoints, publicIps, publicPorts);
        }
        catch (Exception e) {
            if (this.exposeExternallyMode == KubernetesConfig.ExposeExternallyMode.ENABLED) {
                throw e;
            }
            LOGGER.finest(e);
            if (!this.isNoPublicIpAlreadyLogged) {
                LOGGER.warning("Cannot fetch public IPs of Hazelcast Member PODs, you won't be able to use Hazelcast Smart Client from outside of the Kubernetes network");
                this.isNoPublicIpAlreadyLogged = true;
            }
            return endpoints;
        }
    }

    @Nullable
    private String extractNodeName(EndpointAddress endpointAddress, Map<EndpointAddress, String> nodes) {
        String nodeName = nodes.get(endpointAddress);
        if (nodeName == null) {
            JsonObject podJson = this.callGet(String.format("%s/api/v1/namespaces/%s/pods/%s", this.kubernetesMaster, this.namespace, endpointAddress.getTargetRefName()));
            return podJson.get("spec").asObject().get("nodeName").asString();
        }
        return nodeName;
    }

    private static List<EndpointAddress> privateAddresses(List<Endpoint> endpoints) {
        ArrayList<EndpointAddress> result = new ArrayList<EndpointAddress>();
        for (Endpoint endpoint : endpoints) {
            result.add(endpoint.getPrivateAddress());
        }
        return result;
    }

    private static String extractLoadBalancerAddress(JsonObject serviceResponse) {
        JsonObject ingress = serviceResponse.get("status").asObject().get("loadBalancer").asObject().get("ingress").asArray().get(0).asObject();
        JsonValue address = ingress.get("ip");
        if (address == null) {
            address = ingress.get("hostname");
        }
        return address.asString();
    }

    private static Integer extractServicePort(JsonObject serviceJson) {
        JsonArray ports = KubernetesClient.toJsonArray(serviceJson.get("spec").asObject().get("ports"));
        if (ports.size() != 1) {
            throw new KubernetesClientException(String.format("Cannot expose externally, service %s needs to have exactly one port defined", serviceJson.get("metadata").asObject().get("name")));
        }
        return ports.get(0).asObject().get("port").asInt();
    }

    private static Integer extractNodePort(JsonObject serviceJson) {
        JsonArray ports = KubernetesClient.toJsonArray(serviceJson.get("spec").asObject().get("ports"));
        if (ports.size() != 1) {
            throw new KubernetesClientException(String.format("Cannot expose externally, service %s needs to have exactly one nodePort defined", serviceJson.get("metadata").asObject().get("name")));
        }
        return ports.get(0).asObject().get("nodePort").asInt();
    }

    private String externalAddressForNode(String node) {
        String nodeExternalAddress;
        if (this.useNodeNameAsExternalAddress) {
            LOGGER.info("Using node name instead of public IP for node, must be available from client: " + node);
            nodeExternalAddress = node;
        } else {
            String nodeUrl = String.format("%s/api/v1/nodes/%s", this.kubernetesMaster, node);
            nodeExternalAddress = KubernetesClient.extractNodePublicIp(this.callGet(nodeUrl));
        }
        return nodeExternalAddress;
    }

    private static String extractNodePublicIp(JsonObject nodeJson) {
        for (JsonValue address : KubernetesClient.toJsonArray(nodeJson.get("status").asObject().get("addresses"))) {
            if (!"ExternalIP".equals(address.asObject().get("type").asString())) continue;
            return address.asObject().get("address").asString();
        }
        throw new KubernetesClientException(String.format("Cannot expose externally, node %s does not have ExternalIP assigned", nodeJson.get("metadata").asObject().get("name")));
    }

    private static List<Endpoint> createEndpoints(List<Endpoint> endpoints, Map<EndpointAddress, String> publicIps, Map<EndpointAddress, Integer> publicPorts) {
        ArrayList<Endpoint> result = new ArrayList<Endpoint>();
        for (Endpoint endpoint : endpoints) {
            EndpointAddress privateAddress = endpoint.getPrivateAddress();
            EndpointAddress publicAddress = new EndpointAddress(publicIps.get(privateAddress), publicPorts.get(privateAddress), privateAddress.getTargetRefName());
            result.add(new Endpoint(privateAddress, publicAddress, endpoint.isReady(), endpoint.getAdditionalProperties()));
        }
        return result;
    }

    private JsonObject callGet(String urlString) {
        return RetryUtils.retry(() -> Json.parse(RestClient.create(urlString).withHeader("Authorization", String.format("Bearer %s", this.tokenProvider.getToken())).withCaCertificates(this.caCertificate).get().getBody()).asObject(), this.retries, NON_RETRYABLE_KEYWORDS);
    }

    private List<Endpoint> handleKnownException(RestClientException e) {
        if (e.getHttpErrorCode() == 401) {
            if (!this.isKnownExceptionAlreadyLogged) {
                LOGGER.warning("Kubernetes API authorization failure! To use Hazelcast Kubernetes discovery, please check your 'api-token' property. Starting standalone.");
                this.isKnownExceptionAlreadyLogged = true;
            }
        } else if (e.getHttpErrorCode() == 403) {
            if (!this.isKnownExceptionAlreadyLogged) {
                LOGGER.warning("Kubernetes API access is forbidden! Starting standalone. To use Hazelcast Kubernetes discovery, configure the required RBAC. For 'default' service account in 'default' namespace execute: `kubectl apply -f https://raw.githubusercontent.com/hazelcast/hazelcast/master/kubernetes-rbac.yaml`");
                this.isKnownExceptionAlreadyLogged = true;
            }
        } else {
            throw e;
        }
        LOGGER.finest(e);
        return Collections.emptyList();
    }

    private static JsonArray toJsonArray(JsonValue jsonValue) {
        if (jsonValue == null || jsonValue.isNull()) {
            return new JsonArray();
        }
        return jsonValue.asArray();
    }

    private static String toString(JsonValue jsonValue) {
        if (jsonValue == null || jsonValue.isNull()) {
            return null;
        }
        if (jsonValue.isString()) {
            return jsonValue.asString();
        }
        return jsonValue.toString();
    }

    final class StsMonitor
    implements Runnable {
        private static final int MAX_SPINS = 3;
        private static final int MAX_YIELDS = 10;
        private static final int MIN_PARK_PERIOD_MILLIS = 1;
        private static final int MAX_PARK_PERIOD_SECONDS = 10;
        volatile boolean running = true;
        String latestResourceVersion;
        RuntimeContext latestRuntimeContext;
        int idleCount;
        private final String stsUrlString = this.formatStsListUrl();
        private final BackoffIdleStrategy backoffIdleStrategy = new BackoffIdleStrategy(3L, 10L, TimeUnit.MILLISECONDS.toNanos(1L), TimeUnit.SECONDS.toNanos(10L));

        StsMonitor() {
        }

        @Override
        public void run() {
            while (this.running && !Thread.interrupted()) {
                RestClient.WatchResponse watchResponse;
                try {
                    RuntimeContext previous = this.latestRuntimeContext;
                    this.readInitialStsList();
                    this.updateTracker(previous, this.latestRuntimeContext);
                    watchResponse = this.sendWatchRequest();
                }
                catch (RestClientException e) {
                    this.handleFailure(e);
                    continue;
                }
                this.idleCount = 0;
                try {
                    String message;
                    while ((message = watchResponse.nextLine()) != null) {
                        this.onMessage(message);
                    }
                }
                catch (IOException e) {
                    LOGGER.info("Exception while watching for StatefulSet changes", e);
                    try {
                        watchResponse.disconnect();
                    }
                    catch (Exception t) {
                        LOGGER.fine("Exception while closing connection after an IOException", t);
                    }
                }
            }
        }

        private void handleFailure(RestClientException e) {
            if (e.getHttpErrorCode() == 410) {
                LOGGER.info("StatefulSet watcher has fallen behind, re-reading sts list and resuming watch: " + e.getMessage());
            } else {
                LOGGER.warning("Error while attempting to watch kubernetes API for StatefulSets: " + e.getHttpErrorCode() + " " + e.getMessage() + ". Backing off (n: " + this.idleCount + " ) before retrying.");
                this.backoffIdleStrategy.idle(this.idleCount);
                ++this.idleCount;
            }
        }

        String formatStsListUrl() {
            String fieldSelectorValue = String.format("metadata.name=%s", KubernetesClient.this.stsName);
            try {
                fieldSelectorValue = URLEncoder.encode(fieldSelectorValue, StandardCharsets.UTF_8.name());
            }
            catch (UnsupportedEncodingException e) {
                throw new HazelcastException(e);
            }
            return String.format("%s/apis/apps/v1/namespaces/%s/statefulsets?fieldSelector=%s", KubernetesClient.this.kubernetesMaster, KubernetesClient.this.namespace, fieldSelectorValue);
        }

        void readInitialStsList() {
            JsonObject jsonObject = KubernetesClient.this.callGet(this.stsUrlString);
            this.latestResourceVersion = jsonObject.get("metadata").asObject().getString("resourceVersion", null);
            this.latestRuntimeContext = this.parseStsList(jsonObject);
        }

        @Nonnull
        RestClient.WatchResponse sendWatchRequest() {
            RestClient restClient = RestClient.create(this.stsUrlString).withHeader("Authorization", String.format("Bearer %s", KubernetesClient.this.tokenProvider.getToken())).withCaCertificates(KubernetesClient.this.caCertificate);
            return restClient.watch(this.latestResourceVersion);
        }

        @Nullable
        RuntimeContext parseStsList(JsonObject jsonObject) {
            String resourceVersion = jsonObject.get("metadata").asObject().getString("resourceVersion", null);
            for (JsonValue item : KubernetesClient.toJsonArray(jsonObject.get("items"))) {
                String itemName = item.asObject().get("metadata").asObject().getString("name", null);
                if (!KubernetesClient.this.stsName.equals(itemName)) continue;
                int specReplicas = item.asObject().get("spec").asObject().getInt("replicas", -1);
                int readyReplicas = item.asObject().get("status").asObject().getInt("readyReplicas", -1);
                int replicas = item.asObject().get("status").asObject().getInt("currentReplicas", -1);
                return new RuntimeContext(specReplicas, readyReplicas, replicas, resourceVersion);
            }
            return null;
        }

        void onMessage(String message) {
            if (LOGGER.isFinestEnabled()) {
                LOGGER.finest("Complete message from kubernetes API: " + message);
            }
            JsonObject jsonObject = Json.parse(message).asObject();
            JsonObject sts = jsonObject.get("object").asObject();
            String itemName = sts.asObject().get("metadata").asObject().getString("name", null);
            if (!KubernetesClient.this.stsName.equals(itemName)) {
                return;
            }
            String watchType = jsonObject.getString("type", null);
            RuntimeContext ctx = null;
            switch (watchType) {
                case "MODIFIED": {
                    ctx = KubernetesClient.this.extractSts(sts);
                    this.latestResourceVersion = ctx.getResourceVersion();
                    break;
                }
                case "DELETED": {
                    ctx = KubernetesClient.this.extractSts(sts);
                    this.latestResourceVersion = ctx.getResourceVersion();
                    ctx = new RuntimeContext(0, ctx.getReadyReplicas(), ctx.getCurrentReplicas(), ctx.getResourceVersion());
                    break;
                }
                case "ADDED": {
                    throw new IllegalStateException("A new sts with same name as this cannot be added");
                }
                default: {
                    LOGGER.info("Unknown watch type " + watchType + ", complete message:\n" + message);
                }
            }
            if (this.latestRuntimeContext != null && ctx != null) {
                this.updateTracker(this.latestRuntimeContext, ctx);
            }
            this.latestRuntimeContext = ctx;
        }

        void updateTracker(RuntimeContext previous, RuntimeContext updated) {
            if (previous != null) {
                LOGGER.info("Updating cluster topology tracker with previous: " + previous + ", updated: " + updated);
                KubernetesClient.this.clusterTopologyIntentTracker.update(previous.getSpecifiedReplicaCount(), updated.getSpecifiedReplicaCount(), previous.getReadyReplicas(), updated.getReadyReplicas(), previous.getCurrentReplicas(), updated.getCurrentReplicas());
            } else {
                LOGGER.info("Initializing cluster topology tracker with initial context: " + this.latestRuntimeContext);
                KubernetesClient.this.clusterTopologyIntentTracker.update(-1, updated.getSpecifiedReplicaCount(), -1, updated.getReadyReplicas(), -1, updated.getCurrentReplicas());
            }
        }
    }

    static final class EndpointAddress {
        private final String ip;
        private final Integer port;
        private String targetRefName;

        EndpointAddress(String ip, Integer port) {
            this.ip = ip;
            this.port = port;
        }

        EndpointAddress(String ip, Integer port, String targetRefName) {
            this.ip = ip;
            this.port = port;
            this.targetRefName = targetRefName;
        }

        String getIp() {
            return this.ip;
        }

        Integer getPort() {
            return this.port;
        }

        String getTargetRefName() {
            return this.targetRefName;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            EndpointAddress address = (EndpointAddress)o;
            if (!Objects.equals(this.ip, address.ip) || !Objects.equals(this.targetRefName, address.targetRefName)) {
                return false;
            }
            return Objects.equals(this.port, address.port);
        }

        public int hashCode() {
            return Objects.hash(this.ip, this.port, this.targetRefName);
        }

        public String toString() {
            return String.format("%s:%s", this.ip, this.port);
        }
    }

    static final class Endpoint {
        private final EndpointAddress privateAddress;
        private final EndpointAddress publicAddress;
        private final boolean isReady;
        private final Map<String, String> additionalProperties;

        Endpoint(EndpointAddress privateAddress, boolean isReady) {
            this.privateAddress = privateAddress;
            this.publicAddress = null;
            this.isReady = isReady;
            this.additionalProperties = Collections.emptyMap();
        }

        Endpoint(EndpointAddress privateAddress, boolean isReady, Map<String, String> additionalProperties) {
            this.privateAddress = privateAddress;
            this.publicAddress = null;
            this.isReady = isReady;
            this.additionalProperties = additionalProperties;
        }

        Endpoint(EndpointAddress privateAddress, EndpointAddress publicAddress, boolean isReady, Map<String, String> additionalProperties) {
            this.privateAddress = privateAddress;
            this.publicAddress = publicAddress;
            this.isReady = isReady;
            this.additionalProperties = additionalProperties;
        }

        EndpointAddress getPublicAddress() {
            return this.publicAddress;
        }

        EndpointAddress getPrivateAddress() {
            return this.privateAddress;
        }

        boolean isReady() {
            return this.isReady;
        }

        Map<String, String> getAdditionalProperties() {
            return this.additionalProperties;
        }
    }
}

