/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.client.connection.nio;

import com.hazelcast.client.config.ClientConfig;
import com.hazelcast.client.config.ClientConnectionStrategyConfig;
import com.hazelcast.client.config.ClientNetworkConfig;
import com.hazelcast.client.config.ConnectionRetryConfig;
import com.hazelcast.client.connection.AddressProvider;
import com.hazelcast.client.connection.ClientConnectionManager;
import com.hazelcast.client.connection.ClientConnectionStrategy;
import com.hazelcast.client.connection.nio.ClientConnection;
import com.hazelcast.client.connection.nio.ClientConnectionManagerImpl;
import com.hazelcast.client.impl.clientside.HazelcastClientInstanceImpl;
import com.hazelcast.client.impl.clientside.LifecycleServiceImpl;
import com.hazelcast.client.spi.impl.ClientExecutionServiceImpl;
import com.hazelcast.client.spi.properties.ClientProperty;
import com.hazelcast.core.LifecycleEvent;
import com.hazelcast.core.Member;
import com.hazelcast.logging.ILogger;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.Connection;
import com.hazelcast.util.ExceptionUtil;
import com.hazelcast.util.executor.SingleExecutorThreadFactory;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;

class ClusterConnector {
    private static final int DEFAULT_CONNECTION_ATTEMPT_LIMIT_SYNC = 2;
    private static final int DEFAULT_CONNECTION_ATTEMPT_LIMIT_ASYNC = 20;
    private final ILogger logger;
    private final HazelcastClientInstanceImpl client;
    private final ClientConnectionManagerImpl connectionManager;
    private final ClientConnectionStrategy connectionStrategy;
    private final ExecutorService clusterConnectionExecutor;
    private final boolean shuffleMemberList;
    private final WaitStrategy waitStrategy;
    private final Collection<AddressProvider> addressProviders;
    private volatile Address ownerConnectionAddress;
    private volatile Address previousOwnerConnectionAddress;

    ClusterConnector(HazelcastClientInstanceImpl client, ClientConnectionManagerImpl connectionManager, ClientConnectionStrategy connectionStrategy, Collection<AddressProvider> addressProviders) {
        this.client = client;
        this.connectionManager = connectionManager;
        this.logger = client.getLoggingService().getLogger(ClientConnectionManager.class);
        this.connectionStrategy = connectionStrategy;
        this.clusterConnectionExecutor = this.createSingleThreadExecutorService(client);
        this.shuffleMemberList = client.getProperties().getBoolean(ClientProperty.SHUFFLE_MEMBER_LIST);
        this.addressProviders = addressProviders;
        this.waitStrategy = this.initializeWaitStrategy(client.getClientConfig());
    }

    private WaitStrategy initializeWaitStrategy(ClientConfig clientConfig) {
        ClientConnectionStrategyConfig connectionStrategyConfig = this.client.getClientConfig().getConnectionStrategyConfig();
        ConnectionRetryConfig expoRetryConfig = connectionStrategyConfig.getConnectionRetryConfig();
        if (expoRetryConfig.isEnabled()) {
            return new ExponentialWaitStrategy(expoRetryConfig.getInitialBackoffMillis(), expoRetryConfig.getMaxBackoffMillis(), expoRetryConfig.getMultiplier(), expoRetryConfig.isFailOnMaxBackoff(), expoRetryConfig.getJitter());
        }
        ClientNetworkConfig networkConfig = clientConfig.getNetworkConfig();
        int connectionAttemptPeriod = networkConfig.getConnectionAttemptPeriod();
        boolean isAsync = connectionStrategyConfig.isAsyncStart();
        int connectionAttemptLimit = networkConfig.getConnectionAttemptLimit();
        connectionAttemptLimit = connectionAttemptLimit < 0 ? (isAsync ? 20 : 2) : (connectionAttemptLimit == 0 ? Integer.MAX_VALUE : connectionAttemptLimit);
        return new DefaultWaitStrategy(connectionAttemptPeriod, connectionAttemptLimit);
    }

    void connectToCluster() {
        try {
            this.connectToClusterAsync().get();
        }
        catch (Exception e) {
            throw ExceptionUtil.rethrow((Throwable)e);
        }
    }

    Address getOwnerConnectionAddress() {
        return this.ownerConnectionAddress;
    }

    void setOwnerConnectionAddress(Address ownerConnectionAddress) {
        this.previousOwnerConnectionAddress = this.ownerConnectionAddress;
        this.ownerConnectionAddress = ownerConnectionAddress;
    }

    Connection connectAsOwner(Address address) {
        Connection connection = null;
        try {
            this.logger.info("Trying to connect to " + address + " as owner member");
            connection = this.connectionManager.getOrConnect(address, true);
            this.client.onClusterConnect(connection);
            this.fireConnectionEvent(LifecycleEvent.LifecycleState.CLIENT_CONNECTED);
            this.connectionStrategy.onConnectToCluster();
        }
        catch (Exception e) {
            this.logger.warning("Exception during initial connection to " + address + ", exception " + e);
            if (null != connection) {
                connection.close("Could not connect to " + address + " as owner", (Throwable)e);
            }
            return null;
        }
        return connection;
    }

    void disconnectFromCluster(final ClientConnection connection) {
        this.clusterConnectionExecutor.execute(new Runnable(){

            @Override
            public void run() {
                Address endpoint = connection.getEndPoint();
                if (endpoint == null || !endpoint.equals((Object)ClusterConnector.this.ownerConnectionAddress)) {
                    return;
                }
                ClusterConnector.this.setOwnerConnectionAddress(null);
                ClusterConnector.this.connectionStrategy.onDisconnectFromCluster();
                if (ClusterConnector.this.client.getLifecycleService().isRunning()) {
                    ClusterConnector.this.fireConnectionEvent(LifecycleEvent.LifecycleState.CLIENT_DISCONNECTED);
                }
            }
        });
    }

    private void fireConnectionEvent(LifecycleEvent.LifecycleState state) {
        LifecycleServiceImpl lifecycleService = (LifecycleServiceImpl)this.client.getLifecycleService();
        lifecycleService.fireLifecycleEvent(state);
    }

    private ExecutorService createSingleThreadExecutorService(HazelcastClientInstanceImpl client) {
        ClassLoader classLoader = client.getClientConfig().getClassLoader();
        SingleExecutorThreadFactory threadFactory = new SingleExecutorThreadFactory(classLoader, client.getName() + ".cluster-");
        return Executors.newSingleThreadExecutor((ThreadFactory)threadFactory);
    }

    private void connectToClusterInternal() {
        HashSet<Address> triedAddresses = new HashSet<Address>();
        this.waitStrategy.reset();
        do {
            Collection<Address> addresses = this.getPossibleMemberAddresses();
            for (Address address : addresses) {
                if (!this.client.getLifecycleService().isRunning()) {
                    throw new IllegalStateException("Giving up on retrying to connect to cluster since client is shutdown.");
                }
                triedAddresses.add(address);
                if (this.connectAsOwner(address) == null) continue;
                return;
            }
            if (this.client.getLifecycleService().isRunning()) continue;
            throw new IllegalStateException("Client is being shutdown.");
        } while (this.waitStrategy.sleep());
        throw new IllegalStateException("Unable to connect to any address! The following addresses were tried: " + triedAddresses);
    }

    Future<Void> connectToClusterAsync() {
        return this.clusterConnectionExecutor.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                try {
                    ClusterConnector.this.connectToClusterInternal();
                }
                catch (Exception e) {
                    ClusterConnector.this.logger.warning("Could not connect to cluster, shutting down the client. " + e.getMessage());
                    new Thread(new Runnable(){

                        @Override
                        public void run() {
                            try {
                                ClusterConnector.this.client.getLifecycleService().shutdown();
                            }
                            catch (Exception exception) {
                                ClusterConnector.this.logger.severe("Exception during client shutdown ", (Throwable)exception);
                            }
                        }
                    }, ClusterConnector.this.client.getName() + ".clientShutdown-").start();
                    throw ExceptionUtil.rethrow((Throwable)e);
                }
                return null;
            }
        });
    }

    Collection<Address> getPossibleMemberAddresses() {
        LinkedHashSet addresses = new LinkedHashSet();
        Collection<Member> memberList = this.client.getClientClusterService().getMemberList();
        for (Member member : memberList) {
            addresses.add(member.getAddress());
        }
        if (this.shuffleMemberList) {
            addresses = (LinkedHashSet)ClusterConnector.shuffle(addresses);
        }
        LinkedHashSet providerAddresses = new LinkedHashSet();
        for (AddressProvider addressProvider : this.addressProviders) {
            try {
                providerAddresses.addAll(addressProvider.loadAddresses());
            }
            catch (NullPointerException e) {
                throw e;
            }
            catch (Exception e) {
                this.logger.warning("Exception from AddressProvider: " + addressProvider, (Throwable)e);
            }
        }
        if (this.shuffleMemberList) {
            providerAddresses = (LinkedHashSet)ClusterConnector.shuffle(providerAddresses);
        }
        addresses.addAll(providerAddresses);
        if (this.previousOwnerConnectionAddress != null) {
            addresses.remove(this.previousOwnerConnectionAddress);
            addresses.add(this.previousOwnerConnectionAddress);
        }
        return addresses;
    }

    private static <T> Set<T> shuffle(Set<T> set) {
        ArrayList<T> shuffleMe = new ArrayList<T>(set);
        Collections.shuffle(shuffleMe);
        return new LinkedHashSet<T>(shuffleMe);
    }

    public void shutdown() {
        ClientExecutionServiceImpl.shutdownExecutor("cluster", this.clusterConnectionExecutor, this.logger);
    }

    class ExponentialWaitStrategy
    implements WaitStrategy {
        private final int initialBackoffMillis;
        private final int maxBackoffMillis;
        private final double multiplier;
        private final boolean failOnMaxBackoff;
        private final double jitter;
        private final Random random = new Random();
        private int attempt;
        private int currentBackoffMillis;

        ExponentialWaitStrategy(int initialBackoffMillis, int maxBackoffMillis, double multiplier, boolean failOnMaxBackoff, double jitter) {
            this.initialBackoffMillis = initialBackoffMillis;
            this.maxBackoffMillis = maxBackoffMillis;
            this.multiplier = multiplier;
            this.failOnMaxBackoff = failOnMaxBackoff;
            this.jitter = jitter;
        }

        @Override
        public void reset() {
            this.attempt = 0;
            this.currentBackoffMillis = Math.min(this.maxBackoffMillis, this.initialBackoffMillis);
        }

        @Override
        public boolean sleep() {
            ++this.attempt;
            if (this.failOnMaxBackoff && this.currentBackoffMillis >= this.maxBackoffMillis) {
                ClusterConnector.this.logger.warning(String.format("Unable to get alive cluster connection, attempt %d.", this.attempt));
                return false;
            }
            long actualSleepTime = (long)((double)this.currentBackoffMillis - (double)this.currentBackoffMillis * this.jitter + (double)this.currentBackoffMillis * this.jitter * this.random.nextDouble());
            ClusterConnector.this.logger.warning(String.format("Unable to get alive cluster connection, try in %d ms later, attempt %d , cap retry timeout millis %d", actualSleepTime, this.attempt, this.maxBackoffMillis));
            try {
                Thread.sleep(actualSleepTime);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
            this.currentBackoffMillis = (int)Math.min((double)this.currentBackoffMillis * this.multiplier, (double)this.maxBackoffMillis);
            return true;
        }
    }

    class DefaultWaitStrategy
    implements WaitStrategy {
        private final int connectionAttemptPeriod;
        private final int connectionAttemptLimit;
        private int attempt;

        DefaultWaitStrategy(int connectionAttemptPeriod, int connectionAttemptLimit) {
            this.connectionAttemptPeriod = connectionAttemptPeriod;
            this.connectionAttemptLimit = connectionAttemptLimit;
        }

        @Override
        public void reset() {
            this.attempt = 0;
        }

        @Override
        public boolean sleep() {
            ++this.attempt;
            if (this.attempt >= this.connectionAttemptLimit) {
                ClusterConnector.this.logger.warning(String.format("Unable to get alive cluster connection, attempt %d of %d.", this.attempt, this.connectionAttemptLimit));
                return false;
            }
            ClusterConnector.this.logger.warning(String.format("Unable to get alive cluster connection, try in %d ms later, attempt %d of %d.", this.connectionAttemptPeriod, this.attempt, this.connectionAttemptLimit));
            try {
                Thread.sleep(this.connectionAttemptPeriod);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
            return true;
        }
    }

    static interface WaitStrategy {
        public void reset();

        public boolean sleep();
    }
}

