/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.nio;

import com.hazelcast.cluster.Bind;
import com.hazelcast.config.SSLConfig;
import com.hazelcast.config.SocketInterceptorConfig;
import com.hazelcast.impl.ClusterOperation;
import com.hazelcast.impl.ThreadContext;
import com.hazelcast.logging.ILogger;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.CipherHelper;
import com.hazelcast.nio.Connection;
import com.hazelcast.nio.ConnectionListener;
import com.hazelcast.nio.ConnectionMonitor;
import com.hazelcast.nio.Data;
import com.hazelcast.nio.DefaultSocketChannelWrapper;
import com.hazelcast.nio.IOService;
import com.hazelcast.nio.InOutSelector;
import com.hazelcast.nio.MemberSocketInterceptor;
import com.hazelcast.nio.Packet;
import com.hazelcast.nio.SocketAcceptor;
import com.hazelcast.nio.SocketChannelWrapper;
import com.hazelcast.nio.SocketConnector;
import com.hazelcast.nio.SocketInterceptor;
import com.hazelcast.nio.ssl.BasicSSLContextFactory;
import com.hazelcast.nio.ssl.SSLContextFactory;
import com.hazelcast.nio.ssl.SSLSocketChannelWrapper;
import com.hazelcast.util.Clock;
import com.hazelcast.util.ConcurrentHashSet;
import java.io.IOException;
import java.net.Socket;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

public class ConnectionManager {
    protected final ILogger logger;
    final int SOCKET_RECEIVE_BUFFER_SIZE;
    final int SOCKET_SEND_BUFFER_SIZE;
    final int SOCKET_LINGER_SECONDS;
    final boolean SOCKET_KEEP_ALIVE;
    final boolean SOCKET_NO_DELAY;
    private final Map<Address, Connection> mapConnections = new ConcurrentHashMap<Address, Connection>(100);
    private final ConcurrentMap<Address, ConnectionMonitor> mapMonitors = new ConcurrentHashMap<Address, ConnectionMonitor>(100);
    private final Set<Address> setConnectionInProgress = new ConcurrentHashSet<Address>();
    private final Set<ConnectionListener> setConnectionListeners = new CopyOnWriteArraySet<ConnectionListener>();
    private final Set<Connection> setActiveConnections = new ConcurrentHashSet<Connection>();
    private final AtomicInteger allTextConnections = new AtomicInteger();
    private final AtomicInteger connectionIdGen = new AtomicInteger();
    private volatile boolean live = false;
    final IOService ioService;
    private final ServerSocketChannel serverSocketChannel;
    private final InOutSelector[] selectors;
    private final AtomicInteger nextSelectorIndex = new AtomicInteger();
    private final MemberSocketInterceptor memberSocketInterceptor;
    private final ExecutorService es = Executors.newCachedThreadPool();
    private final SocketChannelWrapperFactory socketChannelWrapperFactory;
    private Thread socketAcceptorThread;

    public ConnectionManager(IOService ioService, ServerSocketChannel serverSocketChannel) {
        this.ioService = ioService;
        this.serverSocketChannel = serverSocketChannel;
        this.logger = ioService.getLogger(ConnectionManager.class.getName());
        this.SOCKET_RECEIVE_BUFFER_SIZE = ioService.getSocketReceiveBufferSize() * 1024;
        this.SOCKET_SEND_BUFFER_SIZE = ioService.getSocketSendBufferSize() * 1024;
        this.SOCKET_LINGER_SECONDS = ioService.getSocketLingerSeconds();
        this.SOCKET_KEEP_ALIVE = ioService.getSocketKeepAlive();
        this.SOCKET_NO_DELAY = ioService.getSocketNoDelay();
        int selectorCount = ioService.getSelectorThreadCount();
        this.selectors = new InOutSelector[selectorCount];
        SSLConfig sslConfig = ioService.getSSLConfig();
        if (sslConfig != null && sslConfig.isEnabled()) {
            this.socketChannelWrapperFactory = new SSLSocketChannelWrapperFactory(sslConfig);
            this.logger.log(Level.INFO, "SSL is enabled");
        } else {
            this.socketChannelWrapperFactory = new DefaultSocketChannelWrapperFactory();
        }
        SocketInterceptorConfig sic = ioService.getSocketInterceptorConfig();
        if (sic != null && sic.isEnabled()) {
            SocketInterceptor implementation = (SocketInterceptor)sic.getImplementation();
            if (implementation == null && sic.getClassName() != null) {
                try {
                    implementation = (SocketInterceptor)Class.forName(sic.getClassName()).newInstance();
                }
                catch (Throwable e) {
                    this.logger.log(Level.SEVERE, "SocketInterceptor class cannot be instantiated!" + sic.getClassName(), e);
                }
            }
            if (implementation != null) {
                if (!(implementation instanceof MemberSocketInterceptor)) {
                    this.logger.log(Level.SEVERE, "SocketInterceptor must be instance of " + MemberSocketInterceptor.class.getName());
                    implementation = null;
                } else {
                    this.logger.log(Level.INFO, "SocketInterceptor is enabled");
                }
            }
            if (implementation != null) {
                this.memberSocketInterceptor = (MemberSocketInterceptor)implementation;
                this.memberSocketInterceptor.init(sic);
            } else {
                this.memberSocketInterceptor = null;
            }
        } else {
            this.memberSocketInterceptor = null;
        }
    }

    public IOService getIOHandler() {
        return this.ioService;
    }

    void executeAsync(Runnable runnable) {
        this.es.execute(runnable);
    }

    public MemberSocketInterceptor getMemberSocketInterceptor() {
        return this.memberSocketInterceptor;
    }

    private InOutSelector nextSelector() {
        if (this.nextSelectorIndex.get() > 1000000) {
            this.nextSelectorIndex.set(0);
        }
        return this.selectors[Math.abs(this.nextSelectorIndex.incrementAndGet()) % this.selectors.length];
    }

    public void addConnectionListener(ConnectionListener listener) {
        this.setConnectionListeners.add(listener);
    }

    public boolean bind(Address endPoint, Connection connection, boolean accept) {
        connection.setEndPoint(endPoint);
        if (this.mapConnections.containsKey(endPoint)) {
            return false;
        }
        if (!endPoint.equals(this.ioService.getThisAddress())) {
            if (!connection.isClient()) {
                connection.setMonitor(this.getConnectionMonitor(endPoint, true));
            }
            if (!accept) {
                Packet bindPacket = this.createBindPacket(new Bind(this.ioService.getThisAddress()));
                connection.getWriteHandler().enqueueSocketWritable(bindPacket);
            }
            this.mapConnections.put(endPoint, connection);
            this.setConnectionInProgress.remove(endPoint);
            for (ConnectionListener listener : this.setConnectionListeners) {
                listener.connectionAdded(connection);
            }
        } else {
            return false;
        }
        return true;
    }

    private Packet createBindPacket(Bind rp) {
        Data value = ThreadContext.get().toData(rp);
        Packet packet = new Packet();
        packet.set("remotelyProcess", ClusterOperation.REMOTELY_PROCESS, null, value);
        packet.client = this.ioService.isClient();
        return packet;
    }

    Connection assignSocketChannel(SocketChannelWrapper channel) {
        InOutSelector selectorAssigned = this.nextSelector();
        Connection connection = new Connection(this, selectorAssigned, this.connectionIdGen.incrementAndGet(), channel);
        this.setActiveConnections.add(connection);
        selectorAssigned.addTask(connection.getReadHandler());
        selectorAssigned.selector.wakeup();
        this.logger.log(Level.INFO, channel.socket().getLocalPort() + " accepted socket connection from " + channel.socket().getRemoteSocketAddress());
        return connection;
    }

    SocketChannelWrapper wrapSocketChannel(SocketChannel socketChannel, boolean client) throws Exception {
        return this.socketChannelWrapperFactory.wrapSocketChannel(socketChannel, client);
    }

    void failedConnection(Address address, Throwable t, boolean silent) {
        this.setConnectionInProgress.remove(address);
        this.ioService.onFailedConnection(address);
        if (!silent) {
            this.getConnectionMonitor(address, false).onError(t);
        }
    }

    public Connection getConnection(Address address) {
        return this.mapConnections.get(address);
    }

    public Connection getOrConnect(Address address) {
        return this.getOrConnect(address, false);
    }

    public Connection getOrConnect(Address address, boolean silent) {
        Connection connection = this.mapConnections.get(address);
        if (connection == null && this.setConnectionInProgress.add(address)) {
            this.ioService.shouldConnectTo(address);
            this.executeAsync(new SocketConnector(this, address, silent));
        }
        return connection;
    }

    private ConnectionMonitor getConnectionMonitor(Address endpoint, boolean reset) {
        ConnectionMonitor monitorOld;
        ConnectionMonitor monitor = (ConnectionMonitor)this.mapMonitors.get(endpoint);
        if (monitor == null && (monitorOld = this.mapMonitors.putIfAbsent(endpoint, monitor = new ConnectionMonitor(this, endpoint))) != null) {
            monitor = monitorOld;
        }
        if (reset) {
            monitor.reset();
        }
        return monitor;
    }

    public Connection detachAndGetConnection(Address address) {
        return this.mapConnections.remove(address);
    }

    public void attachConnection(Address address, Connection conn) {
        this.mapConnections.put(address, conn);
    }

    public void destroyConnection(Connection connection) {
        if (connection == null) {
            return;
        }
        this.setActiveConnections.remove(connection);
        Address endPoint = connection.getEndPoint();
        if (endPoint != null) {
            this.setConnectionInProgress.remove(endPoint);
            Connection existingConn = this.mapConnections.get(endPoint);
            if (existingConn == connection) {
                this.mapConnections.remove(endPoint);
                for (ConnectionListener listener : this.setConnectionListeners) {
                    listener.connectionRemoved(connection);
                }
            }
        }
        if (connection.live()) {
            connection.close();
        }
    }

    public int getTotalWriteQueueSize() {
        int count = 0;
        for (Connection conn : this.mapConnections.values()) {
            if (!conn.live()) continue;
            count += conn.getWriteHandler().size();
        }
        return count;
    }

    protected void initSocket(Socket socket) throws Exception {
        if (this.SOCKET_LINGER_SECONDS > 0) {
            socket.setSoLinger(true, this.SOCKET_LINGER_SECONDS);
        }
        socket.setKeepAlive(this.SOCKET_KEEP_ALIVE);
        socket.setTcpNoDelay(this.SOCKET_NO_DELAY);
        socket.setReceiveBufferSize(this.SOCKET_RECEIVE_BUFFER_SIZE);
        socket.setSendBufferSize(this.SOCKET_SEND_BUFFER_SIZE);
    }

    public synchronized void start() {
        if (this.live) {
            return;
        }
        this.live = true;
        for (int i = 0; i < this.selectors.length; ++i) {
            InOutSelector s;
            this.selectors[i] = s = new InOutSelector(this);
            new Thread(this.ioService.getThreadGroup(), s, this.ioService.getThreadPrefix() + i).start();
        }
        if (this.serverSocketChannel != null) {
            if (this.socketAcceptorThread != null) {
                this.logger.log(Level.WARNING, "SocketAcceptor thread is already live! Shutting down old acceptor...");
                this.shutdownSocketAcceptor();
            }
            SocketAcceptor acceptRunnable = new SocketAcceptor(this.serverSocketChannel, this);
            this.socketAcceptorThread = new Thread(this.ioService.getThreadGroup(), acceptRunnable, this.ioService.getThreadPrefix() + "Acceptor");
            this.socketAcceptorThread.start();
        }
    }

    public synchronized void onRestart() {
        this.stop();
        this.start();
    }

    public synchronized void shutdown() {
        if (!this.live) {
            return;
        }
        this.live = false;
        this.stop();
        if (this.serverSocketChannel != null) {
            try {
                this.serverSocketChannel.close();
            }
            catch (IOException ignore) {
                this.logger.log(Level.FINEST, ignore.getMessage(), ignore);
            }
        }
        this.es.shutdownNow();
    }

    private void stop() {
        this.live = false;
        this.shutdownSocketAcceptor();
        this.ioService.onShutdown();
        for (Connection conn : this.mapConnections.values()) {
            try {
                this.destroyConnection(conn);
            }
            catch (Throwable ignore) {
                this.logger.log(Level.FINEST, ignore.getMessage(), ignore);
            }
        }
        for (Connection conn : this.setActiveConnections) {
            try {
                this.destroyConnection(conn);
            }
            catch (Throwable ignore) {
                this.logger.log(Level.FINEST, ignore.getMessage(), ignore);
            }
        }
        this.shutdownIOSelectors();
        this.setConnectionInProgress.clear();
        this.mapConnections.clear();
        this.mapMonitors.clear();
        this.setActiveConnections.clear();
    }

    private synchronized void shutdownIOSelectors() {
        for (int i = 0; i < this.selectors.length; ++i) {
            InOutSelector ioSelector = this.selectors[i];
            if (ioSelector != null) {
                ioSelector.shutdown();
            }
            this.selectors[i] = null;
        }
    }

    private synchronized void shutdownSocketAcceptor() {
        this.socketAcceptorThread.interrupt();
        this.socketAcceptorThread = null;
    }

    public int getCurrentClientConnections() {
        int count = 0;
        for (Connection conn : this.setActiveConnections) {
            if (!conn.live() || !conn.isClient()) continue;
            ++count;
        }
        return count;
    }

    public int getAllTextConnections() {
        return this.allTextConnections.get();
    }

    public void incrementTextConnections() {
        this.allTextConnections.incrementAndGet();
    }

    public boolean isLive() {
        return this.live;
    }

    public void appendState(StringBuffer sbState) {
        long now = Clock.currentTimeMillis();
        sbState.append("\nConnectionManager {");
        for (Connection conn : this.mapConnections.values()) {
            long wr = (now - conn.getWriteHandler().lastRegistration) / 1000L;
            long wh = (now - conn.getWriteHandler().lastHandle) / 1000L;
            long rr = (now - conn.getReadHandler().lastRegistration) / 1000L;
            long rh = (now - conn.getReadHandler().lastHandle) / 1000L;
            sbState.append("\n\tEndPoint: ").append(conn.getEndPoint());
            sbState.append("  ").append(conn.live());
            sbState.append("  ").append(conn.getWriteHandler().size());
            sbState.append("  w:").append(wr).append("/").append(wh);
            sbState.append("  r:").append(rr).append("/").append(rh);
        }
        sbState.append("\n}");
    }

    public String toString() {
        StringBuffer sb = new StringBuffer("Connections {");
        for (Connection conn : this.mapConnections.values()) {
            sb.append("\n");
            sb.append(conn);
        }
        sb.append("\nlive=");
        sb.append(this.live);
        sb.append("\n}");
        return sb.toString();
    }

    class SSLSocketChannelWrapperFactory
    implements SocketChannelWrapperFactory {
        final SSLContextFactory sslContextFactory;

        SSLSocketChannelWrapperFactory(SSLConfig sslConfig) {
            if (CipherHelper.isSymmetricEncryptionEnabled(ConnectionManager.this.ioService)) {
                throw new RuntimeException("SSL and SymmetricEncryption cannot be both enabled!");
            }
            if (CipherHelper.isAsymmetricEncryptionEnabled(ConnectionManager.this.ioService)) {
                throw new RuntimeException("SSL and AsymmetricEncryption cannot be both enabled!");
            }
            SSLContextFactory sslContextFactoryObject = (SSLContextFactory)sslConfig.getFactoryImplementation();
            try {
                String factoryClassName = sslConfig.getFactoryClassName();
                if (sslContextFactoryObject == null && factoryClassName != null) {
                    sslContextFactoryObject = (SSLContextFactory)Class.forName(factoryClassName).newInstance();
                }
                if (sslContextFactoryObject == null) {
                    sslContextFactoryObject = new BasicSSLContextFactory();
                }
                sslContextFactoryObject.init(sslConfig.getProperties());
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
            this.sslContextFactory = sslContextFactoryObject;
        }

        @Override
        public SocketChannelWrapper wrapSocketChannel(SocketChannel socketChannel, boolean client) throws Exception {
            return new SSLSocketChannelWrapper(this.sslContextFactory.getSSLContext(), socketChannel, client);
        }
    }

    class DefaultSocketChannelWrapperFactory
    implements SocketChannelWrapperFactory {
        DefaultSocketChannelWrapperFactory() {
        }

        @Override
        public SocketChannelWrapper wrapSocketChannel(SocketChannel socketChannel, boolean client) throws Exception {
            return new DefaultSocketChannelWrapper(socketChannel);
        }
    }

    static interface SocketChannelWrapperFactory {
        public SocketChannelWrapper wrapSocketChannel(SocketChannel var1, boolean var2) throws Exception;
    }
}

