/*******************************************************************************
 * Copyright 2011 See AUTHORS file.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
//--------------------------------- PACKAGE ------------------------------------
package com.guidebee.game.physics.joints;

//--------------------------------- IMPORTS ------------------------------------

import com.guidebee.game.physics.Joint;
import com.guidebee.game.physics.World;
import com.guidebee.math.Vector2;

//[------------------------------ MAIN CLASS ----------------------------------]

/**
 * Friction joint. This is used for top-down friction. It provides 2D
 * translational friction and angular friction.
 */
public class FrictionJoint extends Joint {

    private final float[] tmp = new float[2];
    private final Vector2 localAnchorA = new Vector2();
    private final Vector2 localAnchorB = new Vector2();

    public FrictionJoint(World world, long addr) {
        super(world, addr);
    }

    public Vector2 getLocalAnchorA() {
        jniGetLocalAnchorA(addr, tmp);
        localAnchorA.set(tmp[0], tmp[1]);
        return localAnchorA;
    }

    private native void jniGetLocalAnchorA(long addr, float[] anchor); /*
        b2FrictionJoint* joint = (b2FrictionJoint*)addr;
		anchor[0] = joint->GetLocalAnchorA().x;
		anchor[1] = joint->GetLocalAnchorA().y;
	*/

    public Vector2 getLocalAnchorB() {
        jniGetLocalAnchorB(addr, tmp);
        localAnchorB.set(tmp[0], tmp[1]);
        return localAnchorB;
    }

    private native void jniGetLocalAnchorB(long addr, float[] anchor); /*
		b2FrictionJoint* joint = (b2FrictionJoint*)addr;
		anchor[0] = joint->GetLocalAnchorB().x;
		anchor[1] = joint->GetLocalAnchorB().y;
	*/

    /**
     * Set the maximum friction force in N.
     */
    public void setMaxForce(float force) {
        jniSetMaxForce(addr, force);
    }

    private native void jniSetMaxForce(long addr, float force); /*
		b2FrictionJoint* joint = (b2FrictionJoint*)addr;
		joint->SetMaxForce( force );
	*/

    /**
     * Get the maximum friction force in N.
     */
    public float getMaxForce() {
        return jniGetMaxForce(addr);
    }

    private native float jniGetMaxForce(long addr); /*
		b2FrictionJoint* joint = (b2FrictionJoint*)addr;
		return joint->GetMaxForce();
	*/

    /**
     * Set the maximum friction torque in N*m.
     */
    public void setMaxTorque(float torque) {
        jniSetMaxTorque(addr, torque);
    }

    private native void jniSetMaxTorque(long addr, float torque); /*
		b2FrictionJoint* joint = (b2FrictionJoint*)addr;
		joint->SetMaxTorque( torque );
	*/

    /**
     * Get the maximum friction torque in N*m.
     */
    public float getMaxTorque() {
        return jniGetMaxTorque(addr);
    }

    private native float jniGetMaxTorque(long addr); /*
		b2FrictionJoint* joint = (b2FrictionJoint*)addr;
		return joint->GetMaxTorque();
	*/
}
