/*******************************************************************************
 * Copyright 2011 See AUTHORS file.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
//--------------------------------- PACKAGE ------------------------------------
package com.guidebee.game.engine.assets.loaders;

//--------------------------------- IMPORTS ------------------------------------
import com.guidebee.game.engine.assets.AssetDescriptor;
import com.guidebee.game.engine.assets.AssetLoaderParameters;
import com.guidebee.game.engine.assets.AssetManager;
import com.guidebee.game.files.FileHandle;
import com.guidebee.game.graphics.TextureAtlas;
import com.guidebee.game.ui.Skin;
import com.guidebee.utils.collections.Array;
import com.guidebee.utils.collections.ObjectMap;

//[------------------------------ MAIN CLASS ----------------------------------]
/**
 * {@link AssetLoader} for {@link Skin} instances. All
 * {@link com.guidebee.game.graphics.Texture} and
 * {@link com.guidebee.game.graphics.BitmapFont} instances will be
 * loaded as
 * dependencies. Passing a {@link SkinParameter} allows the exact name of the
 * texture associated with the skin to be specified.
 * Otherwise the skin texture is looked up just as with a call to
 * {@link Skin#Skin(com.guidebee.game.files.FileHandle)}. A
 * {@link SkinParameter} also allows named resources to be set that will be
 * added to the skin before loading the json file,
 * meaning that they can be referenced from inside the json file itself.
 * This is useful for dynamic resources such as a BitmapFont
 * generated through FreeTypeFontGenerator.
 *
 * @author Nathan Sweet
 */
public class SkinLoader extends AsynchronousAssetLoader<Skin,
        SkinLoader.SkinParameter> {
    public SkinLoader(FileHandleResolver resolver) {
        super(resolver);
    }

    @Override
    public Array<AssetDescriptor> getDependencies(String fileName,
                                                  FileHandle file, SkinParameter parameter) {
        Array<AssetDescriptor> deps = new Array();
        if (parameter == null)
            deps.add(new AssetDescriptor(file.pathWithoutExtension()
                    + ".atlas", TextureAtlas.class));
        else if (parameter.textureAtlasPath != null)
            deps.add(new AssetDescriptor(parameter.textureAtlasPath,
                    TextureAtlas.class));
        return deps;
    }

    @Override
    public void loadAsync(AssetManager manager, String fileName,
                          FileHandle file, SkinParameter parameter) {
    }

    @Override
    public Skin loadSync(AssetManager manager, String fileName,
                         FileHandle file, SkinParameter parameter) {
        String textureAtlasPath;
        ObjectMap<String, Object> resources;
        if (parameter == null) {
            textureAtlasPath = file.pathWithoutExtension() + ".atlas";
            resources = null;
        } else {
            textureAtlasPath = parameter.textureAtlasPath;
            resources = parameter.resources;
        }
        TextureAtlas atlas = manager.get(textureAtlasPath, TextureAtlas.class);
        Skin skin = new Skin(atlas);
        if (resources != null) {
            for (ObjectMap.Entry<String, Object> entry : resources.entries()) {
                skin.add(entry.key, entry.value);
            }
        }
        skin.load(file);
        return skin;
    }

    static public class SkinParameter extends AssetLoaderParameters<Skin> {
        public final String textureAtlasPath;
        public final ObjectMap<String, Object> resources;

        public SkinParameter() {
            this(null, null);
        }

        public SkinParameter(String textureAtlasPath) {
            this(textureAtlasPath, null);
        }

        public SkinParameter(String textureAtlasPath, ObjectMap<String, Object> resources) {
            this.textureAtlasPath = textureAtlasPath;
            this.resources = resources;
        }
    }
}
