/*******************************************************************************
 * Copyright 2011 See AUTHORS file.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
//--------------------------------- PACKAGE ------------------------------------
package com.guidebee.game.graphics;

//--------------------------------- IMPORTS ------------------------------------

import com.guidebee.game.Application;
import com.guidebee.game.GameEngine;
import com.guidebee.game.GameEngineRuntimeException;
import com.guidebee.game.engine.assets.AssetLoaderParameters;
import com.guidebee.game.engine.assets.AssetManager;
import com.guidebee.game.engine.assets.loaders.TextureLoader;
import com.guidebee.game.engine.graphics.GLTexture;
import com.guidebee.game.engine.graphics.opengles.IGL20;
import com.guidebee.game.engine.graphics.opengles.PixmapTextureData;
import com.guidebee.game.files.FileHandle;
import com.guidebee.game.graphics.Pixmap.Format;
import com.guidebee.utils.collections.Array;

import java.util.HashMap;
import java.util.Map;

//[------------------------------ MAIN CLASS ----------------------------------]

/**
 * A Texture wraps a standard OpenGL ES texture.
 * <p/>
 * A Texture can be managed. If the OpenGL context is lost all managed textures
 * get invalidated. This happens when a user switches
 * to another application or receives an incoming call. Managed textures get
 * reloaded automatically.
 * <p/>
 * A Texture has to be bound via the {@link Texture#bind()} method in order for
 * it to be applied to geometry. The texture will be
 * bound to the currently active texture unit specified via
 * {@link com.guidebee.game.engine.graphics.opengles.IGL20#glActiveTexture(int)}.
 * <p/>
 * You can draw {@link Pixmap}s to a texture at any time. The changes will
 * be automatically uploaded to texture memory. This is of
 * course not extremely fast so use it with care. It also only works
 * with unmanaged textures.
 * <p/>
 * A Texture must be disposed when it is no longer used
 *
 * @author badlogicgames@gmail.com
 */
public class Texture extends GLTexture {
    private static AssetManager assetManager;
    final static Map<Application, Array<Texture>>
            managedTextures = new HashMap<Application, Array<Texture>>();

    /**
     * Texture filter types.
     */
    public enum TextureFilter {
        Nearest(IGL20.GL_NEAREST), Linear(IGL20.GL_LINEAR),
        MipMap(IGL20.GL_LINEAR_MIPMAP_LINEAR), MipMapNearestNearest(
                IGL20.GL_NEAREST_MIPMAP_NEAREST),
        MipMapLinearNearest(IGL20.GL_LINEAR_MIPMAP_NEAREST),
        MipMapNearestLinear(
                IGL20.GL_NEAREST_MIPMAP_LINEAR),
        MipMapLinearLinear(IGL20.GL_LINEAR_MIPMAP_LINEAR);

        final int glEnum;

        TextureFilter(int glEnum) {
            this.glEnum = glEnum;
        }

        public boolean isMipMap() {
            return glEnum != IGL20.GL_NEAREST && glEnum != IGL20.GL_LINEAR;
        }

        public int getGLEnum() {
            return glEnum;
        }
    }

    /**
     * Texture wrap type.
     */
    public enum TextureWrap {
        MirroredRepeat(IGL20.GL_MIRRORED_REPEAT),
        ClampToEdge(IGL20.GL_CLAMP_TO_EDGE), Repeat(IGL20.GL_REPEAT);

        final int glEnum;

        TextureWrap(int glEnum) {
            this.glEnum = glEnum;
        }

        public int getGLEnum() {
            return glEnum;
        }
    }

    TextureData data;

    public Texture(String internalPath) {
        this(GameEngine.files.internal(internalPath));
    }

    public Texture(FileHandle file) {
        this(file, null, false);
    }

    public Texture(FileHandle file, boolean useMipMaps) {
        this(file, null, useMipMaps);
    }

    public Texture(FileHandle file, Format format, boolean useMipMaps) {
        this(createTextureData(file, format, useMipMaps));
    }

    public Texture(Pixmap pixmap) {
        this(new PixmapTextureData(pixmap, null, false, false));
    }

    public Texture(Pixmap pixmap, boolean useMipMaps) {
        this(new PixmapTextureData(pixmap, null, useMipMaps, false));
    }

    public Texture(Pixmap pixmap, Format format, boolean useMipMaps) {
        this(new PixmapTextureData(pixmap, format, useMipMaps, false));
    }

    public Texture(int width, int height, Format format) {
        this(new PixmapTextureData(new Pixmap(width, height, format),
                null, false, true));
    }

    public Texture(TextureData data) {
        super(IGL20.GL_TEXTURE_2D, createGLHandle());
        load(data);
        if (data.isManaged()) addManagedTexture(GameEngine.app, this);
    }

    public void load(TextureData data) {
        if (this.data != null && data.isManaged() != this.data.isManaged())
            throw new GameEngineRuntimeException(
                    "New data must have the same managed status as the old data");
        this.data = data;

        if (!data.isPrepared()) data.prepare();

        bind();
        uploadImageData(IGL20.GL_TEXTURE_2D, data);

        setFilter(minFilter, magFilter);
        setWrap(uWrap, vWrap);
        GameEngine.gl.glBindTexture(glTarget, 0);
    }

    /**
     * Used internally to reload after context loss. Creates a new GL handle
     * then calls {@link #load(TextureData)}. Use this only
     * if you know what you do!
     */
    @Override
    protected void reload() {
        if (!isManaged())
            throw new GameEngineRuntimeException("Tried to reload unmanaged Texture");
        glHandle = createGLHandle();
        load(data);
    }

    /**
     * Draws the given {@link Pixmap} to the texture at position x, y. No
     * clipping is performed so you have to make sure that you
     * draw only inside the texture region. Note that this will only draw
     * to mipmap level 0!
     *
     * @param pixmap The Pixmap
     * @param x      The x coordinate in pixels
     * @param y      The y coordinate in pixels
     */
    public void draw(Pixmap pixmap, int x, int y) {
        if (data.isManaged())
            throw new GameEngineRuntimeException("can't draw to a managed texture");

        bind();
        GameEngine.gl.glTexSubImage2D(glTarget, 0, x, y, pixmap.getWidth(),
                pixmap.getHeight(), pixmap.getGLFormat(), pixmap.getGLType(),
                pixmap.getPixels());
    }

    @Override
    public int getWidth() {
        return data.getWidth();
    }

    @Override
    public int getHeight() {
        return data.getHeight();
    }

    @Override
    public int getDepth() {
        return 0;
    }

    public TextureData getTextureData() {
        return data;
    }

    /**
     * @return whether this texture is managed or not.
     */
    public boolean isManaged() {
        return data.isManaged();
    }

    /**
     * Disposes all resources associated with the texture
     */
    public void dispose() {
        // this is a hack. reason: we have to set the glHandle to 0 for textures that are
        // reloaded through the asset manager as we first remove (and thus dispose) the texture
        // and then reload it. the glHandle is set to 0 in invalidateAllTextures prior to
        // removal from the asset manager.
        if (glHandle == 0) return;
        delete();
        if (data.isManaged()) if (managedTextures.get(GameEngine.app) != null)
            managedTextures.get(GameEngine.app).removeValue(this, true);
    }

    private static void addManagedTexture(Application app, Texture texture) {
        Array<Texture> managedTextureArray = managedTextures.get(app);
        if (managedTextureArray == null) managedTextureArray = new Array<Texture>();
        managedTextureArray.add(texture);
        managedTextures.put(app, managedTextureArray);
    }

    /**
     * Clears all managed textures. This is an internal method. Do not use it!
     */
    public static void clearAllTextures(Application app) {
        managedTextures.remove(app);
    }

    /**
     * Invalidate all managed textures. This is an internal method. Do not use it!
     */
    public static void invalidateAllTextures(Application app) {
        Array<Texture> managedTextureArray = managedTextures.get(app);
        if (managedTextureArray == null) return;

        if (assetManager == null) {
            for (int i = 0; i < managedTextureArray.size; i++) {
                Texture texture = managedTextureArray.get(i);
                texture.reload();
            }
        } else {
            // first we have to make sure the AssetManager isn't loading anything anymore,
            // otherwise the ref counting trick below wouldn't work (when a texture is
            // currently on the task stack of the manager.)
            assetManager.finishLoading();

            // next we go through each texture and reload either directly or via the
            // asset manager.
            Array<Texture> textures = new Array<Texture>(managedTextureArray);
            for (Texture texture : textures) {
                String fileName = assetManager.getAssetFileName(texture);
                if (fileName == null) {
                    texture.reload();
                } else {
                    // get the ref count of the texture, then set it to 0 so we
                    // can actually remove it from the assetmanager. Also set the
                    // handle to zero, otherwise we might accidentially dispose
                    // already reloaded textures.
                    final int refCount = assetManager.getReferenceCount(fileName);
                    assetManager.setReferenceCount(fileName, 0);
                    texture.glHandle = 0;

                    // create the parameters, passing the reference to the texture as
                    // well as a callback that sets the ref count.
                    TextureLoader.TextureParameter params = new TextureLoader.TextureParameter();
                    params.textureData = texture.getTextureData();
                    params.minFilter = texture.getMinFilter();
                    params.magFilter = texture.getMagFilter();
                    params.wrapU = texture.getUWrap();
                    params.wrapV = texture.getVWrap();
                    params.genMipMaps = texture.data.useMipMaps(); // not sure about this?
                    params.texture = texture; // special parameter which will
                    // ensure that the references stay the same.
                    params.loadedCallback = new AssetLoaderParameters.LoadedCallback() {
                        @Override
                        public void finishedLoading(AssetManager assetManager,
                                                    String fileName, Class type) {
                            assetManager.setReferenceCount(fileName, refCount);
                        }
                    };

                    // unload the texture, create a new gl handle then reload it.
                    assetManager.unload(fileName);
                    texture.glHandle = Texture.createGLHandle();
                    assetManager.load(fileName, Texture.class, params);
                }
            }
            managedTextureArray.clear();
            managedTextureArray.addAll(textures);
        }
    }

    /**
     * Sets the {@link AssetManager}. When the context is lost, textures managed
     * by the asset manager are reloaded by the manager
     * on a separate thread (provided that a suitable
     * {@link com.guidebee.game.engine.assets.loaders.AssetLoader} is registered
     * with the manager). Textures not managed by
     * the AssetManager are reloaded via the usual means on the rendering thread.
     *
     * @param manager the asset manager.
     */
    public static void setAssetManager(AssetManager manager) {
        Texture.assetManager = manager;
    }

    public static String getManagedStatus() {
        StringBuilder builder = new StringBuilder();
        builder.append("Managed textures/app: { ");
        for (Application app : managedTextures.keySet()) {
            builder.append(managedTextures.get(app).size);
            builder.append(" ");
        }
        builder.append("}");
        return builder.toString();
    }

    /**
     * @return the number of managed textures currently loaded
     */
    public static int getNumManagedTextures() {
        return managedTextures.get(GameEngine.app).size;
    }
}
