/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.csv;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.ExtendedBufferedReader;
import org.apache.commons.csv.Lexer;
import org.apache.commons.csv.Token;
import org.apache.commons.csv.TokenMatchers;
import org.hamcrest.MatcherAssert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class LexerTest {
    private CSVFormat formatWithEscaping;

    @BeforeEach
    public void setUp() {
        this.formatWithEscaping = CSVFormat.DEFAULT.withEscape('\\');
    }

    private Lexer createLexer(String input, CSVFormat format) {
        return new Lexer(format, new ExtendedBufferedReader((Reader)new StringReader(input)));
    }

    @Test
    public void testSurroundingSpacesAreDeleted() throws IOException {
        String code = "noSpaces,  leadingSpaces,trailingSpaces  ,  surroundingSpaces  ,  ,,";
        try (Lexer parser = this.createLexer("noSpaces,  leadingSpaces,trailingSpaces  ,  surroundingSpaces  ,  ,,", CSVFormat.DEFAULT.withIgnoreSurroundingSpaces());){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "noSpaces"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "leadingSpaces"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "trailingSpaces"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "surroundingSpaces"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
        }
    }

    @Test
    public void testSurroundingTabsAreDeleted() throws IOException {
        String code = "noTabs,\tleadingTab,trailingTab\t,\tsurroundingTabs\t,\t\t,,";
        try (Lexer parser = this.createLexer("noTabs,\tleadingTab,trailingTab\t,\tsurroundingTabs\t,\t\t,,", CSVFormat.DEFAULT.withIgnoreSurroundingSpaces());){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "noTabs"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "leadingTab"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "trailingTab"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "surroundingTabs"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
        }
    }

    @Test
    public void testIgnoreEmptyLines() throws IOException {
        String code = "first,line,\n\n\nsecond,line\n\n\nthird line \n\n\nlast, line \n\n\n\n";
        CSVFormat format = CSVFormat.DEFAULT.withIgnoreEmptyLines();
        try (Lexer parser = this.createLexer("first,line,\n\n\nsecond,line\n\n\nthird line \n\n\nlast, line \n\n\n\n", format);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "first"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "line"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "second"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "line"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "third line "));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "last"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, " line "));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
        }
    }

    @Test
    public void testComments() throws IOException {
        String code = "first,line,\nsecond,line,tokenWith#no-comment\n# comment line \nthird,line,#no-comment\n# penultimate comment\n# Final comment\n";
        CSVFormat format = CSVFormat.DEFAULT.withCommentMarker('#');
        try (Lexer parser = this.createLexer("first,line,\nsecond,line,tokenWith#no-comment\n# comment line \nthird,line,#no-comment\n# penultimate comment\n# Final comment\n", format);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "first"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "line"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "second"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "line"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "tokenWith#no-comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.COMMENT, "comment line"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "third"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "line"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "#no-comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.COMMENT, "penultimate comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.COMMENT, "Final comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
        }
    }

    @Test
    public void testCommentsAndEmptyLines() throws IOException {
        String code = "1,2,3,\n\n\na,b x,c#no-comment\n#foo\n\n\nd,e,#no-comment\n\n\n# penultimate comment\n\n\n# Final comment\n";
        CSVFormat format = CSVFormat.DEFAULT.withCommentMarker('#').withIgnoreEmptyLines(false);
        Assertions.assertFalse((boolean)format.getIgnoreEmptyLines(), (String)"Should not ignore empty lines");
        try (Lexer parser = this.createLexer("1,2,3,\n\n\na,b x,c#no-comment\n#foo\n\n\nd,e,#no-comment\n\n\n# penultimate comment\n\n\n# Final comment\n", format);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "1"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "2"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "3"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "b x"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "c#no-comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.COMMENT, "foo"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "d"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "e"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "#no-comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.COMMENT, "penultimate comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.COMMENT, "Final comment"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
        }
    }

    @Test
    public void testBackslashWithoutEscaping() throws IOException {
        String code = "a,\\,,b\\\n\\,,";
        CSVFormat format = CSVFormat.DEFAULT;
        Assertions.assertFalse((boolean)format.isEscapeCharacterSet());
        try (Lexer parser = this.createLexer("a,\\,,b\\\n\\,,", format);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "\\"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b\\"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "\\"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, ""));
        }
    }

    @Test
    public void testBackslashWithEscaping() throws IOException {
        String code = "a,\\,,b\\\\\n\\,,\\\nc,d\\\r\ne";
        CSVFormat format = this.formatWithEscaping.withIgnoreEmptyLines(false);
        Assertions.assertTrue((boolean)format.isEscapeCharacterSet());
        try (Lexer parser = this.createLexer("a,\\,,b\\\\\n\\,,\\\nc,d\\\r\ne", format);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ","));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b\\"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ","));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "\nc"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "d\r"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, "e"));
        }
    }

    @Test
    public void testNextToken4() throws IOException {
        String code = "a,\"foo\",b\na,   \" foo\",b\na,\"foo \"  ,b\na,  \" foo \"  ,b";
        try (Lexer parser = this.createLexer("a,\"foo\",b\na,   \" foo\",b\na,\"foo \"  ,b\na,  \" foo \"  ,b", CSVFormat.DEFAULT.withIgnoreSurroundingSpaces());){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "foo"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, " foo"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "foo "));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, " foo "));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, "b"));
        }
    }

    @Test
    public void testNextToken5() throws IOException {
        String code = "a,\"foo\n\",b\n\"foo\n  baar ,,,\"\n\"\n\t \n\"";
        try (Lexer parser = this.createLexer("a,\"foo\n\",b\n\"foo\n  baar ,,,\"\n\"\n\t \n\"", CSVFormat.DEFAULT);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "foo\n"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "foo\n  baar ,,,"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, "\n\t \n"));
        }
    }

    @Test
    public void testNextToken6() throws IOException {
        String code = "a;'b and '' more\n'\n!comment;;;;\n;;";
        CSVFormat format = CSVFormat.DEFAULT.withQuote('\'').withCommentMarker('!').withDelimiter(';');
        try (Lexer parser = this.createLexer("a;'b and '' more\n'\n!comment;;;;\n;;", format);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "a"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EORECORD, "b and ' more\n"));
        }
    }

    @Test
    public void testDelimiterIsWhitespace() throws IOException {
        String code = "one\ttwo\t\tfour \t five\t six";
        try (Lexer parser = this.createLexer("one\ttwo\t\tfour \t five\t six", CSVFormat.TDF);){
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "one"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "two"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, ""));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "four"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.TOKEN, "five"));
            MatcherAssert.assertThat((Object)parser.nextToken(new Token()), TokenMatchers.matches(Token.Type.EOF, "six"));
        }
    }

    @Test
    public void testEscapedCR() throws Exception {
        try (Lexer lexer = this.createLexer("character\\\rEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\rEscaped"));
        }
    }

    @Test
    public void testCR() throws Exception {
        try (Lexer lexer = this.createLexer("character\rNotEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character"));
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("NotEscaped"));
        }
    }

    @Test
    public void testEscapedLF() throws Exception {
        try (Lexer lexer = this.createLexer("character\\\nEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\nEscaped"));
        }
    }

    @Test
    public void testLF() throws Exception {
        try (Lexer lexer = this.createLexer("character\nNotEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character"));
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("NotEscaped"));
        }
    }

    @Test
    public void testEscapedTab() throws Exception {
        try (Lexer lexer = this.createLexer("character\\\tEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\tEscaped"));
        }
    }

    @Test
    public void testTab() throws Exception {
        try (Lexer lexer = this.createLexer("character\tNotEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\tNotEscaped"));
        }
    }

    @Test
    public void testEscapedBackspace() throws Exception {
        try (Lexer lexer = this.createLexer("character\\\bEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\bEscaped"));
        }
    }

    @Test
    public void testBackspace() throws Exception {
        try (Lexer lexer = this.createLexer("character\bNotEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\bNotEscaped"));
        }
    }

    @Test
    public void testEscapedFF() throws Exception {
        try (Lexer lexer = this.createLexer("character\\\fEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\fEscaped"));
        }
    }

    @Test
    public void testFF() throws Exception {
        try (Lexer lexer = this.createLexer("character\fNotEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\fNotEscaped"));
        }
    }

    @Test
    public void testEscapedMySqlNullValue() throws Exception {
        try (Lexer lexer = this.createLexer("character\\NEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\\NEscaped"));
        }
    }

    @Test
    public void testEscapedCharacter() throws Exception {
        try (Lexer lexer = this.createLexer("character\\aEscaped", this.formatWithEscaping);){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\\aEscaped"));
        }
    }

    @Test
    public void testEscapedControlCharacter() throws Exception {
        try (Lexer lexer = this.createLexer("character!rEscaped", CSVFormat.DEFAULT.withEscape('!'));){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\rEscaped"));
        }
    }

    @Test
    public void testEscapedControlCharacter2() throws Exception {
        try (Lexer lexer = this.createLexer("character\\rEscaped", CSVFormat.DEFAULT.withEscape('\\'));){
            MatcherAssert.assertThat((Object)lexer.nextToken(new Token()), TokenMatchers.hasContent("character\rEscaped"));
        }
    }

    @Test
    public void testEscapingAtEOF() throws Exception {
        String code = "escaping at EOF is evil\\";
        try (Lexer lexer = this.createLexer("escaping at EOF is evil\\", this.formatWithEscaping);){
            Assertions.assertThrows(IOException.class, () -> lexer.nextToken(new Token()));
        }
    }
}

