package com.groobee.message.push.utils;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;

import androidx.core.app.NotificationCompat;

import com.groobee.message.Groobee;
import com.groobee.message.R;
import com.groobee.message.common.Constants;
import com.groobee.message.utils.IntentUtils;
import com.groobee.message.utils.LoggerUtils;
import com.groobee.message.utils.StringUtils;

public class GroobeeNotificationActionUtils {
    private static final String TAG = LoggerUtils.getClassLogTag(GroobeeNotificationActionUtils.class);

    public static void addNotificationActions(Context context, NotificationCompat.Builder notificationBuilder, Bundle notificationExtras) {
        try {
            if (notificationExtras == null) {
                LoggerUtils.w(TAG, context.getString(R.string.GROOBEE_NOTIFICATION_ACTION_UTILS_EXTRA_IS_NULL));
                return;
            }

            int actionIndex = 0;
            while (!StringUtils.isNullOrBlank(getActionFieldAtIndex(actionIndex, notificationExtras, Constants.PUSH_ACTION_TYPE_KEY_TEMPLATE))) {
                addNotificationAction(context, notificationBuilder, notificationExtras, actionIndex);
                actionIndex++;
            }
        } catch (Exception e) {
            LoggerUtils.e(TAG, context.getString(R.string.GROOBEE_NOTIFICATION_ACTION_UTILS_EXCEPTION), e);
        }
    }

    public static void handleNotificationActionClicked(Context context, Intent intent) {
        try {
            String actionType = intent.getStringExtra(Constants.PUSH_ACTION_TYPE_KEY);
            if (StringUtils.isNullOrBlank(actionType)) {
                LoggerUtils.w(TAG, context.getString(R.string.GROOBEE_NOTIFICATION_ACTION_UTILS_HANDLE_NOTIFICATION_ACTION_CLICKED_ACTION_TYPE_IS_NULL));
                return;
            }

            int notificationId = intent.getIntExtra(Constants.PUSH_NOTIFICATION_ID, Constants.PUSH_DEFAULT_NOTIFICATION_ID);

            // Logs that the notification action was clicked.
            // Click analytics for all action types are logged.
            logNotificationActionClicked(context, intent, actionType);

            if (actionType.equals(Constants.PUSH_ACTION_TYPE_URI) || actionType.equals(Constants.PUSH_ACTION_TYPE_OPEN)) {
                GroobeeNotificationUtils.cancelNotification(context, notificationId);
                context.sendBroadcast(new Intent(Intent.ACTION_CLOSE_SYSTEM_DIALOGS));
                if (actionType.equals(Constants.PUSH_ACTION_TYPE_URI) && intent.getExtras().containsKey(Constants.PUSH_ACTION_URI_KEY)) {
                    // Set the deep link that to open to the correct action's deep link.
                    intent.putExtra(Constants.PUSH_DEEP_LINK_KEY, intent.getStringExtra(Constants.PUSH_URL_LINK_KEY));
                    if (intent.getExtras().containsKey(Constants.PUSH_ACTION_USE_WEBVIEW_KEY)) {
                        intent.putExtra(Constants.PUSH_OPEN_URI_IN_WEBVIEW_KEY, intent.getStringExtra(Constants.PUSH_ACTION_USE_WEBVIEW_KEY));
                    }
                } else {
                    // Otherwise, remove any existing deep links.
                    intent.removeExtra(Constants.PUSH_DEEP_LINK_KEY);
                }
                GroobeeNotificationUtils.sendNotificationOpenedBroadcast(context, intent);

//                GroobeeConfigProvider appConfigurationProvider = new GroobeeConfigProvider(context);
//                if (appConfigurationProvider.getHandlePushDeepLinksAutomatically()) {
//                    GroobeeNotificationUtils.routeUserWithNotificationOpenedIntent(context, intent);
//                }
            } else if (actionType.equals(Constants.PUSH_ACTION_TYPE_NONE)) {
                GroobeeNotificationUtils.cancelNotification(context, notificationId);
            } else {
                LoggerUtils.w(TAG, context.getString(R.string.GROOBEE_NOTIFICATION_ACTION_UTILS_HANDLE_NOTIFICATION_ACTION_CLICKED_UNKNOWN_ACTION));
            }
        } catch (Exception e) {
            LoggerUtils.e(TAG, context.getString(R.string.GROOBEE_NOTIFICATION_ACTION_UTILS_HANDLE_NOTIFICATION_ACTION_CLICKED_EXCEPTION), e);
        }
    }

    private static void addNotificationAction(Context context, NotificationCompat.Builder notificationBuilder, Bundle notificationExtras, int actionIndex) {
        Bundle notificationActionExtras = new Bundle(notificationExtras);

        String actionType = getActionFieldAtIndex(actionIndex, notificationExtras, Constants.PUSH_ACTION_TYPE_KEY_TEMPLATE);
        notificationActionExtras.putInt(Constants.PUSH_ACTION_INDEX_KEY, actionIndex);
        notificationActionExtras.putString(Constants.PUSH_ACTION_TYPE_KEY, actionType);
        notificationActionExtras.putString(Constants.PUSH_ACTION_ID_KEY, getActionFieldAtIndex(actionIndex, notificationExtras, Constants.PUSH_ACTION_ID_KEY_TEMPLATE));
        notificationActionExtras.putString(Constants.PUSH_ACTION_URI_KEY, getActionFieldAtIndex(actionIndex, notificationExtras, Constants.PUSH_ACTION_URI_KEY_TEMPLATE));
        notificationActionExtras.putString(Constants.PUSH_ACTION_USE_WEBVIEW_KEY,
        getActionFieldAtIndex(actionIndex, notificationExtras, Constants.PUSH_ACTION_USE_WEBVIEW_KEY_TEMPLATE));

        PendingIntent pendingSendIntent;
        if (actionType.equals(Constants.PUSH_ACTION_TYPE_NONE)) {
            LoggerUtils.v(TAG, context.getString(R.string.GROOBEE_NOTIFICATION_ACTION_UTILS_ADD_NOTIFICATION_ACTION, actionType));
            Intent sendIntent = new Intent(Constants.PUSH_ACTION_CLICKED_ACTION).setClass(context, GroobeeNotificationUtils.getNotificationReceiverClass());
            sendIntent.putExtras(notificationActionExtras);
            pendingSendIntent = PendingIntent.getBroadcast(context, IntentUtils.getRequestCode(), sendIntent, PendingIntent.FLAG_UPDATE_CURRENT);

            String actionText = getActionFieldAtIndex(actionIndex, notificationExtras, Constants.PUSH_ACTION_TEXT_KEY_TEMPLATE);
            NotificationCompat.Action.Builder notificationActionBuilder = new NotificationCompat.Action.Builder(0, actionText, pendingSendIntent);
            notificationActionBuilder.addExtras(new Bundle(notificationActionExtras));
            notificationBuilder.addAction(notificationActionBuilder.build());
        }
    }

    private static void logNotificationActionClicked(Context context, Intent intent, String actionType) {
        String campaignId = intent.getStringExtra(Constants.PUSH_CAMPAIGN_ID_KEY);
        String actionButtonId = intent.getStringExtra(Constants.PUSH_ACTION_ID_KEY);
        Groobee.getInstance().logPushNotificationActionClicked(campaignId, actionButtonId, actionType);
    }

    public static String getActionFieldAtIndex(int actionIndex, Bundle notificationExtras, String actionFieldKeyTemplate) {
        return getActionFieldAtIndex(actionIndex, notificationExtras, actionFieldKeyTemplate, "");
    }

    public static String getActionFieldAtIndex(int actionIndex, Bundle notificationExtras, String actionFieldKeyTemplate, String defaultValue) {
        String actionFieldKey = actionFieldKeyTemplate.replace("*", String.valueOf(actionIndex));
        String actionFieldValue = notificationExtras.getString(actionFieldKey);
        if (actionFieldValue == null) {
            return defaultValue;
        } else {
            return actionFieldValue;
        }
    }
}
